/*
 * File:     ImportShoeboxWAC.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.gui;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.util.*;

import mpi.eudico.server.corpora.clomimpl.shoebox.ShoeboxPreferences;
import mpi.eudico.server.corpora.clomimpl.shoebox.ToolboxDecoderInfo;

import java.awt.BorderLayout;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import java.io.File;

import java.util.List;

import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.border.TitledBorder;


/**
 * Displays a dialog to either import a WAC file or a Shoebox file. Uses a
 * JOptionPane style mechanism to display a JDialog and return an Object as a
 * user value.<br>
 * <b>Note: </b>localization of the file choosers is not implemented (yet).
 *
 * @author Han Sloetjes
 */
public class ImportShoeboxWAC extends JComponent implements ActionListener,
    ItemListener {
    /** Holds value of property DOCUMENT ME! */
    public static final int SHOEBOX = 0;

    /** Holds value of property DOCUMENT ME! */
    public static final int WAC = 1;
    private static int importType;

    /** Holds value of property DOCUMENT ME! */
    public static final String VALUE_PROPERTY = "value";

    /** Holds value of property DOCUMENT ME! */
    public static final String TEXT_KEY = "TextFile";

    /** Holds value of property DOCUMENT ME! */
    public static final String TYPE_KEY = "TypeFile";

    /** Holds value of property DOCUMENT ME! */

    //    public static final String MEDIA_KEY = "MediaFile";
    private JTextField sbxField = new JTextField("", 23);
    private JTextField typField = new JTextField("", 23);

    //    private JTextField auField = new JTextField("", 23);
    private GridBagLayout gridbag = new GridBagLayout();
    private JButton txtButton;
    private JButton typButton;

    //    private JButton medButton;
    private JButton fieldSpecButton;
    private JButton okButton;
    private JButton cancelButton;
    private JRadioButton typeRB;
    private JRadioButton specRB;
    private JLabel typeLabel;
    private JLabel intervalLabel;
    private JTextField intervalField;
    private JCheckBox timeInRefMarker;
    private JCheckBox allUnicodeCB;

    /** Holds value of property DOCUMENT ME! */
    private final String INTERVAL_PREF = "ShoeboxChatBlockDuration";

    //	private File lastUsedDir;//used for elan properties file

    /** Used for the storage of the filenames and media files */

    //private Hashtable fileNames = new Hashtable();
    private List markers = null;

    //    private Vector mediaFileNames;
    private Object value;

    /**
     * Creates a new ImportShoeboxWAC instance
     *
     * @param type either <code>WAC</code> or <code>SHOEBOX</code>
     */
    private ImportShoeboxWAC(int type) {
        if (type == WAC) {
            importType = WAC;
        } else {
            importType = SHOEBOX;
        }

        createPane();
    }

    private void createPane() {
        setLayout(gridbag);

        Insets insets = new Insets(2, 6, 2, 6);
        GridBagConstraints gbc = new GridBagConstraints();
        int y = 0;

        if (importType == SHOEBOX) {
            ButtonGroup buttonGroup = new ButtonGroup();
            typeRB = new JRadioButton();
            typeRB.setSelected(true);
            specRB = new JRadioButton();
            buttonGroup.add(typeRB);
            buttonGroup.add(specRB);

            gbc.gridx = 1;
            gbc.gridy = y;
            gbc.weightx = 0.0;
            gbc.anchor = GridBagConstraints.WEST;
            gbc.insets = insets;
            add(new JLabel(ElanLocale.getString("ImportDialog.Label.Shoebox")),
                gbc);

            gbc.gridx = 2;
            gbc.weightx = 1.0;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            add(sbxField, gbc);

            txtButton = new JButton("...");
            txtButton.addActionListener(this);
            gbc.gridx = 3;
            gbc.weightx = 0.0;
            gbc.fill = GridBagConstraints.NONE;
            add(txtButton, gbc);
            y++;

            gbc.gridx = 0;
            gbc.gridy = y;
            add(typeRB, gbc);

            typeLabel = new JLabel(ElanLocale.getString(
                        "ImportDialog.Label.Type"));
            gbc.gridx = 1;
            add(typeLabel, gbc);

            gbc.gridx = 2;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 1.0;
            add(typField, gbc);

            typButton = new JButton("...");
            typButton.addActionListener(this);
            gbc.gridx = 3;
            gbc.fill = GridBagConstraints.NONE;
            gbc.weightx = 0.0;
            add(typButton, gbc);
            y++;

            allUnicodeCB = new JCheckBox(ElanLocale.getString(
                        "ImportDialog.Label.AllUnicode"));
            gbc.gridx = 1;
            gbc.gridy = y;
            gbc.gridwidth = 2;
            add(allUnicodeCB, gbc);
            y++;

            gbc.gridx = 0;
            gbc.gridy = y;
            add(specRB, gbc);

            fieldSpecButton = new JButton(ElanLocale.getString(
                        "ImportDialog.Button.FieldSpec"));
            fieldSpecButton.addActionListener(this);
            fieldSpecButton.setEnabled(false);
            gbc.gridx = 1;
            add(fieldSpecButton, gbc);
            y++;

            JPanel optionsPanel = new JPanel(new GridBagLayout());
            optionsPanel.setBorder(new TitledBorder(ElanLocale.getString(
                        "ImportDialog.Label.Options")));

            intervalLabel = new JLabel(ElanLocale.getString(
                        "ImportDialog.Label.BlockDuration"));
            intervalField = new JTextField();

            if (Preferences.get(INTERVAL_PREF, null) != null) {
                Integer val = (Integer) Preferences.get(INTERVAL_PREF, null);
                intervalField.setText("" + val.intValue());
            } else {
                intervalField.setText("" +
                    ShoeboxPreferences.DEFAULT_BLOCK_DURATION);
            }

            GridBagConstraints gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = 0;
            gridBagConstraints.insets = gbc.insets;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            optionsPanel.add(intervalLabel, gridBagConstraints);

            gridBagConstraints.gridx = 1;
            gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
            gridBagConstraints.weightx = 1.0;
            optionsPanel.add(intervalField, gridBagConstraints);

            timeInRefMarker = new JCheckBox(ElanLocale.getString(
                        "ImportDialog.Label.TimeInRefMarker"));
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = 1;
            gridBagConstraints.gridwidth = 2;
            gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
            gridBagConstraints.weightx = 1.0;
            optionsPanel.add(timeInRefMarker, gridBagConstraints);

            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = y;
            gridBagConstraints.gridwidth = 4;
            gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
            gridBagConstraints.weightx = 1.0;
            add(optionsPanel, gridBagConstraints);

            setShoeboxMarkerRB();

            typeRB.addItemListener(this);
            specRB.addItemListener(this);
            y++;
        } else {
            gbc.gridx = 0;
            gbc.gridy = y;
            gbc.weightx = 0.0;
            gbc.anchor = GridBagConstraints.WEST;
            gbc.insets = insets;
            add(new JLabel(ElanLocale.getString("ImportDialog.Label.WAC")), gbc);

            gbc.gridx = 1;
            gbc.weightx = 1.0;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            add(sbxField, gbc);

            txtButton = new JButton("...");
            txtButton.addActionListener(this);
            gbc.gridx = 2;
            gbc.weightx = 0.0;
            gbc.fill = GridBagConstraints.NONE;
            add(txtButton, gbc);
            y++;

            typeLabel = new JLabel(ElanLocale.getString(
                        "ImportDialog.Label.Type"));
            gbc.gridx = 0;
            gbc.gridy = y;
            add(typeLabel, gbc);

            gbc.gridx = 1;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 1.0;
            add(typField, gbc);

            typButton = new JButton("...");
            typButton.addActionListener(this);
            gbc.gridx = 2;
            gbc.fill = GridBagConstraints.NONE;
            gbc.weightx = 0.0;
            add(typButton, gbc);
            y++;
        }

        // ok - cancel buttons //
        JPanel buttonPanel = new JPanel(new GridLayout(1, 2, 6, 2));
        okButton = new JButton(ElanLocale.getString("Button.OK"));
        okButton.addActionListener(this);

        cancelButton = new JButton(ElanLocale.getString("Button.Cancel"));
        cancelButton.addActionListener(this);

        buttonPanel.add(okButton);
        buttonPanel.add(cancelButton);

        gbc.gridwidth = GridBagConstraints.REMAINDER;
        gbc.anchor = GridBagConstraints.SOUTH;
        gbc.fill = GridBagConstraints.NONE;
        gbc.gridx = 0;
        gbc.gridy = y;
        ;
        add(buttonPanel, gbc);
    }

    private void setShoeboxMarkerRB() {
        Object useTyp = Preferences.get("LastUsedShoeboxImportWithType", null);

        if (useTyp instanceof Boolean && !((Boolean) useTyp).booleanValue()) {
            specRB.setSelected(true);
            typButton.setEnabled(false);
            fieldSpecButton.setEnabled(true);
            allUnicodeCB.setEnabled(false);
        } else {
            typeRB.setSelected(true);

            Object luTypFile = Preferences.get("LastUsedShoeboxTypFile", null);

            if (luTypFile instanceof String) {
                typField.setText((String) luTypFile);
            }

            typButton.setEnabled(true);
            allUnicodeCB.setEnabled(true);
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public static Object showDialog() {
        return showDialog(SHOEBOX);
    }

    /**
     * DOCUMENT ME!
     *
     * @param type DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public static Object showDialog(int type) {
        return showDialog(null, type);
    }

    /**
     * Creates a Dialog to select some files for import.
     *
     * @param parent the parent Frame, can be null
     * @param type the type of import, either WAC or Shoebox
     *
     * @return a Hashtable with file names, or null
     */
    public static Object showDialog(Frame parent, int type) {
        ImportShoeboxWAC pane = new ImportShoeboxWAC(type);
        JDialog dialog = pane.createDialog(parent, type);
        dialog.setVisible(true);
        dialog.dispose();

        Object o = pane.getValue();

        //System.out.println("Return Value: " + o);
        return o;
    }

    /**
     * Creates the dialog with <code>this</code> as content pane.
     *
     * @param parent the parent Frame or null
     * @param type the type of import, either WAC or Shoebox
     *
     * @return a modal JDialog
     */
    private JDialog createDialog(Frame parent, int type) {
        final JDialog dialog = new ClosableDialog();

        if (type == WAC) {
            dialog.setTitle(ElanLocale.getString("ImportDialog.Title.WAC"));
        } else {
            dialog.setTitle(ElanLocale.getString("ImportDialog.Title.Shoebox"));
        }

        dialog.getContentPane().setLayout(new BorderLayout());
        dialog.getContentPane().add(this, BorderLayout.CENTER);
        dialog.setModal(true);
        dialog.pack();

        if (parent != null) {
            dialog.setLocationRelativeTo(parent);
        }

        dialog.addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent we) {
                    setValue(null);
                }
            });

        // use the java.beans PropertyChangeSupport mechanism
        addPropertyChangeListener(new PropertyChangeListener() {
                public void propertyChange(PropertyChangeEvent event) {
                    if (dialog.isVisible() &&
                            (event.getSource() == ImportShoeboxWAC.this) &&
                            event.getPropertyName().equals(VALUE_PROPERTY)) {
                        dialog.setVisible(false);
                    }
                }
            });

        return dialog;
    }

    /**
     * DOCUMENT ME!
     *
     * @param e DOCUMENT ME!
     */
    public void actionPerformed(ActionEvent e) {
        Object source = e.getSource();

        if (source == txtButton) {
            chooseSbxWAC();
        } else if (source == typButton) {
            chooseTyp();
        } else if (source == fieldSpecButton) {
            specifyFieldSpecs();

            //    }
            //      else if (source == medButton) {
            //          chooseMedia();
        } else if (source == okButton) {
            checkFields();

            /*
            if (checkFields()) {
                setValue(fileNames);
            }
            */
        } else if (source == cancelButton) {
            setValue(null);
        }
    }

    private void chooseSbxWAC() {
        String lastUsedDir = (String) Preferences.get("LastUsedShoeboxDir", null);

        if (lastUsedDir == null) {
            lastUsedDir = System.getProperty("user.dir");
        }

        JFileChooser chooser = new JFileChooser(lastUsedDir);
        chooser.setDialogTitle(ElanLocale.getString("ImportDialog.Title.Select"));

        if (importType == WAC) {
            chooser.setFileFilter(ElanFileFilter.createFileFilter(
                    ElanFileFilter.WAC_TYPE));
        } else {
            chooser.setAcceptAllFileFilterUsed(false);
            chooser.setFileFilter(ElanFileFilter.createFileFilter(
                    ElanFileFilter.SHOEBOX_TEXT_TYPE));
        }

        chooser.setApproveButtonText(ElanLocale.getString(
                "ImportDialog.Approve"));
        chooser.setDialogType(JFileChooser.OPEN_DIALOG);

        int option = chooser.showDialog(this, null);

        if (option == JFileChooser.APPROVE_OPTION) {
            File curDir = chooser.getCurrentDirectory();

            if (curDir != null) {
                Preferences.set("LastUsedShoeboxDir", curDir.getAbsolutePath(),
                    null);
            }

            File f = chooser.getSelectedFile();

            if (f != null) {
                sbxField.setText(f.getAbsolutePath());
            }
        }
    }

    private void chooseTyp() {
        String lastUsedDir = (String) Preferences.get("LastUsedShoeboxTypDir",
                null);

        if (lastUsedDir == null) {
            lastUsedDir = System.getProperty("user.dir");
        }

        JFileChooser chooser = new JFileChooser(lastUsedDir);
        chooser.setDialogTitle(ElanLocale.getString("ImportDialog.Title.Select"));
        chooser.setFileFilter(ElanFileFilter.createFileFilter(
                ElanFileFilter.SHOEBOX_TYP_TYPE));
        chooser.setApproveButtonText(ElanLocale.getString(
                "ImportDialog.Approve"));
        chooser.setDialogType(JFileChooser.OPEN_DIALOG);

        int option = chooser.showDialog(this, null);

        if (option == JFileChooser.APPROVE_OPTION) {
            File curDir = chooser.getCurrentDirectory();

            if (curDir != null) {
                Preferences.set("LastUsedShoeboxTypDir",
                    curDir.getAbsolutePath(), null);
            }

            File f = chooser.getSelectedFile();

            if (f != null) {
                typField.setText(f.getAbsolutePath());
                Preferences.set("LastUsedShoeboxTypFile", f.getAbsolutePath(),
                    null);
                Preferences.set("LastUsedShoeboxImportWithType", Boolean.TRUE,
                    null);
            }
        }
    }

    private void specifyFieldSpecs() {
        ShoeboxMarkerDialog smd = new ShoeboxMarkerDialog(null, true);
        smd.setVisible(true);
        markers = (List) smd.getValue();
        Preferences.set("LastUsedShoeboxImportWithType", Boolean.FALSE, null);

        //Preferences.set("LastUsedShoeboxMarkers", markers, null);
    }

    /**
     * Checks the contents of all input fields and next the existence of the
     * designated files.
     *
     * @return true if the files exist, false otherwise
     */
    private boolean checkFields() {
        String sbxPath = null;
        String typPath = null;

        if (sbxField.getText() != null) {
            sbxPath = sbxField.getText().trim();
        }

        if ((sbxPath == null) || (sbxPath.length() == 0)) {
            if (importType == WAC) {
                showError(ElanLocale.getString(
                        "ImportDialog.Message.SpecifyWAC"));
            } else {
                showError(ElanLocale.getString(
                        "ImportDialog.Message.SpecifyShoebox"));
            }

            return false;
        }

        if (typField.getText() != null) {
            typPath = typField.getText().trim();
        }

        if (typeRB.isSelected() &&
                ((typPath == null) || (typPath.length() == 0))) {
            showError(ElanLocale.getString("ImportDialog.Message.SpecifyType"));

            return false;
        }

        if (!(new File(sbxPath).exists())) {
            if (importType == WAC) {
                showError(ElanLocale.getString("ImportDialog.Message.NoWAC"));
            } else {
                showError(ElanLocale.getString("ImportDialog.Message.NoShoebox"));
            }

            return false;
        }

        if (typeRB.isSelected()) {
            if (!(new File(typPath).exists())) {
                showError(ElanLocale.getString("ImportDialog.Message.NoType"));

                return false;
            }
        } else {
            if ((markers == null) || (markers.size() == 0)) {
                showError(ElanLocale.getString(
                        "ImportDialog.Message.SpecifyMarkers"));

                return false;
            }
        }

        int durVal = ShoeboxPreferences.DEFAULT_BLOCK_DURATION;

        if (intervalField != null) {
            String dur = intervalField.getText();

            try {
                durVal = Integer.parseInt(dur);
                ShoeboxPreferences.preferredBlockDuration = durVal;
                Preferences.set(INTERVAL_PREF, durVal, null);
            } catch (NumberFormatException nfe) {
                // ignore
            }
        }

        ToolboxDecoderInfo decInfo = new ToolboxDecoderInfo(sbxPath);
        decInfo.setBlockDuration(durVal);

        if (typeRB.isSelected()) {
            decInfo.setTypeFile(typPath);
            decInfo.setAllUnicode(allUnicodeCB.isSelected());
        } else {
            decInfo.setShoeboxMarkers(markers);
        }

        decInfo.setTimeInRefMarker(timeInRefMarker.isSelected());

        setValue(decInfo);

        return true;
    }

    /**
     * Shows an error dialog.
     *
     * @param message
     */
    private void showError(String message) {
        JOptionPane.showMessageDialog(this, message,
            ElanLocale.getString("Message.Error"), JOptionPane.ERROR_MESSAGE);
    }

    /**
     * DOCUMENT ME!
     *
     * @return the user object or null
     */
    public Object getValue() {
        return value;
    }

    /**
     * Sets the new value the user has chosen.
     *
     * @param newValue the new value
     */
    public void setValue(Object newValue) {
        Object oldValue = value;
        value = newValue;
        firePropertyChange(VALUE_PROPERTY, oldValue, value);
    }

    /**
     * DOCUMENT ME!
     *
     * @param e DOCUMENT ME!
     */
    public void itemStateChanged(ItemEvent e) {
        if ((e.getSource() == typeRB) &&
                (e.getStateChange() == ItemEvent.SELECTED)) {
            typeLabel.setEnabled(true);
            typButton.setEnabled(true);
            typField.setEnabled(true);
            allUnicodeCB.setEnabled(true);

            fieldSpecButton.setEnabled(false);
            markers = null;

            if ((typField.getText() == null) ||
                    (typField.getText().length() == 0)) {
                typButton.doClick(200);
            }
        } else if ((e.getSource() == specRB) &&
                (e.getStateChange() == ItemEvent.SELECTED)) {
            typeLabel.setEnabled(false);
            typButton.setEnabled(false);
            typField.setEnabled(false);
            allUnicodeCB.setEnabled(false);

            fieldSpecButton.setEnabled(true);
            fieldSpecButton.doClick(200);
        }
    }
}
