/*
 * File:     JFontChooser.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.gui;

import mpi.eudico.client.annotator.ElanLocale;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.GraphicsEnvironment;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;


/**
 * A simple font chooser displaying two lists, one with the font family names
 * of the system and one with the font varieties per family. Only one font can
 * be selected. There are no options to specify size or weight.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class JFontChooser extends JPanel implements ListSelectionListener,
    ActionListener {
    private JList familyList;
    private JList fontList;
    private JLabel selFontLabel;
    private JButton okButton;
    private JButton cancelButton;
    private GraphicsEnvironment ge;
    private String[] families;
    private Font[] allFonts;
    private Font selFont;
    private JDialog dialog;

    /**
     * Creates a new JFontChooser instance
     */
    public JFontChooser() {
        super();
        ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        families = ge.getAvailableFontFamilyNames();
        allFonts = ge.getAllFonts();
        initComponents();
    }

    /**
     * Initializes the ui components.
     */
    protected void initComponents() {
        familyList = new JList(families);
        familyList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        fontList = new JList(new DefaultListModel());
        fontList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        selFontLabel = new JLabel("-");
        okButton = new JButton(ElanLocale.getString("Button.Apply"));
        cancelButton = new JButton(ElanLocale.getString("Button.Cancel"));

        JPanel buttonPanel = new JPanel(new GridLayout(1, 2, 6, 2));

        setLayout(new GridBagLayout());

        JScrollPane familySP = new JScrollPane(familyList);
        JScrollPane fontSP = new JScrollPane(fontList);
        Dimension dim = new Dimension(200, 120);
        familySP.setPreferredSize(dim);
        fontSP.setPreferredSize(dim);

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = new Insets(10, 10, 2, 6);
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;

        add(new JScrollPane(familySP), gbc);
        gbc.gridx = 1;
        gbc.insets = new Insets(10, 6, 2, 10);
        add(new JScrollPane(fontSP), gbc);

        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.insets = new Insets(2, 10, 2, 10);
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.gridwidth = 2;
        gbc.weighty = 0;
        add(selFontLabel, gbc);

        buttonPanel.add(okButton);
        buttonPanel.add(cancelButton);

        gbc.insets = new Insets(10, 10, 2, 10);
        gbc.gridy = 3;
        gbc.anchor = GridBagConstraints.SOUTH;
        gbc.fill = GridBagConstraints.NONE;
        gbc.weightx = 0;
        add(buttonPanel, gbc);

        familyList.addListSelectionListener(this);
        fontList.addListSelectionListener(this);
        okButton.addActionListener(this);
        cancelButton.addActionListener(this);
    }

    private void updateFontLabel() {
        String ff = "";

        if (familyList.getSelectedValue() != null) {
            ff = (String) familyList.getSelectedValue();
        }

        if (fontList.getSelectedValue() != null) {
            ff += (" - " + (String) fontList.getSelectedValue());
        }

        selFontLabel.setText(ff);
    }

    private void updateFontList() {
        if (familyList.getSelectedValue() != null) {
            String ff = (String) familyList.getSelectedValue();

            ((DefaultListModel) fontList.getModel()).clear();

            for (int i = 0; i < allFonts.length; i++) {
                if (allFonts[i].getFamily().equals(ff)) {
                    ((DefaultListModel) fontList.getModel()).addElement(allFonts[i].getName());
                }
            }

            if (((DefaultListModel) fontList.getModel()).getSize() > 0) {
                fontList.setSelectedIndex(0);
                updateFont();
            }

            updateFontLabel();
        }
    }

    private void updateFont() {
        String name = (String) fontList.getSelectedValue();

        if (name == null) {
            selFont = null;
        } else {
            for (int i = 0; i < allFonts.length; i++) {
                if (allFonts[i].getName().equals(name)) {
                    selFont = allFonts[i];

                    break;
                }
            }
        }
    }

    private void doClose() {
        selFont = null;

        if (dialog != null) {
            dialog.setVisible(false);
            dialog.dispose();
        }
    }

    private void doApply() {
        updateFont();

        if ((dialog != null) && dialog.isVisible()) {
            dialog.setVisible(false);
            dialog.dispose();
        }
    }

    /**
     * Creates a dialog (that blocks) and returns the selected font or null if
     * the dialog has been canceled.
     *
     * @param parent the parent dialog
     * @param modal the modal flag
     *
     * @return the selected font
     */
    public Font showDialog(JDialog parent, boolean modal) {
        dialog = new JDialog(parent, modal);
        dialog.setTitle(ElanLocale.getString("FontDialog.Title"));
        dialog.setContentPane(this);
        dialog.addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent we) {
                    doClose();
                }
            });
        dialog.pack();
        dialog.setLocationRelativeTo(parent);
        dialog.setVisible(true);

        return getValue();
    }

    /**
     * Creates a dialog (that blocks) and returns the selected font or null if
     * the dialog has been canceled.
     *
     * @param parent the parent frame
     * @param modal the modal flag
     *
     * @return the selected font
     */
    public Font showDialog(JFrame parent, boolean modal) {
        dialog = new JDialog(parent, modal);
        dialog.setTitle(ElanLocale.getString("FontDialog.Title"));
        dialog.setContentPane(this);
        dialog.addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent we) {
                    doClose();
                }
            });
        dialog.pack();
        dialog.setLocationRelativeTo(parent);
        dialog.setVisible(true);

        return getValue();
    }

    /**
     * Returns the selected font, can be null;
     *
     * @return the elected font
     */
    public Font getValue() {
        return selFont;
    }

    /**
     * List selection event handling.
     *
     * @param e the event
     */
    public void valueChanged(ListSelectionEvent e) {
        if (e.getSource() == familyList) {
            updateFontList();
            updateFontLabel();
        } else if (e.getSource() == fontList) {
            updateFont();
            updateFontLabel();
        }
    }

    /**
     * The button actions.
     *
     * @param e the action event
     */
    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == okButton) {
            doApply();
        } else if (e.getSource() == cancelButton) {
            doClose();
        }
    }
}
