/*
 * File:     ResizeComponent.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.gui;

import mpi.eudico.client.annotator.Constants;
import mpi.eudico.client.annotator.ElanLayoutManager;
import mpi.eudico.client.annotator.ElanLocale;

import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;


/**
 * A ui component that can be used to change the size of the media area in the
 * main window by dragging the component with the mouse. <br>
 * <b>Note: </b> only vertical resizing is supported in the current version,
 * horizontal size (width) is calculated based on the height.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class ResizeComponent extends JPanel implements MouseListener,
    MouseMotionListener, ActionListener {
    private JLabel iconLabel;
    private ElanLayoutManager manager;
    private int orientation = SwingConstants.HORIZONTAL;
    private JPopupMenu popup;

    // dragging
    private int startXY = 0;
    private boolean dragging = false;

    /**
     * Creates a new ResizeComponent instance
     *
     * @param manager the layout manager that manages the layout of the window
     * @param orientation either <code>SwingConstants.HORIZONTAL</code> or
     *        <code>SwingConstants.VERTICAL</code>
     */
    public ResizeComponent(ElanLayoutManager manager, int orientation) {
        this.manager = manager;

        if (orientation == SwingConstants.VERTICAL) {
            this.orientation = orientation;
        }

        initComponents();
    }

    /**
     * Loads the icon image ands adds it to a label that is added to this
     * panel. Sets preferred size and adds mouse listeners.
     */
    private void initComponents() {
        setLayout(null);
        iconLabel = new JLabel();

        try {
            ImageIcon icon = new ImageIcon(this.getClass().getResource("/mpi/eudico/client/annotator/resources/ResizeDivider.gif"));
            iconLabel.setIcon(icon);
        } catch (Exception ex) {
            // if the icon could not be loaded
            iconLabel.setText("^");
        }

        add(iconLabel);
        iconLabel.setBounds(0, 0, 16, 16);
        setPreferredSize(new Dimension(16, 16));
        setBorder(new LineBorder(Constants.DEFAULTBACKGROUNDCOLOR, 1));
        addMouseListener(this);
        addMouseMotionListener(this);
    }

    /**
     * Creates the popup menu and adds one item, to restore the default height.
     */
    private void createPopupMenu() {
        popup = new JPopupMenu();

        JMenuItem item = new JMenuItem(ElanLocale.getString("Button.Default"));
        item.addActionListener(this);
        popup.add(item);
    }

    /**
     * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
     */
    public void mouseClicked(MouseEvent e) {
    }

    /**
     * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
     */
    public void mousePressed(MouseEvent e) {
        // HS jun 2006: change to discriminate Ctrl-click and Command-click on Mac
        // isRightMouseButton() returns true for both, isMetaDown() returns true for Command only 
        // but on Windows isMetaDown() returns true when the right mouse button (BUTTON3) has been pressed
        if ((SwingUtilities.isRightMouseButton(e) &&
                ((e.getButton() == MouseEvent.BUTTON1) ^ e.isMetaDown())) ||
                e.isPopupTrigger()) {
            if (popup == null) {
                createPopupMenu();
            }

            popup.show(this, e.getX(), e.getY());

            return;
        }

        if (orientation == SwingConstants.HORIZONTAL) {
            startXY = e.getX();
        } else {
            setCursor(Cursor.getPredefinedCursor(Cursor.N_RESIZE_CURSOR));
            startXY = e.getY();
        }

        dragging = true;
    }

    /**
     * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
     */
    public void mouseReleased(MouseEvent e) {
        if (dragging) {
            if (orientation == SwingConstants.HORIZONTAL) {
                //int dist = e.getY() - startXY;
                //manager.setMediaWidth(manager.getMediaWidth() - dist);
            } else {
                int dist = e.getY() - startXY;
                manager.setMediaAreaHeight(manager.getMediaAreaHeight() + dist);
            }
        }

        startXY = 0;
        dragging = false;

        //setCursor(Cursor.getDefaultCursor());
    }

    /**
     * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
     */
    public void mouseEntered(MouseEvent e) {
        setCursor(Cursor.getPredefinedCursor(Cursor.N_RESIZE_CURSOR));
    }

    /**
     * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
     */
    public void mouseExited(MouseEvent e) {
        if (!dragging) {
            //setCursor(Cursor.getDefaultCursor());    
        }
    }

    /**
     * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
     */
    public void mouseDragged(MouseEvent e) {
        if (orientation == SwingConstants.HORIZONTAL) {
            //int dist = e.getY() - startXY;
            //manager.setMediaWidth(manager.getMediaWidth() - dist);
        } else {
            int dist = e.getY() - startXY;
            manager.setMediaAreaHeight(manager.getMediaAreaHeight() + dist);
        }
    }

    /**
     * @see java.awt.event.MouseMotionListener#mouseMoved(java.awt.event.MouseEvent)
     */
    public void mouseMoved(MouseEvent e) {
    }

    /**
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent e) {
        // only one item
        manager.setMediaAreaHeight(ElanLayoutManager.MASTER_MEDIA_HEIGHT);
    }
}
