/*
 * File:     StatisticsFrame.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * File:     RegularAnnotationDialog.java
 * Project:  MPI Linguistic Application
 * Date:     27 January 2007
 *
 * Feature added by Ouriel Grynszpan, European contract MATHESIS IST-027574
 * CNRS UMR 7593, Paris, France
 *
 * Copyright (C) 2001-2005  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package mpi.eudico.client.annotator.gui;

import mpi.eudico.client.annotator.Constants;
import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.annotator.commands.ELANCommandFactory;

import mpi.eudico.client.annotator.export.ExportStatistics;

import mpi.eudico.client.annotator.player.ElanMediaPlayer;

import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.AbstractAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import mpi.util.ControlledVocabulary;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import java.util.Collections;
import java.util.Iterator;
import java.util.TreeSet;
import java.util.Vector;
import java.util.logging.Logger;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.border.TitledBorder;
import javax.swing.table.DefaultTableModel;


/**
 * A frame to display statistics about annotations on root tiers Statistics
 * computed are: number of occurrences, frequency, average duration, time
 * ratio, latency
 *
 * @author Ouriel Grynszpan
 */
public class StatisticsFrame extends JFrame implements ItemListener,
    ActionListener {
    /** Holds value of property DOCUMENT ME! */
    private static final Logger LOG = Logger.getLogger(StatisticsFrame.class.getName());
    private TranscriptionImpl transcription;

    //Title GUI
    private JLabel titleLabel;
    private JPanel titlePanel;

    //Tier selection GUI
    private JComboBox tierComboBox;
    private Vector rootTiers;
    private String curTier;
    private JLabel tierLabel;
    private JPanel tierSelectionPanel;

    //Statistics table GUI 
    private JScrollPane statPane;
    private JTable statTable;
    private JPanel statPanel;

    //Button panel
    private JButton saveButton;
    private JButton closeButton;
    private JPanel buttonPanel;

    //Statistics table headers
    private String annotations = "Annotations";
    private String occurrences = "Occurrences";
    private String frequency = "Frequency";
    private String averageDuration = "Average Duration";
    private String timeRatio = "Time Ratio";
    private String latency = "Latency";

    //Time boundaries
    private ElanMediaPlayer player;
    private long mediaDuration;
    private long beginBoundary;
    private long endBoundary;

    /** no selection */
    private final String EMPTY = "-";

    /**
     * Creates a new StatisticsFrame instance
     *
     * @param transcription the transcription
     */
    public StatisticsFrame(Transcription transcription) {
        super();
        this.transcription = (TranscriptionImpl) transcription;
        player = ELANCommandFactory.getViewerManager(transcription)
                                   .getMasterMediaPlayer();

        if (player != null) {
            mediaDuration = player.getMediaDuration();
            initComponents();
            postInit();
        }
    }

    /**
     * Initializes UI elements.
     */
    private void initComponents() {
        //Title components
        titleLabel = new JLabel();
        titlePanel = new JPanel();

        //Tier selection components
        tierComboBox = new JComboBox();
        rootTiers = new Vector();
        extractRootTiers();
        tierLabel = new JLabel();
        tierSelectionPanel = new JPanel();

        //Statistics table components
        statPanel = new JPanel();
        statTable = new JTable();
        statTable.setBackground(Constants.DEFAULTBACKGROUNDCOLOR);

        //statTable.setPreferredScrollableViewportSize(new Dimension(500, 500));
        statTable.setEnabled(false);

        //Initializing table
        setStatTable();

        //statTable.getTableHeader().setPreferredSize(new Dimension(500, 24));
        statPane = new JScrollPane(statTable);

        Dimension size = new Dimension(500, 100);

        //	statPane.setMinimumSize(size);
        statPane.setPreferredSize(size);

        //Button components
        saveButton = new JButton();
        closeButton = new JButton();
        buttonPanel = new JPanel();

        updateLocale();

        GridBagConstraints gridBagConstraints;
        getContentPane().setLayout(new GridBagLayout());

        Insets insets = new Insets(2, 6, 2, 6);

        //Setting title
        titleLabel.setFont(titleLabel.getFont().deriveFont((float) 16));
        titlePanel.add(titleLabel);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = GridBagConstraints.NORTH;
        gridBagConstraints.insets = insets;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 0.0;
        getContentPane().add(titlePanel, gridBagConstraints);

        //Setting tier selection panel
        tierSelectionPanel.setLayout(new GridBagLayout());

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        tierSelectionPanel.add(tierLabel, gridBagConstraints);

        tierComboBox.addItemListener(this);
        tierComboBox.setMaximumRowCount(Constants.COMBOBOX_VISIBLE_ROWS);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        tierSelectionPanel.add(tierComboBox, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = GridBagConstraints.NORTH;
        gridBagConstraints.insets = insets;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 0.0;
        getContentPane().add(tierSelectionPanel, gridBagConstraints);

        //Setting statistics panel
        //Dimension tpd = new Dimension(550, 200);
        //statPanel.setMinimumSize(tpd);
        //statPanel.setPreferredSize(tpd);
        statPanel.setLayout(new GridBagLayout());

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = GridBagConstraints.BOTH;
        gridBagConstraints.anchor = GridBagConstraints.NORTH;
        gridBagConstraints.insets = insets;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        statPanel.add(statPane, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = GridBagConstraints.BOTH;
        gridBagConstraints.anchor = GridBagConstraints.NORTH;
        gridBagConstraints.insets = insets;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 10.0;
        getContentPane().add(statPanel, gridBagConstraints);

        //Setting buttons
        buttonPanel.setLayout(new GridLayout(1, 2, 6, 0));

        saveButton.addActionListener(this);
        buttonPanel.add(saveButton);

        closeButton.addActionListener(this);
        buttonPanel.add(closeButton);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = GridBagConstraints.NORTH;
        gridBagConstraints.insets = insets;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        getContentPane().add(buttonPanel, gridBagConstraints);
    }

    /**
     * Pack, size and set location.
     */
    private void postInit() {
        pack();

        int w = 550;
        int h = 400;
        setSize((getSize().width < w) ? w : getSize().width,
            (getSize().height < h) ? h : getSize().height);
        setLocationRelativeTo(getParent());

        //setResizable(false);
    }

    /**
     * Applies localized strings to the ui elements.
     */
    public void updateLocale() {
        setTitle(ElanLocale.getString("Menu.View.Statistics"));
        titleLabel.setText(ElanLocale.getString("Statistics.Title"));
        tierSelectionPanel.setBorder(new TitledBorder(ElanLocale.getString(
                    "Statistics.Panel.Tier")));
        tierLabel.setText(ElanLocale.getString("Statistics.Label.Tier"));
        statPanel.setBorder(new TitledBorder(ElanLocale.getString(
                    "Statistics.Pane.Table")));
        saveButton.setText(ElanLocale.getString("Button.Save"));
        closeButton.setText(ElanLocale.getString("Button.Close"));
        repaint();
    }

    /**
     * Applies localized strings for statistics table header.
     */
    private void updateLocalsForVariables() {
        annotations = ElanLocale.getString("Frame.GridFrame.ColumnAnnotation");
        occurrences = ElanLocale.getString("Statistics.Occurrences");
        frequency = ElanLocale.getString("Statistics.Frequency");
        averageDuration = ElanLocale.getString("Statistics.AverageDuration");
        timeRatio = ElanLocale.getString("Statistics.TimeRatio");
        latency = ElanLocale.getString("Statistics.Latency");
    }

    /**
     * Builds statistics table for current selected tier.
     */
    private void setStatTable() {
        TierImpl tier = (TierImpl) transcription.getTierWithId(curTier);
        setTimeBoundaries();

        DefaultTableModel statTableModel = getStatistics(tier);
        statTable.setModel(statTableModel);
    }

    /**
     * Create table model with statistics for a tier
     *
     * @param tier the tier to calculate statistics for
     *
     * @return a table model
     */
    private DefaultTableModel getStatistics(TierImpl tier) {
        DefaultTableModel statTableModel = null;

        //Table headers
        String[] tableHeader = new String[6];
        updateLocalsForVariables();
        tableHeader[0] = annotations;
        tableHeader[1] = occurrences;
        tableHeader[2] = frequency;
        tableHeader[3] = averageDuration;
        tableHeader[4] = timeRatio;
        tableHeader[5] = latency;

        TreeSet annValues = getAnnotationValues(tier);
        String[][] tableStat;

        //if no annotations at all
        if ((null == annValues) || (annValues.size() == 0)) {
            tableStat = new String[0][6];
        } else {
            tableStat = new String[annValues.size()][6];

            //Fills the first columns with annotations' values 
            Iterator itAnnValues = annValues.iterator();
            int r = 0;

            while (itAnnValues.hasNext()) {
                tableStat[r][0] = (String) itAnnValues.next();
                r++;
            }

            //Occurrences
            for (int i = 0; i < annValues.size(); i++)
                tableStat[i][1] = "" + getOccurrences(tier, tableStat[i][0]);

            //Frequency
            for (int i = 0; i < annValues.size(); i++)
                tableStat[i][2] = "" + getFrequency(tier, tableStat[i][0]);

            //Average Duration
            for (int i = 0; i < annValues.size(); i++)
                tableStat[i][3] = "" +
                    getAverageDuration(tier, tableStat[i][0]);

            //Time Ratio
            for (int i = 0; i < annValues.size(); i++)
                tableStat[i][4] = "" + getTimeRatio(tier, tableStat[i][0]);

            //Latency
            for (int i = 0; i < annValues.size(); i++) {
                double latency = getLatency(tier, tableStat[i][0]);

                if (latency == (((double) (mediaDuration - beginBoundary)) / 1000)) {
                    tableStat[i][5] = "-";
                } else {
                    tableStat[i][5] = "" + latency;
                }
            }
        }

        statTableModel = new DefaultTableModel(tableStat, tableHeader);

        return statTableModel;
    }

    /**
     * Returns the set of different annotation values for a tier
     *
     * @param tier the tier to get annotation values from
     *
     * @return a treeset of annotations
     */
    private TreeSet getAnnotationValues(TierImpl tier) {
        TreeSet annValues = null;

        if ((null == tier) || (null == tier.getLinguisticType())) {
            return annValues;
        }

        //If the tier is using controlled vocabulary
        //returns all the vocabulary entries
        if (tier.getLinguisticType().isUsingControlledVocabulary()) {
            String CVname = tier.getLinguisticType()
                                .getControlledVocabylaryName();
            ControlledVocabulary CV = transcription.getControlledVocabulary(CVname);

            if ((CV != null) && (CV.getEntryValues().length > 0)) {
                String[] cvEntriesVal = CV.getEntryValues();
                annValues = new TreeSet();

                for (int i = 0; i < cvEntriesVal.length; i++)
                    if ((cvEntriesVal[i] != null) &&
                            (cvEntriesVal[i].length() > 0)) {
                        annValues.add(cvEntriesVal[i]);
                    }
            }
        }
        //If controlled vocabulary is not used, scans the annotations
        //and retrieves values
        else {
            Vector annotations = tier.getAnnotations();

            if ((annotations != null) && (annotations.size() > 0)) {
                annValues = new TreeSet();

                Iterator annosIt = annotations.iterator();

                while (annosIt.hasNext()) {
                    AbstractAnnotation ann = (AbstractAnnotation) annosIt.next();

                    if ((ann.getValue() != null) &&
                            (ann.getValue().length() > 0)) {
                        annValues.add(ann.getValue());
                    }
                }
            }
        }

        return annValues;
    }

    /**
     * Extract the root tiers
     */
    private void extractRootTiers() {
        if (transcription != null) {
            Vector tiers = transcription.getTiers();
            Iterator tierIt = tiers.iterator();
            TierImpl tier = null;

            while (tierIt.hasNext()) {
                tier = (TierImpl) tierIt.next();

                if (tier.getLinguisticType().getConstraints() == null) {
                    tierComboBox.addItem(tier.getName());
                    rootTiers.add(tier);
                }
            }

            // if there are no tiers yet
            if (tierComboBox.getModel().getSize() == 0) {
                tierComboBox.addItem(EMPTY);
            }
        } else {
            tierComboBox.addItem(EMPTY);
        }

        curTier = (String) tierComboBox.getSelectedItem();
    }

    /**
     * Selection of a different tier.
     *
     * @param ie the item event
     */
    public void itemStateChanged(ItemEvent ie) {
        if (ie.getStateChange() == ItemEvent.SELECTED) {
            String newSel = (String) tierComboBox.getSelectedItem();

            if (newSel.equals(curTier)) {
                return;
            }

            curTier = newSel;

            //Reset the statistics table
            setStatTable();
            repaint();
        }
    }

    /**
     * Computes the number of occurrences. Contiguous annotations holding the
     * same value account for  only one occurrence.
     *
     * @param tier tier, the annotation value
     * @param annValue the value of the annotation
     *
     * @return the number of occurrences
     */
    private long getOccurrences(TierImpl tier, String annValue) {
        long occ = 0;
        Vector annotations = tier.getAnnotations();

        if (annotations != null) {
            Collections.sort(annotations);

            for (int i = 0; i < annotations.size(); i++) {
                AbstractAnnotation ann = (AbstractAnnotation) annotations.get(i);

                if (annValue.equals(ann.getValue())) {
                    AbstractAnnotation prevAnn = (AbstractAnnotation) tier.getAnnotationBefore(ann);

                    //first annotation
                    if (null == prevAnn) {
                        occ++;
                    }
                    //previous annotation is not contiguous to current
                    else if (prevAnn.getEndTimeBoundary() < ann.getBeginTimeBoundary()) {
                        occ++;
                    }
                    //previous annotation is contiguous to current
                    //but values are different
                    else if (!annValue.equals(prevAnn.getValue())) {
                        occ++;
                    }
                }
            }
        }

        return occ;
    }

    /**
     * Sets the observation time boundary: - the begining boundary equals the
     * lower time boundary  of the first annotation in the whole transcription
     * - the ending boundary equals the upper time boundary  of the last
     * annotation in the whole transcription
     */
    private void setTimeBoundaries() {
        beginBoundary = mediaDuration;
        endBoundary = 0;

        Iterator itRootTier = rootTiers.iterator();

        while (itRootTier.hasNext()) {
            TierImpl tier = (TierImpl) itRootTier.next();
            Vector annotations = tier.getAnnotations();

            if (annotations != null) {
                Iterator itAnnos = annotations.iterator();

                while (itAnnos.hasNext()) {
                    AbstractAnnotation ann = (AbstractAnnotation) itAnnos.next();
                    beginBoundary = Math.min(beginBoundary,
                            ann.getBeginTimeBoundary());
                    endBoundary = Math.max(endBoundary, ann.getEndTimeBoundary());
                }
            }
        }

        LOG.info("Observation beginning boundary = " + beginBoundary +
            ", observation ending boundary = " + endBoundary);
    }

    /**
     * Computes the total observation period
     *
     * @return the duration in seconds
     */
    private double getObservationTime() {
        //is in seconds
        return (double) (endBoundary - beginBoundary) / 1000;
    }

    /**
     * Computes the frequency defined as the number of occurrences divided by
     * the total observation period
     *
     * @param tier the tier
     * @param annValue the annotation value
     *
     * @return the frequency
     */
    private double getFrequency(TierImpl tier, String annValue) {
        double freq = 0;
        long occ = getOccurrences(tier, annValue);

        // frequency is the number of occurrences divided
        // by the total duration of observation
        if (getObservationTime() > 0) {
            freq = (double) occ / getObservationTime();
        }

        return freq;
    }

    /**
     * Computes the duration for an annotation value, which is the sum of the
     * durations of all annotations holding this value
     *
     * @param tier the tier
     * @param annValue the annotation value
     *
     * @return the sum of the durations (in seconds)
     */
    private double getAnnValueDuration(TierImpl tier, String annValue) {
        double valDuration = 0;
        Vector annotations = tier.getAnnotations();

        if (annotations != null) {
            Iterator itAnnos = annotations.iterator();

            while (itAnnos.hasNext()) {
                AbstractAnnotation ann = (AbstractAnnotation) itAnnos.next();

                if (annValue.equals(ann.getValue())) {
                    double annDur = ann.getEndTimeBoundary() -
                        ann.getBeginTimeBoundary();
                    valDuration = valDuration + annDur;
                }
            }
        }

        //result in seconds
        return valDuration / 1000;
    }

    /**
     * Computes the average duration for an annotation value, defined as the
     * duration for the annotation value divided by the number of occurrences
     *
     * @param tier tier
     * @param annValue the annotation value
     *
     * @return the average duration in seconds
     */
    private double getAverageDuration(TierImpl tier, String annValue) {
        double avDuration = 0;

        if (getOccurrences(tier, annValue) > 0) {
            avDuration = getAnnValueDuration(tier, annValue) / (double) getOccurrences(tier,
                    annValue);
        }

        return avDuration;
    }

    /**
     * Computes the time ratio for an annotation value, defined as the duration
     * for the annotation value divided by total observation period
     *
     * @param tier the tier
     * @param annValue the annotation value
     *
     * @return the ratio of this value (value duration / observation time)
     *
     * @see #setTimeBoundaries()
     */
    private double getTimeRatio(TierImpl tier, String annValue) {
        double timeRatio = 0;

        if (getObservationTime() > 0) {
            timeRatio = getAnnValueDuration(tier, annValue) / getObservationTime();
        }

        return timeRatio;
    }

    /**
     * Computes the latency for an annotation value, defined as the length of
     * time between the begining of the observation period and the first
     * occurrence of the value
     *
     * @param tier tier
     * @param annValue the annotation value
     *
     * @return the latency
     */
    private double getLatency(TierImpl tier, String annValue) {
        double latency = mediaDuration - beginBoundary;
        Vector annotations = tier.getAnnotations();

        if (annotations != null) {
            Iterator itAnnos = annotations.iterator();

            while (itAnnos.hasNext()) {
                AbstractAnnotation ann = (AbstractAnnotation) itAnnos.next();

                if (annValue.equals(ann.getValue())) {
                    latency = Math.min(latency,
                            ann.getBeginTimeBoundary() - beginBoundary);
                }
            }
        }

        //result in seconds
        return latency / 1000;
    }

    /**
     * The button actions.
     *
     * @param ae the action event
     */
    public void actionPerformed(ActionEvent ae) {
        Object source = ae.getSource();

        if (source == saveButton) {
            new ExportStatistics(this, true, transcription, statTable);
        } else if (source == closeButton) {
            setVisible(false);
            dispose();
        }
    }
}
