/*
 * File:     MultiStepControl.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.gui.multistep;

import java.util.Set;


/**
 * MultiStepControl. Methods controlling navigation and actions
 * in a multiple step process (wizard).
 *
 * @author Han Sloetjes
 */
public interface MultiStepControl {
    /**
     * Called when the "Next" button is clicked.
     */
    public void nextStep();

    /**
     * Called when the "Back" button is clicked.
     */
    public void previousStep();

    /**
     * Called when the "Finish" button is clicked, or "Next" on the last panel.
     */
    public void finish();

    /**
     * Called when the "Cancel" button is clicked.
     */
    public void cancel();

    /**
     * Called when the "Help" button is clicked.
     */
    public void showHelp();

    /**
     * Returns the index of the current step in a multiple step process.
     *
     * @return the index of the current step in a multiple step process
     */
    public int getCurrentStepIndex();

    /**
     * Returns the current step.
     *
     * @return the current step.
     */
    public StepPane getCurrentStep();

    /**
     * Initiates a jump to the step at the specified index.
     *
     * @param stepIndex the index of the step to activate
     */
    public void goToStep(int stepIndex);

    /**
     * Initiates a jump to the step identified by the specified name.
     *
     * @param name the identifier of the step to activate
     */
    public void goToStep(String name);

    /**
     * Each step can store objects in a HashMap for use by other steps.
     *
     * @param key the key
     * @param value the value
     */
    public void putStepProperty(Object key, Object value);

    /**
     * Each step can retrieve objects from a HashMap, stored by other steps.
     *
     * @param key the key
     *
     * @return the objet
     */
    public Object getStepProperty(Object key);

    /**
     * Returns an enumeration of all step keys.
     *
     * @return an enumeration of all stored step keys
     */
    public Set getPropertyKeys();
}
