/*
 * File:     ImportPraatTGStep1.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.imports.praat;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.gui.multistep.MultiStepPane;
import mpi.eudico.client.annotator.gui.multistep.StepPane;

import mpi.eudico.client.annotator.util.ElanFileFilter;

import mpi.util.gui.FileAndEncodingChooser;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.io.File;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;


/**
 * First step of an import Praat TextGrid process.
 */
public class ImportPraatTGStep1 extends StepPane implements ActionListener,
    ChangeListener {
    private JTextField sourceField;
    private JButton browseButton;
    private JCheckBox pointTiersCB;
    private JSpinner durationSp;
    private JLabel durationLabel;

    /** Holds value of property DOCUMENT ME! */
    private final Integer DEF_DUR = new Integer(40);
    private String encoding;

    /**
     * Creates a new instance of the first step of the wizard.
     *
     * @param multiPane the parent pane
     */
    public ImportPraatTGStep1(MultiStepPane multiPane) {
        super(multiPane);

        initComponents();
    }

    /**
     * Initializes ui components.
     */
    public void initComponents() {
        setLayout(new GridBagLayout());
        setBorder(new EmptyBorder(12, 12, 12, 12));
        sourceField = new JTextField();
        sourceField.setEditable(false);
        browseButton = new JButton(ElanLocale.getString("Button.Browse"));
        pointTiersCB = new JCheckBox(ElanLocale.getString(
                    "ImportDialog.Praat.Label.PointTiers"));
        durationSp = new JSpinner(new SpinnerNumberModel(DEF_DUR.intValue(), 1,
                    10000, 10));
        durationSp.setEnabled(false);
        durationLabel = new JLabel(ElanLocale.getString(
                    "ImportDialog.Praat.Label.PointDuration"));
        durationLabel.setEnabled(false);

        Insets insets = new Insets(4, 6, 4, 6);
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 2;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = insets;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        add(sourceField, gbc);

        gbc.gridx = 2;
        gbc.gridwidth = 1;
        gbc.fill = GridBagConstraints.NONE;
        gbc.weightx = 0.0;
        add(browseButton, gbc);

        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.gridwidth = 3;
        gbc.insets = new Insets(20, 6, 4, 6);
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        add(pointTiersCB, gbc);

        gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.gridwidth = 1;
        gbc.insets = insets;
        gbc.fill = GridBagConstraints.NONE;
        gbc.weightx = 0.0;
        add(durationSp, gbc);

        gbc.gridx = 1;
        gbc.gridy = 2;
        gbc.gridwidth = 2;
        gbc.fill = GridBagConstraints.NONE;
        gbc.weightx = 0.0;
        add(durationLabel, gbc);

        browseButton.addActionListener(this);
        pointTiersCB.addChangeListener(this);
    }

    /**
     * @see mpi.eudico.client.annotator.gui.multistep.Step#getStepTitle()
     */
    public String getStepTitle() {
        return ElanLocale.getString("ImportDialog.Praat.Title1");
    }

    /**
     * @see mpi.eudico.client.annotator.gui.multistep.Step#enterStepBackward()
     */
    public void enterStepBackward() {
        multiPane.setButtonEnabled(MultiStepPane.NEXT_BUTTON, true);
        multiPane.setButtonEnabled(MultiStepPane.CANCEL_BUTTON, true);
    }

    /**
     * @see mpi.eudico.client.annotator.gui.multistep.Step#leaveStepForward()
     */
    public boolean leaveStepForward() {
        multiPane.putStepProperty("Source", sourceField.getText());
        multiPane.putStepProperty("Encoding", encoding);

        if (pointTiersCB.isSelected()) {
            multiPane.putStepProperty("PointTier", Boolean.TRUE);
            System.out.println(durationSp.getValue().getClass().getName());
            multiPane.putStepProperty("PointDuration",
                ((durationSp.getValue() != null) ? durationSp.getValue() : DEF_DUR));
        }

        return true;
    }

    /**
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent e) {
        File file = getFileName();

        if (file != null) {
            sourceField.setText(file.getAbsolutePath());
            multiPane.setButtonEnabled(MultiStepPane.NEXT_BUTTON, true);
        }
    }

    private File getFileName() {
        FileAndEncodingChooser chooser = new FileAndEncodingChooser();

        String[] encodings = new String[] {
                ElanLocale.getString("Button.Default"),
                FileAndEncodingChooser.UTF_8, FileAndEncodingChooser.UTF_16
            };
        chooser.setEncodings(encodings);
        chooser.setSelectedEncoding(FileAndEncodingChooser.UTF_8);
        chooser.setFileFilter(ElanFileFilter.createFileFilter(
                ElanFileFilter.PRAAT_TEXTGRID_TYPE));
        chooser.setDialogTitle(ElanLocale.getString("ImportDialog.Title.Select"));

        String dirPath = (String) Preferences.get("LastUsedPraatDir", null);

        if (dirPath == null) {
            // user.dir is probably a better choice than home.dir?
            dirPath = System.getProperty("user.dir");
        }

        chooser.setCurrentDirectory(new File(dirPath));

        int returnVal = chooser.showOpenDialog(null);

        if (returnVal == JFileChooser.APPROVE_OPTION) {
            String fullPath = chooser.getSelectedFile().getAbsolutePath();
            File fileTemp = new File(fullPath);
            encoding = chooser.getSelectedEncoding();

            //check if file exists and is a file
            if (!fileTemp.exists() || fileTemp.isDirectory()) {
                String strMessage = ElanLocale.getString("Menu.Dialog.Message1");
                strMessage += fullPath;
                strMessage += ElanLocale.getString("Menu.Dialog.Message2");

                String strError = ElanLocale.getString("Message.Error");
                JOptionPane.showMessageDialog(null, strMessage, strError,
                    JOptionPane.ERROR_MESSAGE);

                return null;
            }

            // check the file extension??
            Preferences.set("LastUsedPraatDir", fileTemp.getParent(), null);

            return fileTemp;
        }

        return null;
    }

    /**
     * Enables or disables the duration spinner of the PointTier checkbox is checked or
     * unchecked.
     *
     * @see javax.swing.event.ChangeListener#stateChanged(javax.swing.event.ChangeEvent)
     */
    public void stateChanged(ChangeEvent e) {
        durationSp.setEnabled(pointTiersCB.isSelected());
        durationLabel.setEnabled(pointTiersCB.isSelected());
    }
}
