/*
 * File:     ImportPraatTGStep3.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.imports.praat;

import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.annotator.commands.ELANCommandFactory;
import mpi.eudico.client.annotator.commands.ImportPraatGridCommand;

import mpi.eudico.client.annotator.gui.multistep.MultiStepPane;
import mpi.eudico.client.annotator.gui.multistep.StepPane;

import mpi.eudico.client.annotator.util.ProgressListener;

import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;

import java.io.File;
import java.io.IOException;

import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JProgressBar;
import javax.swing.border.EmptyBorder;


/**
 * Final step of the import Praat TextGrid process. Parsing of the file and
 * creation of new tiers and annotations.
 */
public class ImportPraatTGStep3 extends StepPane implements ProgressListener {
    private TranscriptionImpl curTranscription;
    private JProgressBar progressBar;
    private JLabel progressLabel;
    private ImportPraatGridCommand com;

    /**
     * Creates a new instance of the final step of the wizard.
     *
     * @param multiPane
     * @param curTranscriptionthe transcription
     */
    public ImportPraatTGStep3(MultiStepPane multiPane,
        TranscriptionImpl curTranscription) {
        super(multiPane);
        this.curTranscription = curTranscription;

        initComponents();
    }

    /**
     * Shows a progress bar.
     *
     * @see mpi.eudico.client.annotator.gui.multistep.StepPane#initComponents()
     */
    protected void initComponents() {
        setLayout(new GridBagLayout());
        setBorder(new EmptyBorder(12, 12, 12, 12));

        Insets insets = new Insets(4, 6, 4, 6);
        progressLabel = new JLabel("...");

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(progressLabel, gbc);

        progressBar = new JProgressBar(JProgressBar.HORIZONTAL, 0, 100);
        progressBar.setIndeterminate(true);

        gbc.gridy = 1;
        add(progressBar, gbc);
    }

    /**
     * @see mpi.eudico.client.tool.viewer.enhanced.multistep.Step#getStepTitle()
     */
    public String getStepTitle() {
        return ElanLocale.getString("ImportDialog.Praat.Title3");
    }

    /**
     * @see mpi.eudico.client.annotator.gui.multistep.Step#doFinish()
     */
    public boolean doFinish() {
        multiPane.setButtonEnabled(MultiStepPane.ALL_BUTTONS, false);

        String sourcePath = (String) multiPane.getStepProperty("Source");
        final String typeName = (String) multiPane.getStepProperty("Type");
        final Boolean includePT = (Boolean) multiPane.getStepProperty(
                "PointTier");
        final Integer duration = (Integer) multiPane.getStepProperty(
                "PointDuration");
        final String encoding = (String) multiPane.getStepProperty("Encoding");

        if (sourcePath != null) {
            final File impFile = new File(sourcePath);

            if ((impFile == null) || !impFile.exists()) {
                progressInterrupted(null, "The TextGrid file does not exist");
            }

            try {
                new Thread(new Runnable() {
                        public void run() {
                            try {
                                boolean ipt = false;
                                int dur = 40;

                                if (includePT != null) {
                                    ipt = includePT.booleanValue();

                                    if (duration != null) {
                                        dur = duration.intValue();
                                    }
                                }

                                PraatTextGrid ptg = new PraatTextGrid(impFile,
                                        ipt, dur, encoding);

                                com = (ImportPraatGridCommand) ELANCommandFactory.createCommand(curTranscription,
                                        ELANCommandFactory.IMPORT_PRAAT_GRID);
                                com.addProgressListener(ImportPraatTGStep3.this);
                                progressBar.setIndeterminate(false);
                                progressBar.setValue(0);
                                com.execute(curTranscription,
                                    new Object[] { ptg, typeName });
                            } catch (IOException ioe) {
                                progressInterrupted(null, ioe.getMessage());
                            }
                        }
                    }).start();
            } catch (Exception e) {
                progressInterrupted(null, e.getMessage());
            }
        } else {
            progressInterrupted(null, "No TextGrid file selected");
        }

        return false;
    }

    /**
     * @see mpi.eudico.client.annotator.gui.multistep.Step#enterStepForward()
     */
    public void enterStepForward() {
        doFinish();
    }

    /**
     * @see mpi.eudico.client.annotator.util.ProgressListener#progressUpdated(java.lang.Object,
     *      int, java.lang.String)
     */
    public void progressUpdated(Object source, int percent, String message) {
        if ((progressLabel != null) && (message != null)) {
            progressLabel.setText(message);
        }

        if (percent < 0) {
            percent = 0;
        } else if (percent > 100) {
            percent = 100;
        }

        progressBar.setValue(percent);

        if (percent >= 100) {
            if (com != null) {
                com.removeProgressListener(this);
            }

            showMessageDialog("Operation completed");

            multiPane.close();
        }
    }

    /**
     * @see mpi.eudico.client.annotator.util.ProgressListener#progressCompleted(java.lang.Object,
     *      java.lang.String)
     */
    public void progressCompleted(Object source, String message) {
        if (progressLabel != null) {
            progressLabel.setText(message);
        }

        if (com != null) {
            com.removeProgressListener(this);
        }

        showMessageDialog("Operation completed");

        multiPane.close();
    }

    /**
     * @see mpi.eudico.client.annotator.util.ProgressListener#progressInterrupted(java.lang.Object,
     *      java.lang.String)
     */
    public void progressInterrupted(Object source, String message) {
        if (progressLabel != null) {
            progressLabel.setText(message);
        }

        // message dialog
        showWarningDialog("Operation interrupted: " + message);

        if (com != null) {
            com.removeProgressListener(this);
        }

        multiPane.close();
    }

    /**
     * Shows a warning/error dialog with the specified message string.
     *
     * @param message the message to display
     */
    private void showWarningDialog(String message) {
        JOptionPane.showMessageDialog(this, message,
            ElanLocale.getString("Message.Warning"), JOptionPane.WARNING_MESSAGE);
    }

    /**
     * Shows a message dialog with the specified message string.
     *
     * @param message the message to display
     */
    private void showMessageDialog(String message) {
        JOptionPane.showMessageDialog(this, message, null,
            JOptionPane.INFORMATION_MESSAGE);
    }
}
