/*
 * File:     CalcOverlapsStep3.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.tier;

import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.annotator.commands.AnnotationsFromOverlapsCommand;
import mpi.eudico.client.annotator.commands.ELANCommandFactory;

import mpi.eudico.client.annotator.gui.multistep.MultiStepPane;
import mpi.eudico.client.annotator.gui.multistep.StepPane;

import mpi.eudico.client.annotator.util.ProgressListener;

import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;

import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JProgressBar;
import javax.swing.border.EmptyBorder;


/**
 * The final step, the actual calculation.  A command is created and this pane
 * is connected as progress listener. The ui is a progress monitor.
 *
 * @author Han Sloetjes
 * @version 1.0 Jan 2007
 */
public class CalcOverlapsStep3 extends StepPane implements ProgressListener {
    private TranscriptionImpl transcription;
    private JProgressBar progressBar;
    private JLabel progressLabel;
    private AnnotationsFromOverlapsCommand com;

    /**
     * Constructor
     *
     * @param multiPane the container pane
     * @param transcription the transcription
     */
    public CalcOverlapsStep3(MultiStepPane multiPane,
        TranscriptionImpl transcription) {
        super(multiPane);
        this.transcription = transcription;
        initComponents();
    }

    /**
     * @see mpi.eudico.client.annotator.gui.multistep.StepPane#initComponents()
     */
    protected void initComponents() {
        setLayout(new GridBagLayout());
        setBorder(new EmptyBorder(12, 12, 12, 12));

        Insets insets = new Insets(4, 6, 4, 6);
        GridBagConstraints gbc = new GridBagConstraints();

        progressLabel = new JLabel("...");
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.insets = insets;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        add(progressLabel, gbc);

        progressBar = new JProgressBar(JProgressBar.HORIZONTAL, 0, 100);
        progressBar.setValue(0);

        gbc.gridy = 1;
        add(progressBar, gbc);
    }

    /**
     * @see mpi.eudico.client.annotator.gui.multistep.Step#getStepTitle()
     */
    public String getStepTitle() {
        return ElanLocale.getString("OverlapsDialog.Calculating");
    }

    /**
     * Calls doFinish.
     *
     * @see mpi.eudico.client.annotator.gui.multistep.Step#enterStepForward()
     */
    public void enterStepForward() {
        doFinish();
    }

    /**
     * @see mpi.eudico.client.annotator.gui.multistep.Step#doFinish()
     */
    public boolean doFinish() {
        // disable buttons
        multiPane.setButtonEnabled(MultiStepPane.ALL_BUTTONS, false);

        String tierName1 = (String) multiPane.getStepProperty("Source-1");
        String tierName2 = (String) multiPane.getStepProperty("Source-2");
        String destTier = (String) multiPane.getStepProperty("DestTier");
        String typeName = (String) multiPane.getStepProperty("Type");
        Boolean content = (Boolean) multiPane.getStepProperty("Content");
        Integer format = (Integer) multiPane.getStepProperty("Format");

        if ((tierName1 == null) || (tierName2 == null) || (destTier == null) ||
                (typeName == null)) {
            progressInterrupted(null,
                "Illegal argument: a tier or type could not be found");
        }

        // create a command and connect as listener
        com = (AnnotationsFromOverlapsCommand) ELANCommandFactory.createCommand(transcription,
                ELANCommandFactory.ANN_FROM_OVERLAP);
        com.addProgressListener(this);
        com.execute(transcription,
            new Object[] {
                tierName1, tierName2, destTier, typeName, content, format
            });

        return false;
    }

    /**
     * @see mpi.eudico.client.annotator.util.ProgressListener#progressUpdated(java.lang.Object,
     *      int, java.lang.String)
     */
    public void progressUpdated(Object source, int percent, String message) {
        if ((progressLabel != null) && (message != null)) {
            progressLabel.setText(message);
        }

        if (percent < 0) {
            percent = 0;
        } else if (percent > 100) {
            percent = 100;
        }

        progressBar.setValue(percent);

        if (percent >= 100) {
            if (com != null) {
                com.removeProgressListener(this);
            }

            showMessageDialog("Operation completed");

            multiPane.close();
        }
    }

    /**
     * @see mpi.eudico.client.annotator.util.ProgressListener#progressCompleted(java.lang.Object,
     *      java.lang.String)
     */
    public void progressCompleted(Object source, String message) {
        if (progressLabel != null) {
            progressLabel.setText(message);
        }

        if (com != null) {
            com.removeProgressListener(this);
        }

        showMessageDialog("Operation completed");

        multiPane.close();
    }

    /**
     * @see mpi.eudico.client.annotator.util.ProgressListener#progressInterrupted(java.lang.Object,
     *      java.lang.String)
     */
    public void progressInterrupted(Object source, String message) {
        if (progressLabel != null) {
            progressLabel.setText(message);
        }

        // message dialog
        showWarningDialog("Operation interrupted: " + message);

        if (com != null) {
            com.removeProgressListener(this);
        }

        multiPane.close();
    }

    /**
     * Shows a warning/error dialog with the specified message string.
     *
     * @param message the message to display
     */
    private void showWarningDialog(String message) {
        JOptionPane.showMessageDialog(this, message,
            ElanLocale.getString("Message.Warning"), JOptionPane.WARNING_MESSAGE);
    }

    /**
     * Shows a message dialog with the specified message string.
     *
     * @param message the message to display
     */
    private void showMessageDialog(String message) {
        JOptionPane.showMessageDialog(this, message, null,
            JOptionPane.INFORMATION_MESSAGE);
    }
}
