/*
 * File:     NonContinuousRateTSTrack.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.timeseries;

import java.util.Collections;
import java.util.List;


/**
 * Implementation of a time series data track. Data are stored in a list
 * of TimeValue objects; there's no fixed sample rate.
 * There can be gaps in the track (one point is not connected with the previous one). A TimeValueStart
 * object instead of a TimeValue object indicates the begin of a new segment.
 * By default a line is drawn to connect subsequent points.
 */
public class NonContinuousRateTSTrack extends AbstractTSTrack {
    private List data;

    /**
     * Creates a new NonContinuousRateTSTrack
     */
    public NonContinuousRateTSTrack() {
        super();
        setType(TimeSeriesTrack.TIME_VALUE_LIST);
    }

    /**
     * Creates a new NonContinuousRateTSTrack
     *
     * @param name the name of the track
     * @param description the description of the track
     */
    public NonContinuousRateTSTrack(String name, String description) {
        super(name, description, TimeSeriesTrack.TIME_VALUE_LIST);
    }

    /**
     * If the time is between two times in the List return the lesser/lowest index.
     *
     * @see mpi.eudico.client.annotator.timeseries.AbstractTSTrack#getIndexForTime(long)
     */
    public int getIndexForTime(long time) {
        if (time < 0) {
            throw new IllegalArgumentException(
                "Time should be greater than or equal to 0");
        }

        if ((data == null) || data.isEmpty()) {
            return -1;
        }

        int index = -1;

        if ((time + timeOffset) >= 0) {
            TimeValue key = new TimeValue(time + timeOffset, 0);
            index = Collections.binarySearch(data, key);

            // if the time is not in the list binarySearch returns -(insertion point) -1
            // insertion point is the first value greater than the key, we want the last element
            // smaller than the key
            if (index < 0) {
                // not in the list
                index = -(index + 1);

                // index is insertion point
                if (index > 0) {
                    // get the index of the element before the insertion point
                    index--;
                }
            }
        }

        return index;
    }

    /**
     * @see mpi.eudico.client.annotator.timeseries.AbstractTSTrack#getTimeForIndex(int)
     */
    public long getTimeForIndex(int index) {
        if (index < 0) {
            throw new ArrayIndexOutOfBoundsException(
                "Index should be greater than or equal to 0");
        }

        long time = 0L;

        if (data != null) {
            if (index >= data.size()) {
                throw new ArrayIndexOutOfBoundsException("Index (" + index +
                    ") is greater than " + (data.size() - 1));
            }

            time = ((TimeValue) data.get(index)).time;
        }

        return time + timeOffset;
    }

    /**
     * @see mpi.eudico.client.annotator.timeseries.TimeSeriesTrack#setSampleRate(float)
     */
    public void setSampleRate(float rate) {
        // ignore
    }

    /**
     * @see mpi.eudico.client.annotator.timeseries.TimeSeriesTrack#getSampleRate()
     */
    public float getSampleRate() {
        return 0; // or -1??
    }

    /**
     * @see mpi.eudico.client.annotator.timeseries.TimeSeriesTrack#getSampleCount()
     */
    public int getSampleCount() {
        if (data == null) {
            return 0;
        }

        return data.size();
    }

    /**
     * @see mpi.eudico.client.annotator.timeseries.TimeSeriesTrack#getData()
     */
    public Object getData() {
        return data;
    }

    /**
     * @param data should be an ordered List of TimeValue objects
     *
     * @throws IllegalArgumentException if the data is not a List (of TimeValue objects, but this is assumed)
     *
     * @see mpi.eudico.client.annotator.timeseries.TimeSeriesTrack#setData(java.lang.Object)
     */
    public void setData(Object data) {
        if (!(data instanceof List)) {
            throw new IllegalArgumentException(
                "This track only accepts a List of TimeValue objects");
        }

        this.data = (List) data;
    }

    /**
     * @see mpi.eudico.client.annotator.timeseries.TimeSeriesTrack#getMinimum(long, long)
     */
    public float getMinimum(long begin, long end) {
        if (data == null) {
            return 0; // throw an exception?
        }

        if (begin >= end) {
            throw new IllegalArgumentException(
                "Begin time should not be greater than or equal to the end time");
        }

        if (begin < 0) {
            throw new ArrayIndexOutOfBoundsException(begin + " < 0");
        }

        if (end > data.size()) {
            // in fact: end + timeOffset > msPerSample * data.length - timeOffset
            throw new ArrayIndexOutOfBoundsException(end + " > " + data.size());
        }

        int bi = getIndexForTime(begin);
        int ei = getIndexForTime(end);
        float min = Integer.MAX_VALUE;
        float val;

        for (int i = bi; (i <= ei) && (i < data.size()); i++) {
            val = ((TimeValue) data.get(i)).value;

            if (val < min) {
                min = val;
            }
        }

        return min;
    }

    /**
     * @see mpi.eudico.client.annotator.timeseries.TimeSeriesTrack#getMaximum(long, long)
     */
    public float getMaximum(long begin, long end) {
        if (data == null) {
            return 0; // throw an exception?
        }

        if (begin >= end) {
            throw new IllegalArgumentException(
                "Begin time should not be greater than or equal to the end time");
        }

        if (begin < 0) {
            throw new ArrayIndexOutOfBoundsException(begin + " < 0");
        }

        if (end > data.size()) {
            // in fact: end + timeOffset > msPerSample * data.length - timeOffset
            throw new ArrayIndexOutOfBoundsException(end + " > " + data.size());
        }

        int bi = getIndexForTime(begin);
        int ei = getIndexForTime(end);
        float max = Integer.MIN_VALUE;
        float val;

        for (int i = bi; (i <= ei) && (i < data.size()); i++) {
            val = ((TimeValue) data.get(i)).value;

            if (val > max) {
                max = val;
            }
        }

        return max;
    }

    /**
     * @see mpi.eudico.client.annotator.timeseries.TimeSeriesTrack#getAverage(long, long)
     */
    public float getAverage(long begin, long end) {
        if (data == null) {
            return 0; // throw an exception?
        }

        if (begin >= end) {
            throw new IllegalArgumentException(
                "Begin time should not be greater than or equal to the end time");
        }

        if (begin < 0) {
            throw new ArrayIndexOutOfBoundsException(begin + " < 0");
        }

        if (end > data.size()) {
            // in fact: end + timeOffset > msPerSample * data.length - timeOffset
            throw new ArrayIndexOutOfBoundsException(end + " > " + data.size());
        }

        int bi = getIndexForTime(begin);
        int ei = getIndexForTime(end);
        int count = 0;
        float total = 0f;

        for (int i = bi; (i <= ei) && (i < data.size()); i++) {
            total += ((TimeValue) data.get(i)).value;
            count++;
        }

        if (count == 0) {
            return 0;
        } else {
            return total / count;
        }
    }
}
