/*
 * File:     AnnotationDataRecord.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.util;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.TimeSlot;

import mpi.eudico.server.corpora.clomimpl.abstr.AlignableAnnotation;

import java.io.Serializable;


/**
 * A class to store annotation data that are essential for the programmatic
 * re-creation of an annotation.
 *
 * @author Han Sloetjes
 */
public class AnnotationDataRecord implements Serializable {
    /** the name of the tier the annotation belongs to */
    private String tierName;

    /** the value of the annotation */
    private String value;

    /** the annotation id */
    private String id;

    /**
     * the value returned by annotation.getBeginTimeBoundary, can be an
     * interpolated value
     */
    private long beginTime;

    /**
     * the value returned by annotation.getEndTimeBoundary, can be an
     * interpolated value
     */
    private long endTime;

    /** the isTimeAligned value of the begin time TimeSlot */
    private boolean beginTimeAligned;

    /** the isTimeAligned value of the end time TimeSlot */
    private boolean endTimeAligned;

    /**
     * Creates an AnnotationData object from the specified Annotation.
     *
     * @param annotation the Annotation
     */
    public AnnotationDataRecord(Annotation annotation) {
        TimeSlot ts = null;

        if (annotation != null) {
            value = annotation.getValue();
            beginTime = annotation.getBeginTimeBoundary();
            endTime = annotation.getEndTimeBoundary();
            id = annotation.getId();

            tierName = annotation.getTier().getName();

            if (annotation instanceof AlignableAnnotation) {
                ts = ((AlignableAnnotation) annotation).getBegin();
                beginTimeAligned = ts.isTimeAligned();
                ts = ((AlignableAnnotation) annotation).getEnd();
                endTimeAligned = ts.isTimeAligned();
            }
        }
    }

    /**
     * Creates a new AnnotationData object from a tier name, annotation value, begintime and
     * an endtime.
     *
     * @param tierName the tiername
     * @param value the annotation value
     * @param beginTime the begintime
     * @param endTime the endtime
     */
    public AnnotationDataRecord(String tierName, String value, long beginTime,
        long endTime) {
        this.tierName = tierName;
        this.value = value;
        this.beginTime = beginTime;
        this.endTime = endTime;

        if ((this.endTime < this.beginTime) && (this.endTime >= 0)) {
            this.endTime = this.beginTime + 1;
        }

        if (this.beginTime > -1) {
            beginTimeAligned = true;
        }

        if (this.endTime > -1) {
            endTimeAligned = true;
        }
    }

    /**
     * Returns the begin time of the annotation. <br>
     * This can be  an interpolated time value.
     *
     * @return the begin time
     */
    public long getBeginTime() {
        return beginTime;
    }

    /**
     * Returns true when the TimeSlot belonging to the begin boundary is  time
     * aligned. Only an AlignableAnnotation has a TimeSlot reference.
     *
     * @return true if the begin time TimeSlot is timealignable, false
     *         otherwise
     */
    public boolean isBeginTimeAligned() {
        return beginTimeAligned;
    }

    /**
     * Returns the end time of the annotation. <br>
     * This can be  an interpolated time value.
     *
     * @return the end time
     */
    public long getEndTime() {
        return endTime;
    }

    /**
     * Returns true when the TimeSlot belonging to the end boundary is  time
     * aligned. Only an AlignableAnnotation has a TimeSlot reference.
     *
     * @return true if the end time TimeSlot is timealignable, false otherwise
     */
    public boolean isEndTimeAligned() {
        return endTimeAligned;
    }

    /**
     * Returns the name of the tier this annotation belongs to.
     *
     * @return the tier name
     */
    public String getTierName() {
        return tierName;
    }

    /**
     * The text value of annotation.
     *
     * @return the text value of the annotation
     */
    public String getValue() {
        return value;
    }

    /**
     * Returns the value of the annotation.
     *
     * @return a String representation of this object; is the same as the value
     */
    public String toString() {
        return value;
    }

    /**
     * DOCUMENT ME!
     *
     * @param beginTime DOCUMENT ME!
     */
    public void setBeginTime(long beginTime) {
        this.beginTime = beginTime;
    }

    /**
     * DOCUMENT ME!
     *
     * @param endTime DOCUMENT ME!
     */
    public void setEndTime(long endTime) {
        this.endTime = endTime;
    }

    /**
     * Returns the id.
     *
     * @return the id, or null
     */
    public String getId() {
        return id;
    }

    /**
     * Sets the id of the annotation.
     * @param id
     */
    public void setId(String id) {
        this.id = id;
    }
}
