/*
 * File:     FileUtility.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;


/**
 * A utility class that performs some Elan specific File operations.
 *
 * @author Han Sloetjes
 */
public class FileUtility {
    /**
     * Converts a path to a file URL string. Takes care of Samba related
     * problems file:///path works for all files except for samba file
     * systems, there we need file://machine/path, i.e. 2 slashes instead of 3
     * Does not support relative paths.
     *
     * @param path the path to convert
     *
     * @return a file url string
     */
    public static String pathToURLString(String path) {
        if (path == null) {
            return null;
        }

        // replace all back slashes by forward slashes
        path = path.replace('\\', '/');

        if (path.startsWith("rtsp:")) {
            // could check '//' on position 5-6
            return path;
        }

        if (path.startsWith("file:")) {
            path = path.substring(5);
        }

        // remove leading slashes and count them
        int n = 0;

        while (path.charAt(0) == '/') {
            path = path.substring(1);
            n++;
        }

        // add the file:// or file:/// prefix
        if (n == 2) {
            return "file://" + path;
        } else {
            return "file:///" + path;
        }
    }

    /**
     * Method to compare the file names in two file/url paths without their
     * path and extensions.
     *
     * @param path1 first file path
     * @param path2 seconds file path
     *
     * @return boolean true if the file names without path and extension are
     *         the same
     */
    public static boolean sameNameIgnoreExtension(String path1, String path2) {
        // no name gives fals, two nulls are equal but have no name
        if ((path1 == null) || (path2 == null)) {
            return false;
        }

        path1 = path1.replace('\\', '/'); // give all paths the same delimiter, as URLs can now be compared

        String name1 = path1;
        int delimiterPos = path1.lastIndexOf('/');

        if (delimiterPos >= 0) {
            name1 = path1.substring(delimiterPos + 1);
        }

        int extensionPos = name1.lastIndexOf('.');

        if (extensionPos >= 0) {
            name1 = name1.substring(0, extensionPos);
        }

        path2 = path2.replace('\\', '/');

        String name2 = path2;

        delimiterPos = path2.lastIndexOf('/');

        if (delimiterPos >= 0) {
            name2 = path2.substring(delimiterPos + 1);
        }

        extensionPos = name2.lastIndexOf('.');

        if (extensionPos >= 0) {
            name2 = name2.substring(0, extensionPos);
        }

        return name1.equals(name2);
    }

    /**
     * Returns the file name from a file path.
     *
     * @param path the path
     *
     * @return the filename part of the path
     */
    public static String fileNameFromPath(final String path) {
        if (path == null) {
            return null;
        }

        String name = path.replace('\\', '/');
        int delimiterPos = name.lastIndexOf('/');

        if (delimiterPos >= 0) {
            name = name.substring(delimiterPos + 1);
        }

        return name;
    }

    /**
     * Tests wether a file exists.
     *
     * @param path the filepath to test
     *
     * @return true if the file exists, false otherwise
     */
    public static boolean fileExists(final String path) {
        if (path == null) {
            return false;
        }

        // remove the file: part of the URL, leading slashes are no problem
        int colonPos = path.indexOf(':');
        String fileName = path;

        if (colonPos > -1) {
            fileName = path.substring(colonPos + 1);
        }

        // replace all back slashes by forward slashes
        fileName = fileName.replace('\\', '/');

        File file = new File(fileName);

        if (!file.exists()) {
            return false;
        } else {
            return true;
        }
    }

    //////////////////////////////////////
    // Copied from 'old' FileUtil  class
    //////////////////////////////////////

    /**
     * If file f is a file and has an extension, it is returned. Otherwise,
     * null is returned.
     *
     * @param f a File
     *
     * @return the file extension or null
     */
    public static final String getExtenstion(final File f) {
        String name = f.getName();
        int li = name.lastIndexOf(".");

        return (li == -1) ? null : name.substring(li + 1);
    }

    /**
     * Copies a file to a destination directory.
     *
     * @param sourceFile the source
     * @param destDir the destination directory
     *
     * @throws Exception DOCUMENT ME!
     */
    public void copyToDir(final File sourceFile, final File destDir)
        throws Exception {
        // this should better not be static!
        byte[] buffer = new byte[4096]; // You can change the size of this if you want.
        destDir.mkdirs(); // creates the directory if it doesn't already exist.

        File destFile = new File(destDir, sourceFile.getName());
        FileInputStream in = new FileInputStream(sourceFile);
        FileOutputStream out = new FileOutputStream(destFile);
        int len;

        while ((len = in.read(buffer)) != -1) {
            out.write(buffer, 0, len);
        }

        out.close();
        in.close();
    }

    /**
     * Copies a source file to a destination file.
     *
     * @param sourceFile the source file
     * @param destFile the destination file
     *
     * @throws Exception DOCUMENT ME!
     */
    public static void copyToFile(final File sourceFile, final File destFile)
        throws Exception {
        // this should better not be static!
        byte[] buffer = new byte[4096]; // You can change the size of this if you want.
        FileInputStream in = new FileInputStream(sourceFile);
        FileOutputStream out = new FileOutputStream(destFile);
        int len;

        while ((len = in.read(buffer)) != -1) {
            out.write(buffer, 0, len);
        }

        out.close();
        in.close();
    }

    //########## methods copied from mpi.bcarchive.repaircheck.Util ###################

    /**
     * translates URL to absolute path (removes "file:" prefix from URL)
     * @param filename name of file
     * @return returns filename without url prefixes
     */
    public static String urlToAbsPath(String filename) {
        if (filename == null) {
            return filename;
        }

        if (filename.startsWith("file:")) {
            filename = filename.substring("file:".length());
        }

        return filename;
    }

    /**
     * getRelativePath gets path of resource relative to a reference file
     * @param referenceFilename path+filename of the reference file
     * @param resourceFilename path+filename of resource
     * @return relative path
     */
    public static String getRelativePath(String referenceFilename,
        String resourceFilename) {
        if ((referenceFilename == null) || (resourceFilename == null)) {
            return resourceFilename;
        }

        if (resourceFilename.startsWith("../") ||
                resourceFilename.startsWith("./")) {
            return resourceFilename;
        }

        String[] refFields = FileUtility.urlToAbsPath(referenceFilename).split("/");
        String[] resourceFields = FileUtility.urlToAbsPath(resourceFilename)
                                             .split("/");

        int refFieldsLen = refFields.length;
        int resourceFieldsLen = resourceFields.length;

        int minFieldsLen = refFieldsLen;

        if (resourceFieldsLen < refFieldsLen) {
            minFieldsLen = resourceFieldsLen;
        }

        int i = 0;

        while ((i < minFieldsLen) &&
                refFields[i].equals((String) resourceFields[i])) {
            i++;
        }

        int restIndex = i;
        StringBuffer relPathBuf = new StringBuffer();

        if (i == (refFieldsLen - 1)) {
            relPathBuf.append("./");
        }

        while (i < (refFieldsLen - 1)) {
            relPathBuf.append("../");
            i++;
        }

        i = restIndex;

        while (i < (resourceFieldsLen - 1)) {
            relPathBuf.append(resourceFields[i] + "/");
            i++;
        }

        relPathBuf.append(resourceFields[i]);

        return relPathBuf.toString();
    }

    /**
     * getAbsolutePath returns the absolute path of a resource of which a relative path is known
     * @param referenceFilename path+filename of reference file
     * @param resourceFilename relative path of resource to reference
     * @return absolute path
     */
    public static String getAbsolutePath(String referenceFilename,
        String resourceFilename) {
        if ((referenceFilename == null) || (resourceFilename == null)) {
            return resourceFilename;
        }

        if (resourceFilename.startsWith("../") ||
                resourceFilename.startsWith("./")) {
            String[] refFields = FileUtility.urlToAbsPath(referenceFilename)
                                            .split("/");
            String[] resourceFields = FileUtility.urlToAbsPath(resourceFilename)
                                                 .split("/");

            int refFieldsLen = refFields.length;
            int resourceFieldsLen = resourceFields.length;

            int minFieldsLen = refFieldsLen;

            if (resourceFieldsLen < refFieldsLen) {
                minFieldsLen = resourceFieldsLen;
            }

            int i = 0;

            while ((i < resourceFieldsLen) &&
                    resourceFields[i].equalsIgnoreCase("..")) {
                i++;
            }

            int refBaseIndex = i;
            StringBuffer absPathBuf = new StringBuffer();

            int s = 0;

            while (s < (refFieldsLen - refBaseIndex - 1)) {
                absPathBuf.append(refFields[s] + "/");
                s++;
            }

            if (resourceFilename.startsWith("./")) {
                i = 1;
            }

            while (i < (resourceFieldsLen - 1)) {
                absPathBuf.append(resourceFields[i] + "/");
                i++;
            }

            absPathBuf.append(resourceFields[i]);

            return absPathBuf.toString();
        } else {
            return resourceFilename;
        }
    }
}
