/*
 * File:     EAFCheck.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.util;

import mpi.eudico.client.annotator.util.ClientLogger;
import mpi.eudico.client.annotator.util.FileExtension;

import mpi.eudico.server.corpora.clom.Annotation;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import java.io.File;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;


/**
 * A command line tool that checks the files that are passed. Currently it
 * calculates a per tier annotation ratio, i.e. the number of unique values
 * divided by the number of annotations.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class EAFCheck implements ClientLogger {
    private String[] files;

    /**
     * Creates a new EAFCheck instance
     */
    public EAFCheck() {
        super();
    }

    /**
     * Creates a new EAFCheck instance
     *
     * @param files the files to check
     */
    public EAFCheck(String[] files) {
        super();
        this.files = files;
    }

    private void check() {
        if ((files == null) || (files.length == 0)) {
            LOG.warning("No files specified.");
        }

        String filename;
        List allEAFFiles = new ArrayList();
        File f;

        for (int i = 0; i < files.length; i++) {
            filename = files[i];

            if (filename == null) {
                LOG.warning("File name is null: " + i);
            }

            f = new File(filename);

            if (!f.exists()) {
                LOG.warning("File does not exist: " + filename);

                continue;
            }

            if (!f.canRead()) {
                LOG.warning("No permission to read file: " + filename);

                continue;
            }

            if (f.isDirectory()) {
                LOG.info("Checking contents of directory: " + filename);
                addFiles(f, allEAFFiles);
            } else {
                allEAFFiles.add(f);
            }
        }

        for (int i = 0; i < allEAFFiles.size(); i++) {
            f = (File) allEAFFiles.get(i);
            checkFile(f);
        }
    }

    /**
     * Scans the folders for eaf files and adds them to files list,
     * recursively.
     *
     * @param dir the  or folder
     * @param files the list to add the files to
     */
    protected void addFiles(File dir, List files) {
        if ((dir == null) && (files == null)) {
            return;
        }

        File[] allSubs = dir.listFiles();

        for (int i = 0; i < allSubs.length; i++) {
            if (allSubs[i].isDirectory() && allSubs[i].canRead()) {
                addFiles(allSubs[i], files);
            } else {
                if (allSubs[i].canRead()) {
                    if (allSubs[i].getName().toLowerCase().endsWith(FileExtension.EAF_EXT[0])) {
                        // test if the file is already there??
                        files.add(allSubs[i]);
                    }
                }
            }
        }
    }

    /**
     * Checks the file.
     *
     * @param file the eaf file
     */
    private void checkFile(File file) {
        if (file == null) {
            LOG.warning("File doe not exist: null");

            return;
        }

        if (!file.exists()) {
            LOG.warning("File doe not exist: " + file.getAbsolutePath());

            return;
        }

        LOG.info("Checking file: " + file.getAbsolutePath());

        DecimalFormat format = new DecimalFormat("#0.##",
                new DecimalFormatSymbols(Locale.US));
        TranscriptionImpl trans;
        TierImpl tier;
        List annotations;
        Annotation ann;
        String val;
        List uniqueValues = new ArrayList();

        try {
            trans = new TranscriptionImpl(file.getAbsolutePath());

            List tiers = trans.getTiers();

            for (int j = 0; j < tiers.size(); j++) {
                tier = (TierImpl) tiers.get(j);

                if (tier == null) {
                    LOG.warning("Tier is null (index = " + j + ")");

                    continue;
                }

                annotations = tier.getAnnotations();
                uniqueValues.clear();

                int numAnnos = tier.getNumberOfAnnotations();

                if (numAnnos == 0) {
                    LOG.info("No annotations on Tier: " + tier.getName());

                    continue;
                }

                for (int k = 0; k < numAnnos; k++) {
                    ann = (Annotation) annotations.get(k);
                    val = ann.getValue();

                    if (!uniqueValues.contains(val)) {
                        uniqueValues.add(val);
                    }
                }

                int numVals = uniqueValues.size();

                LOG.info("\tTier: " + tier.getName() + "\n" +
                    "\t  Annotation Ratio: " +
                    format.format((numVals / (double) numAnnos)) + " A: " +
                    numAnnos + " V: " + numVals);
            }
        } catch (Exception ex) {
            // catch any exception that could occur and continue
            LOG.warning("Could not handle file: " + file.getAbsolutePath());
        }

        LOG.info("File checking complete: " + file.getAbsolutePath());
    }

    /**
     * Runs the test on the files passed on the command line.
     *
     * @param args the filenames (can be directories)
     */
    public static void main(String[] args) {
        if ((args == null) || (args.length == 0)) {
            LOG.info("usage: EAFCheck filename1 filename2");
            System.exit(0);
        }

        new EAFCheck(args).check();
    }
}
