/*
 * File:     Transcription2QtSubtitle.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.util;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;

import mpi.eudico.util.TimeRelation;

import mpi.util.TimeFormatter;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;

import java.util.List;


/**
 * Exports the annotations of a selection of tiers as a QuickTime (subtitle)
 * text file.   Created on Jul 2, 2004
 *
 * @author Alexander Klassmann
 * @version Dec 2007 support for an offset, a minimal duration per entry and merging of  tiers added
 */
public class Transcription2QtSubtitle {
    /** new line character */
    final static private String NEWLINE = "\n";

    /**
     * Exports all annotations on specified tiers
     *
     * @param transcription the source transcription
     * @param tierNames the names of the tiers to export
     * @param exportFile the file to export to
     *
     * @throws IOException any io exception
     */
    static public void exportTiers(Transcription transcription,
        String[] tierNames, File exportFile) throws IOException {
        exportTiers(transcription, tierNames, exportFile, 0L, Long.MAX_VALUE);
    }

    /**
     * Exports all annotations on specified tiers, applying the specified
     * minimal duration.
     *
     * @param transcription the source transcription
     * @param tierNames the names of the tiers to export
     * @param exportFile the file to export to
     * @param minimalDuration the minimal duration for each annotation/subtitle
     *
     * @throws IOException any io exception
     */
    static public void exportTiers(Transcription transcription,
        String[] tierNames, File exportFile, int minimalDuration)
        throws IOException {
        exportTiers(transcription, tierNames, exportFile, 0L, Long.MAX_VALUE);
    }

    /**
     * Exports all annotations on specified tiers that overlap the interval
     * specified.
     *
     * @param transcription the source transcription
     * @param tierNames the names of the tiers to export
     * @param exportFile the file to export to
     * @param beginTime the interval begin time
     * @param endTime the interval end time
     *
     * @throws IOException any io exception
     */
    static public void exportTiers(Transcription transcription,
        String[] tierNames, File exportFile, long beginTime, long endTime)
        throws IOException {
        exportTiers(transcription, tierNames, exportFile, beginTime, endTime,
            0L, 0, 0L);
    }

    /**
     * Exports all annotations on specified tiers that overlap the interval
     * specified,  applying the specified offset and minimal duration. For
     * each tier a separate text file is created.
     *
     * @param transcription the source transcription
     * @param tierNames the names of the tiers to export
     * @param exportFile the file to export to
     * @param beginTime the interval begin time
     * @param endTime the interval end time
     * @param offset the offset to add to all time values
     * @param minimalDuration the minimal duration for each annotation/subtitle
     * @param mediaDuration the total duration of the media, used to insert a
     *        dummy  subtitle at the end of the media file
     *
     * @throws IOException any io exception
     */
    static public void exportTiers(Transcription transcription,
        String[] tierNames, File exportFile, long beginTime, long endTime,
        long offset, int minimalDuration, long mediaDuration)
        throws IOException {
        if (exportFile == null) {
            return;
        }

        Annotation[] annotations = null;
        FileOutputStream out = null;
        BufferedWriter writer = null;

        if (tierNames.length == 1) {
            out = new FileOutputStream(exportFile);
            writer = new BufferedWriter(new OutputStreamWriter(out, "UTF-8"));
        }

        for (int j = 0; j < tierNames.length; j++) {
            if (tierNames.length > 1) {
                String nextName = exportFile.getAbsolutePath();
                int index = nextName.lastIndexOf('.');

                if (index > 0) {
                    nextName = nextName.substring(0, index) + "_" +
                        tierNames[j] + nextName.substring(index);
                } else {
                    nextName = nextName + "_" + tierNames[j];
                }

                out = new FileOutputStream(new File(nextName));
                writer = new BufferedWriter(new OutputStreamWriter(out, "UTF-8"));
            }

            TierImpl tier = (TierImpl) transcription.getTierWithId(tierNames[j]);

            annotations = (Annotation[]) tier.getAnnotations().toArray(new Annotation[0]);

            writer.write(
                "{QTtext}{timescale:100}{font:Arial Unicode MS}{size:12}{backColor:0,0,0}");
            writer.write(
                "{textColor:65535,65535,65535}{width:320}{justify:left}" +
                NEWLINE);

            long b;
            long e;
            long d;
            long nextB;
            long lastE = 0L;

            for (int i = 0; i < annotations.length; i++) {
                if (annotations[i] != null) {
                    if (TimeRelation.overlaps(annotations[i], beginTime, endTime)) {
                        b = annotations[i].getBeginTimeBoundary();
                        d = b + minimalDuration;
                        e = Math.max(annotations[i].getEndTimeBoundary(), d);

                        if (i < (annotations.length - 1)) {
                            nextB = annotations[i + 1].getBeginTimeBoundary();
                            e = Math.min(e, nextB);
                        }

                        if (lastE < e) {
                            lastE = e;
                        }

                        writer.write("[" + TimeFormatter.toString(b + offset) +
                            "]" + NEWLINE);
                        writer.write("{textEncoding:256} ");
                        writer.write(annotations[i].getValue() + NEWLINE);
                        writer.write("[" + TimeFormatter.toString(e + offset) +
                            "]" + NEWLINE);
                    }
                }
            }

            if (mediaDuration > lastE) {
                writer.write("[" +
                    TimeFormatter.toString(mediaDuration -
                        Math.min(40, mediaDuration - lastE)) + "]" + NEWLINE);
                writer.write("[" + TimeFormatter.toString(mediaDuration) + "]");
            }

            if (tierNames.length > 1) {
                writer.close();
            }
        }

        writer.close();
    }

    /**
     * Exports the annotations on specified tiers to one text file. The tiers
     * are "merged" overlaps are corrected.
     *
     * @param transcription the source transcription
     * @param tierNames the names of the tiers to export
     * @param exportFile the file to export to
     * @param beginTime the interval begin time
     * @param endTime the interval end time
     * @param offset the offset to add to all time values
     * @param minimalDuration the minimal duration for each annotation/subtitle
     * @param mediaDuration the total duration of the media, used to insert a
     *        dummy  subtitle at the end of the media file
     *
     * @throws IOException any io exception
     */
    static public void exportTiersMerged(Transcription transcription,
        String[] tierNames, File exportFile, long beginTime, long endTime,
        long offset, int minimalDuration, long mediaDuration)
        throws IOException {
        if (exportFile == null) {
            return;
        }

        FileOutputStream out = new FileOutputStream(exportFile);
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(out,
                    "UTF-8"));

        writer.write(
            "{QTtext}{timescale:100}{font:Arial Unicode MS}{size:12}{backColor:0,0,0}");
        writer.write("{textColor:65535,65535,65535}{width:320}{justify:left}" +
            NEWLINE);

        SubtitleSequencer sequencer = new SubtitleSequencer();

        List allUnits = sequencer.createSequence(transcription, tierNames,
                beginTime, endTime, minimalDuration, offset, true);

        SubtitleUnit unit = null;

        for (int i = 0; i < allUnits.size(); i++) {
            unit = (SubtitleUnit) allUnits.get(i);
            writer.write("[" + TimeFormatter.toString(unit.getBegin()) + "]" +
                NEWLINE);
            writer.write("{textEncoding:256} ");

            if (unit.getValue() != null) {
                //writer.write(unit.getValue().replace('\n', ' '));
                writer.write(unit.getValue() + NEWLINE);
            } else {
                for (int j = 0; j < unit.getValues().length; j++) {
                    //writer.write(unit.getValues()[j].replace('\n', ' '));
                    writer.write(unit.getValues()[j] + NEWLINE);
                }
            }

            writer.write("[" + TimeFormatter.toString(unit.getCalcEnd()) + "]" +
                NEWLINE);
        }

        if ((unit != null) && (mediaDuration > unit.getCalcEnd())) { // unit is the last unit
            writer.write("[" +
                TimeFormatter.toString(mediaDuration -
                    Math.min(40, mediaDuration - unit.getCalcEnd())) + "]" +
                NEWLINE);
            writer.write("[" + TimeFormatter.toString(mediaDuration) + "]");
        }

        writer.close();
    }

    /**
     * Exports the annotations on specified tiers to one text file. The tiers
     * are "merged" overlaps are corrected.
     *
     * @param transcription the source transcription
     * @param tierNames the names of the tiers to export
     * @param exportFile the file to export to
     * @param minimalDuration the minimal duration for each annotation/subtitle
     *
     * @throws IOException any io exception
     */
    static public void exportTiersMerged(Transcription transcription,
        String[] tierNames, File exportFile, int minimalDuration)
        throws IOException {
        exportTiersMerged(transcription, tierNames, exportFile, 0L,
            Long.MAX_VALUE, 0L, minimalDuration, 0L);
    }
}
