/*
 * File:     Transcription2TabDelimitedText.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.util;

import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.annotator.export.TabExportTableModel;

import mpi.eudico.client.annotator.util.AnnotationDataComparator;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.AnnotationCore;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import mpi.eudico.util.TimeRelation;

import mpi.search.content.result.model.ContentMatch;

import mpi.util.TimeFormatter;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;

import java.nio.charset.UnsupportedCharsetException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;


/**
 * Created on Apr 20, 2004 Jun 2005: added optional export of time values in
 * milliseconds
 *
 * @author Alexander Klassmann
 * @version June 30, 2005
 * @version Aug 2005 Identity removed
 */
public class Transcription2TabDelimitedText {
    /** Holds value of property DOCUMENT ME! */
    final static public String TAB = "\t";

    /** Holds value of property DOCUMENT ME! */
    final static private String NEWLINE = "\n";

    /**
     * Exports all annotations on specified tiers
     *
     * @param transcription
     * @param tierNames
     * @param exportFile
     *
     * @throws IOException
     */
    static public void exportTiers(Transcription transcription,
        String[] tierNames, File exportFile) throws IOException {
        exportTiers(transcription, tierNames, exportFile, 0L, Long.MAX_VALUE);
    }

    /**
     * Exports annotations that overlap with specified time interval
     *
     * @param transcription
     * @param tierNames
     * @param exportFile
     * @param beginTime
     * @param endTime
     *
     * @throws IOException
     */
    static public void exportTiers(Transcription transcription,
        String[] tierNames, File exportFile, long beginTime, long endTime)
        throws IOException {
        exportTiers(transcription, tierNames, exportFile, beginTime, endTime,
            true, true, true, true, true, false);
    }

    /**
     * Exports annotations of the specified tiers that overlap the specified
     * interval. Which time information and in which time formats should be
     * included in the output is specified by the last 6 params.
     *
     * @param transcription the source transcription
     * @param tierNames the names of the tiers to export
     * @param exportFile the destination file
     * @param beginTime begin time of the selected interval
     * @param endTime end time of the selected interval
     * @param includeBeginTime if true include the begin time of annotations
     * @param includeEndTime if true include the end time of annotations
     * @param includeDuration if true include the duration of annotations
     * @param includeHHMM if true include the times in hh:mm:ss.ms format
     * @param includeSSMS if true include the times in ss.ms format
     * @param includeMS if true include the times in ms format
     *
     * @throws IOException i/o exception
     */
    static public void exportTiers(Transcription transcription,
        String[] tierNames, File exportFile, long beginTime, long endTime,
        boolean includeBeginTime, boolean includeEndTime,
        boolean includeDuration, boolean includeHHMM, boolean includeSSMS,
        boolean includeMS) throws IOException {
        exportTiers(transcription, tierNames, exportFile, "UTF-8", beginTime,
            endTime, includeBeginTime, includeEndTime, includeDuration,
            includeHHMM, includeSSMS, includeMS);
    }

    /**
     * Exports annotations of the specified tiers that overlap the specified
     * interval. Which time information and in which time formats should be
     * included in the output is specified by the last 6 params.
     *
     * @param transcription the source transcription
     * @param tierNames the names of the tiers to export
     * @param exportFile the destination file
     * @param charEncoding the character encoding
     * @param beginTime begin time of the selected interval
     * @param endTime end time of the selected interval
     * @param includeBeginTime if true include the begin time of annotations
     * @param includeEndTime if true include the end time of annotations
     * @param includeDuration if true include the duration of annotations
     * @param includeHHMM if true include the times in hh:mm:ss.ms format
     * @param includeSSMS if true include the times in ss.ms format
     * @param includeMS if true include the times in ms format
     *
     * @throws IOException i/o exception
     */
    static public void exportTiers(Transcription transcription,
        String[] tierNames, File exportFile, String charEncoding,
        long beginTime, long endTime, boolean includeBeginTime,
        boolean includeEndTime, boolean includeDuration, boolean includeHHMM,
        boolean includeSSMS, boolean includeMS) throws IOException {
        exportTiers(transcription, tierNames, exportFile, charEncoding,
            beginTime, endTime, includeBeginTime, includeEndTime,
            includeDuration, includeHHMM, includeSSMS, includeMS, false, false,
            0L);
    }

    /**
     * Exports annotations of the specified tiers that overlap the specified
     * interval. Which time information and in which time formats should be
     * included in the output is specified by the last 6 params.
     *
     * @param transcription the source transcription
     * @param tierNames the names of the tiers to export
     * @param exportFile the destination file
     * @param charEncoding the character encoding
     * @param beginTime begin time of the selected interval
     * @param endTime end time of the selected interval
     * @param includeBeginTime if true include the begin time of annotations
     * @param includeEndTime if true include the end time of annotations
     * @param includeDuration if true include the duration of annotations
     * @param includeHHMM if true include the times in hh:mm:ss.ms format
     * @param includeSSMS if true include the times in ss.ms format
     * @param includeMS if true include the times in ms format
     * @param includeSMPTE if true include times in hh:mm:ss:ff format
     * @param palFormat if includeSMPTE is true: use PAL timecode if palFormat is true,
     * otherwise use NTSC drop frame
     * @param mediaOffset the (master) media offset to be added to the annotations' time values
     *
     * @throws IOException i/o exception
     */
    static public void exportTiers(Transcription transcription,
        String[] tierNames, File exportFile, String charEncoding,
        long beginTime, long endTime, boolean includeBeginTime,
        boolean includeEndTime, boolean includeDuration, boolean includeHHMM,
        boolean includeSSMS, boolean includeMS, boolean includeSMPTE,
        boolean palFormat, long mediaOffset) throws IOException {
        if (exportFile == null) {
            throw new IOException("No destination file specified for export");
        }

        FileOutputStream out = new FileOutputStream(exportFile);
        OutputStreamWriter osw = null;

        try {
            osw = new OutputStreamWriter(out, charEncoding);
        } catch (UnsupportedCharsetException uce) {
            osw = new OutputStreamWriter(out, "UTF-8");
        }

        BufferedWriter writer = new BufferedWriter(osw);
        AnnotationCore[] annotations = null;

        for (int j = 0; j < tierNames.length; j++) {
            TierImpl tier = (TierImpl) transcription.getTierWithId(tierNames[j]);

            annotations = (AnnotationCore[]) tier.getAnnotations().toArray(new AnnotationCore[0]);

            for (int i = 0; i < annotations.length; i++) {
                if (annotations[i] != null) {
                    if (TimeRelation.overlaps(annotations[i], beginTime, endTime)) {
                        //writer.write(tierNames[j] + getTabString(annotations[i]));
                        writer.write(tierNames[j] +
                            getTabString(annotations[i], includeBeginTime,
                                includeEndTime, includeDuration, includeHHMM,
                                includeSSMS, includeMS, includeSMPTE,
                                palFormat, mediaOffset));
                    }
                }
            }
        }

        writer.close();
    }

    /**
     * Exports the annotations of each tier in a separate column. If annotations
     * of multiple tiers share the same begin AND end time they will be on the same
     * row in the output. All annotations are collected in one list and sorted on
     * the time values.
     * Exports annotations of the specified tiers that overlap the specified
     * interval. Which time information and in which time formats should be
     * included in the output is specified by the last 6 params.
     *
     * @param transcription the source transcription
     * @param tierNames the names of the tiers to export
     * @param exportFile the destination file
     * @param charEncoding the character encoding
     * @param beginTime begin time of the selected interval
     * @param endTime end time of the selected interval
     * @param includeBeginTime if true include the begin time of annotations
     * @param includeEndTime if true include the end time of annotations
     * @param includeDuration if true include the duration of annotations
     * @param includeHHMM if true include the times in hh:mm:ss.ms format
     * @param includeSSMS if true include the times in ss.ms format
     * @param includeMS if true include the times in ms format
     * @param includeSMPTE if true include times in hh:mm:ss:ff format
     * @param palFormat if includeSMPTE is true: use PAL timecode if palFormat is true,
     * otherwise use NTSC drop frame
     * @param mediaOffset the (master) media offset to be added to the annotations' time values
     *
     * @throws IOException i/o exception
     */
    static public void exportTiersColumnPerTier(Transcription transcription,
        String[] tierNames, File exportFile, String charEncoding,
        long beginTime, long endTime, boolean includeBeginTime,
        boolean includeEndTime, boolean includeDuration, boolean includeHHMM,
        boolean includeSSMS, boolean includeMS, boolean includeSMPTE,
        boolean palFormat, long mediaOffset) throws IOException {
        if (exportFile == null) {
            throw new IOException("No destination file specified for export");
        }

        FileOutputStream out = new FileOutputStream(exportFile);
        OutputStreamWriter osw = null;

        try {
            osw = new OutputStreamWriter(out, charEncoding);
        } catch (UnsupportedCharsetException uce) {
            osw = new OutputStreamWriter(out, "UTF-8");
        }

        BufferedWriter writer = new BufferedWriter(osw);
        List allAnnotations = new ArrayList(100);
        AnnotationCore[] annotations = null;
        TierImpl tier;

        for (int j = 0; j < tierNames.length; j++) {
            tier = (TierImpl) transcription.getTierWithId(tierNames[j]);

            if (tier != null) {
                annotations = (AnnotationCore[]) tier.getAnnotations().toArray(new AnnotationCore[0]);

                for (int i = 0; i < annotations.length; i++) {
                    if (annotations[i] != null) {
                        if (TimeRelation.overlaps(annotations[i], beginTime,
                                    endTime)) {
                            allAnnotations.add(annotations[i]);
                        }

                        if (annotations[i].getBeginTimeBoundary() > endTime) {
                            break;
                        }
                    }
                }
            }
        }
         // end tier loop

        Collections.sort(allAnnotations, new AnnotationDataComparator());

        // group the annotations that share the same begin and end time
        TabExportTableModel model = new TabExportTableModel(allAnnotations,
                tierNames);

        // write header, write each row, taking into account the formatting flags
        writer.write(getHeaders(model, includeBeginTime, includeEndTime,
                includeDuration, includeHHMM, includeSSMS, includeMS,
                includeSMPTE, palFormat));
        writeRows(writer, model, includeBeginTime, includeEndTime,
            includeDuration, includeHHMM, includeSSMS, includeMS, includeSMPTE,
            palFormat, mediaOffset);

        writer.close();
    }

    /**
     * Exports annotations of the specified tiers that overlap the specified
     * interval. Which time information and in which time formats should be
     * included in the output is specified by the last 6 params.
     *
     * @param files the source transcription files
     * @param tierNames the names of the tiers to export
     * @param exportFile the destination file
     * @param charEncoding the character encoding
     * @param includeBeginTime if true include the begin time of annotations
     * @param includeEndTime if true include the end time of annotations
     * @param includeDuration if true include the duration of annotations
     * @param includeHHMM if true include the times in hh:mm:ss.ms format
     * @param includeSSMS if true include the times in ss.ms format
     * @param includeMS if true include the times in ms format
     * @param includeSMPTE if true include times in hh:mm:ss:ff format
     * @param palFormat if includeSMPTE is true: use PAL timecode if palFormat is true,
     * otherwise use NTSC drop frame
     *
     * @throws IOException i/o exception
     */
    static public void exportTiers(List files, String[] tierNames,
        File exportFile, String charEncoding, boolean includeBeginTime,
        boolean includeEndTime, boolean includeDuration, boolean includeHHMM,
        boolean includeSSMS, boolean includeMS, boolean includeSMPTE,
        boolean palFormat) throws IOException {
        if (exportFile == null) {
            throw new IOException("No destination file specified for export");
        }

        if ((files == null) || (files.size() == 0)) {
            throw new IOException("No files specified for export");
        }

        FileOutputStream out = new FileOutputStream(exportFile);
        OutputStreamWriter osw = null;

        try {
            osw = new OutputStreamWriter(out, charEncoding);
        } catch (UnsupportedCharsetException uce) {
            osw = new OutputStreamWriter(out, "UTF-8");
        }

        BufferedWriter writer = new BufferedWriter(osw);
        AnnotationCore[] annotations = null;

        File file;
        TranscriptionImpl trans;
        TierImpl tier;
        String name;
        String tabString;

        for (int i = 0; i < files.size(); i++) {
            file = (File) files.get(i);

            if (file == null) {
                continue;
            }

            try {
                trans = new TranscriptionImpl(file.getAbsolutePath());

                List tiers = trans.getTiers();

tierloop: 
                for (int j = 0; j < tiers.size(); j++) {
                    tier = (TierImpl) tiers.get(j);
                    name = tier.getName();

                    if (tierNames != null) {
                        for (int k = 0; k < tierNames.length; k++) {
                            if (tierNames[k].equals(name)) {
                                break;
                            }

                            if (k == (tierNames.length - 1)) {
                                // not in the list, next tier
                                continue tierloop;
                            }
                        }
                    }

                    annotations = (AnnotationCore[]) tier.getAnnotations()
                                                         .toArray(new AnnotationCore[0]);

                    for (int k = 0; k < annotations.length; k++) {
                        if (annotations[k] != null) {
                            writer.write(name);
                            tabString = getTabString(annotations[k],
                                    includeBeginTime, includeEndTime,
                                    includeDuration, includeHHMM, includeSSMS,
                                    includeMS, includeSMPTE, palFormat, 0);
                            writer.write(tabString, 0, tabString.length() - 1);
                            writer.write(TAB + file.getAbsolutePath() +
                                NEWLINE);
                        }
                    }
                }
            } catch (Exception ex) {
                // catch any exception that could occur and continue
                System.out.println("Could not handle file: " +
                    file.getAbsolutePath());
            }
        }

        writer.close();
    }

    /**
     * Returns the column headers / labels.
     * @param model the table model used to group annotations with the same begin and end time
     * in one row
     * @param includeBeginTime
     * @param includeEndTime
     * @param includeDuration
     * @param includeHHMM
     * @param includeSSMS
     * @param includeMS
     * @param includeSMPTE
     * @param palFormat
     * @return the header labels delimited by tabs
     */
    private static String getHeaders(TabExportTableModel model,
        boolean includeBeginTime, boolean includeEndTime,
        boolean includeDuration, boolean includeHHMM, boolean includeSSMS,
        boolean includeMS, boolean includeSMPTE, boolean palFormat) {
        StringBuffer buf = new StringBuffer();

        if (includeBeginTime) {
            if (includeHHMM) {
                buf.append(ElanLocale.getString(
                        "Frame.GridFrame.ColumnBeginTime") + " - " +
                    ElanLocale.getString("TimeCodeFormat.TimeCode") + TAB);
            }

            if (includeSSMS) {
                buf.append(ElanLocale.getString(
                        "Frame.GridFrame.ColumnBeginTime") + " - " +
                    ElanLocale.getString("TimeCodeFormat.Seconds") + TAB);
            }

            if (includeMS) {
                buf.append(ElanLocale.getString(
                        "Frame.GridFrame.ColumnBeginTime") + " - " +
                    ElanLocale.getString("TimeCodeFormat.MilliSec") + TAB);
            }

            if (includeSMPTE) {
                if (palFormat) {
                    buf.append(ElanLocale.getString(
                            "Frame.GridFrame.ColumnBeginTime") + " - " +
                        ElanLocale.getString("TimeCodeFormat.SMPTE.PAL") + TAB);
                } else {
                    buf.append(ElanLocale.getString(
                            "Frame.GridFrame.ColumnBeginTime") + " - " +
                        ElanLocale.getString("TimeCodeFormat.SMPTE.NTSC") +
                        TAB);
                }
            }
        }

        if (includeEndTime) {
            if (includeHHMM) {
                buf.append(ElanLocale.getString("Frame.GridFrame.ColumnEndTime") +
                    " - " + ElanLocale.getString("TimeCodeFormat.TimeCode") +
                    TAB);
            }

            if (includeSSMS) {
                buf.append(ElanLocale.getString("Frame.GridFrame.ColumnEndTime") +
                    " - " + ElanLocale.getString("TimeCodeFormat.Seconds") +
                    TAB);
            }

            if (includeMS) {
                buf.append(ElanLocale.getString("Frame.GridFrame.ColumnEndTime") +
                    " - " + ElanLocale.getString("TimeCodeFormat.MilliSec") +
                    TAB);
            }

            if (includeSMPTE) {
                if (palFormat) {
                    buf.append(ElanLocale.getString(
                            "Frame.GridFrame.ColumnEndTime") + " - " +
                        ElanLocale.getString("TimeCodeFormat.SMPTE.PAL") + TAB);
                } else {
                    buf.append(ElanLocale.getString(
                            "Frame.GridFrame.ColumnEndTime") + " - " +
                        ElanLocale.getString("TimeCodeFormat.SMPTE.NTSC") +
                        TAB);
                }
            }
        }

        if (includeDuration) {
            if (includeHHMM) {
                buf.append(ElanLocale.getString(
                        "Frame.GridFrame.ColumnDuration") + " - " +
                    ElanLocale.getString("TimeCodeFormat.TimeCode") + TAB);
            }

            if (includeSSMS) {
                buf.append(ElanLocale.getString(
                        "Frame.GridFrame.ColumnDuration") + " - " +
                    ElanLocale.getString("TimeCodeFormat.Seconds") + TAB);
            }

            if (includeMS) {
                buf.append(ElanLocale.getString(
                        "Frame.GridFrame.ColumnDuration") + " - " +
                    ElanLocale.getString("TimeCodeFormat.MilliSec") + TAB);
            }

            if (includeSMPTE) {
                if (palFormat) {
                    buf.append(ElanLocale.getString(
                            "Frame.GridFrame.ColumnDuration") + " - " +
                        ElanLocale.getString("TimeCodeFormat.SMPTE.PAL") + TAB);
                } else {
                    buf.append(ElanLocale.getString(
                            "Frame.GridFrame.ColumnDuration") + " - " +
                        ElanLocale.getString("TimeCodeFormat.SMPTE.NTSC") +
                        TAB);
                }
            }
        }

        for (int i = 2; i < model.getColumnCount(); i++) {
            buf.append(model.getColumnName(i));

            if (i != (model.getColumnCount() - 1)) {
                buf.append(TAB);
            }
        }

        buf.append(NEWLINE);

        return buf.toString();
    }

    private static void writeRows(BufferedWriter writer,
        TabExportTableModel model, boolean includeBeginTime,
        boolean includeEndTime, boolean includeDuration, boolean includeHHMM,
        boolean includeSSMS, boolean includeMS, boolean includeSMPTE,
        boolean palFormat, long mediaOffset) throws IOException {
        long bt;
        long et;
        Object value;

        for (int i = 0; i < model.getRowCount(); i++) {
            bt = ((Long) model.getValueAt(i, 0)).longValue() + mediaOffset;
            et = ((Long) model.getValueAt(i, 1)).longValue() + mediaOffset;

            if (includeBeginTime) {
                if (includeHHMM) {
                    writer.write(TimeFormatter.toString(bt) + TAB);
                }

                if (includeSSMS) {
                    writer.write(Double.toString(bt / 1000.0) + TAB);
                }

                if (includeMS) {
                    writer.write(bt + TAB);
                }

                if (includeSMPTE) {
                    if (palFormat) {
                        writer.write(TimeFormatter.toTimecodePAL(bt) + TAB);
                    } else {
                        writer.write(TimeFormatter.toTimecodeNTSC(bt) + TAB);
                    }
                }
            }

            if (includeEndTime) {
                if (includeHHMM) {
                    writer.write(TimeFormatter.toString(et) + TAB);
                }

                if (includeSSMS) {
                    writer.write(Double.toString(et / 1000.0) + TAB);
                }

                if (includeMS) {
                    writer.write(et + TAB);
                }

                if (includeSMPTE) {
                    if (palFormat) {
                        if (palFormat) {
                            writer.write(TimeFormatter.toTimecodePAL(et) + TAB);
                        } else {
                            writer.write(TimeFormatter.toTimecodeNTSC(et) +
                                TAB);
                        }
                    }
                }
            }

            if (includeDuration) {
                long d = et - bt;

                if (includeHHMM) {
                    writer.write(TimeFormatter.toString(d) + TAB);
                }

                if (includeSSMS) {
                    writer.write(Double.toString(d / 1000.0) + TAB);
                }

                if (includeMS) {
                    writer.write(d + TAB);
                }
            }

            // write annotations in the columns
            for (int j = 2; j < model.getColumnCount(); j++) {
                value = model.getValueAt(i, j);

                if (value != null) {
                    writer.write(value.toString());
                }

                if (j != (model.getColumnCount() - 1)) {
                    writer.write(TAB);
                }
            }

            writer.write(NEWLINE);
        }

        writer.write(NEWLINE);
    }

    /**
     * Exports a List of AnnotationCores to Tab limited text
     *
     * @param tierName DOCUMENT ME!
     * @param annotations
     * @param exportFile
     *
     * @throws IOException
     */
    static public void exportAnnotations(String tierName, List annotations,
        File exportFile) throws IOException {
        if (exportFile == null) {
            return;
        }

        FileOutputStream out = new FileOutputStream(exportFile);
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(out,
                    "UTF-8"));

        for (int i = 0; i < annotations.size(); i++) {
            if (annotations.get(i) instanceof AnnotationCore) {
                writer.write(tierName +
                    getTabString((AnnotationCore) annotations.get(i)));
            }
        }

        writer.close();
    }

    /**
     * DOCUMENT ME!
     *
     * @param annotationCore
     *
     * @return String
     */
    static public String getTabString(AnnotationCore annotationCore) {
        return getTabString(annotationCore, true, true);
    }

    /**
     * DOCUMENT ME!
     *
     * @param annotationCore
     * @param HHMMformat if true, output of times in HHMMss.mmm format
     * @param SSMSFormat if true, output of times in second.milliseconds format
     *
     * @return String
     */
    static public String getTabString(AnnotationCore annotationCore,
        boolean HHMMformat, boolean SSMSFormat) {
        return getTabString(annotationCore, true, true, true, HHMMformat,
            SSMSFormat, false);
    }

    /**
     * Creates a tab delimited string with time information and the annotation
     * value.
     *
     * @param annotationCore the annotation
     * @param beginTime include begintime in output
     * @param endTime include endtime in output
     * @param duration include duration in output
     * @param HHMMformat if true, output of times in HHMMss.mmm format
     * @param SSMSFormat if true, output of times in sec.milliseconds format
     * @param MSFormat if true, output of times in milliseconds format
     *
     * @return String the tab seperated result string
     */
    static public String getTabString(AnnotationCore annotationCore,
        boolean beginTime, boolean endTime, boolean duration,
        boolean HHMMformat, boolean SSMSFormat, boolean MSFormat) {
        return getTabString(annotationCore, beginTime, endTime, duration,
            HHMMformat, SSMSFormat, MSFormat, false, false, 0L);
    }

    /**
     * Creates a tab delimited string with time information and the annotation
     * value.
     *
     * @param annotationCore the annotation
     * @param beginTime include begintime in output
     * @param endTime include endtime in output
     * @param duration include duration in output
     * @param HHMMformat if true, output of times in HHMMss.mmm format
     * @param SSMSFormat if true, output of times in sec.milliseconds format
     * @param MSFormat if true, output of times in milliseconds format
     * @param SMPTEFormat if true, output times in SMPTE timecode format
     * @param PAL if SMPTEFormat is true and PAL is true use PAL timecode,
     * if SMPTEFormat is true and PAL is false use NTSC drop frame timecode
     * @param mediaOffset the (master) media offset to be added to the annotations' time values
     *
     * @return String the tab seperated result string
     */
    static public String getTabString(AnnotationCore annotationCore,
        boolean beginTime, boolean endTime, boolean duration,
        boolean HHMMformat, boolean SSMSFormat, boolean MSFormat,
        boolean SMPTEFormat, boolean PAL, long mediaOffset) {
        StringBuffer sb = new StringBuffer(TAB);

        long bt = annotationCore.getBeginTimeBoundary() + mediaOffset;
        long et = annotationCore.getEndTimeBoundary() + mediaOffset;

        // begin time
        if (beginTime) {
            if (HHMMformat) {
                sb.append(TimeFormatter.toString(bt) + TAB);
            }

            if (SSMSFormat) {
                sb.append(Double.toString(bt / 1000.0) + TAB);
            }

            if (MSFormat) {
                sb.append(bt + TAB);
            }

            if (SMPTEFormat) {
                if (PAL) {
                    sb.append(TimeFormatter.toTimecodePAL(bt) + TAB);
                } else {
                    sb.append(TimeFormatter.toTimecodeNTSC(bt) + TAB);
                }
            }
        }

        // end time
        if (endTime) {
            if (HHMMformat) {
                sb.append(TimeFormatter.toString(et) + TAB);
            }

            if (SSMSFormat) {
                sb.append(Double.toString(et / 1000.0) + TAB);
            }

            if (MSFormat) {
                sb.append(et + TAB);
            }

            if (SMPTEFormat) {
                if (PAL) {
                    sb.append(TimeFormatter.toTimecodePAL(et) + TAB);
                } else {
                    sb.append(TimeFormatter.toTimecodeNTSC(et) + TAB);
                }
            }
        }

        // duration
        if (duration) {
            long d = et - bt;

            if (HHMMformat) {
                sb.append(TimeFormatter.toString(d) + TAB);
            }

            if (SSMSFormat) {
                sb.append(Double.toString(d / 1000.0) + TAB);
            }

            if (MSFormat) {
                sb.append(d + TAB);
            }
        }

        sb.append(annotationCore.getValue().replace('\n', ' ') + NEWLINE);

        return sb.toString();
    }
}
