/*
 * File:     PraatTextGridEncoder.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.server.corpora.clomimpl.praat;

import mpi.eudico.client.annotator.util.AnnotationDataRecord;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.AnnotationDocEncoder;
import mpi.eudico.server.corpora.clom.EncoderInfo;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;

import mpi.eudico.server.corpora.util.ServerLogger;

import java.io.BufferedWriter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;

import java.util.ArrayList;
import java.util.List;
import java.util.Vector;


/**
 * An encoder class that encodes tiers and annotations as Praat TextGrid
 * Interval Tiers.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class PraatTextGridEncoder implements AnnotationDocEncoder, ServerLogger {
    /** Holds value of property DOCUMENT ME! */
    private final String indent = "    ";

    /** Holds value of property DOCUMENT ME! */
    private final String indent2 = "        ";

    /** Holds value of property DOCUMENT ME! */
    private final String indent3 = "            ";

    /** Holds value of property DOCUMENT ME! */
    private final String xmin = "xmin = ";

    /** Holds value of property DOCUMENT ME! */
    private final String xmax = "xmax = ";

    /** Holds value of property DOCUMENT ME! */
    private final String intervals = "intervals [";

    /** Holds value of property DOCUMENT ME! */
    private final String tx = "text = ";

    /** Holds value of property DOCUMENT ME! */
    private final String NEW_LINE = "\n";
    private String encoding;

    /**
     * Constructor.
     */
    public PraatTextGridEncoder() {
        super();
    }

    /**
     * @see mpi.eudico.server.corpora.clom.AnnotationDocEncoder#encodeAndSave(mpi.eudico.server.corpora.clom.Transcription,
     *      mpi.eudico.server.corpora.clom.EncoderInfo, java.util.Vector,
     *      java.lang.String)
     */
    public void encodeAndSave(Transcription theTranscription,
        EncoderInfo theEncoderInfo, List tierOrder, String path)
        throws IOException {
        // if the transcription or tier vector or path is null throw exception
        if (theTranscription == null) {
            LOG.warning("The transcription is null");
            throw new IllegalArgumentException("The transcription is null.");
        }

        if ((tierOrder == null) || (tierOrder.size() == 0)) {
            LOG.warning("No tiers have been specified for export");
            throw new IllegalArgumentException("No tiers specified for export");
        }

        if (path == null) {
            LOG.warning("No file path for the TextGrid file has been specified");
            throw new IllegalArgumentException(
                "No file path for the TextGrid file has been specified");
        }

        long bt = 0;
        long et = 0;

        if (theEncoderInfo instanceof PraatTGEncoderInfo) {
            bt = ((PraatTGEncoderInfo) theEncoderInfo).getBeginTime();
            et = ((PraatTGEncoderInfo) theEncoderInfo).getEndTime();
            encoding = ((PraatTGEncoderInfo) theEncoderInfo).getEncoding();
        }

        //try {
        OutputStreamWriter out = null;

        try {
            if (encoding != null) {
                out = new OutputStreamWriter(new FileOutputStream(path),
                        encoding);
            } else {
                out = new OutputStreamWriter(new FileOutputStream(path));
            }
        } catch (UnsupportedEncodingException uee) {
            LOG.warning("Encoding not supported: " + uee.getMessage());
            out = new OutputStreamWriter(new FileOutputStream(path));
        }

        BufferedWriter writer = new BufferedWriter(out);
        writer.write("File type = \"ooTextFile\"");
        writer.write(NEW_LINE);
        writer.write("Object class = \"TextGrid\"");
        writer.write(NEW_LINE + NEW_LINE);
        writer.write(xmin + (bt / 1000d));
        writer.write(NEW_LINE);
        writer.write(xmax + (et / 1000d));
        writer.write(NEW_LINE);
        writer.write("tiers? <exists>");
        writer.write(NEW_LINE);
        writer.write("size = " + tierOrder.size());
        writer.write(NEW_LINE);
        writer.write("item []:");
        writer.write(NEW_LINE);

        // System.out.println("Enc: " + out.getEncoding());
        writeTiers(theTranscription, tierOrder, bt, et, writer);
        writer.flush();
        writer.close();

        //} catch (IOException  ex) {
        //throw new Exception(ex);
        //  LOG.severe("IO Exception occurred: " + ex.getMessage());
        //ex.printStackTrace();
        //} 
    }

    private void writeTiers(Transcription theTranscription, List tiers,
        long bt, long et, Writer out) throws IOException {
        if ((tiers == null) || (tiers.size() == 0)) {
            return;
        }

        List annotations = new ArrayList();
        TierImpl tier = null;
        Annotation ann1 = null;
        Annotation ann2 = null;

        for (int i = 0; i < tiers.size(); i++) {
            tier = (TierImpl) theTranscription.getTierWithId((String) tiers.get(
                        i));
            out.write(indent);
            out.write("item[" + (i + 1) + "]:");
            out.write(NEW_LINE);
            out.write(indent2);
            out.write("class = \"IntervalTier\"");
            out.write(NEW_LINE);
            out.write(indent2);
            out.write("name = \"" + tier.getName() + "\"");
            out.write(NEW_LINE);
            out.write(indent2);
            out.write(xmin + (bt / 1000d));
            out.write(NEW_LINE);
            out.write(indent2);
            out.write(xmax + (et / 1000d));
            out.write(NEW_LINE);

            // fill the list of annotation objects, while replacing gaps with new empty annotations.
            // this is necessary in order to be able to calculate and export the number of intervals
            annotations.clear();

            Vector anns = tier.getAnnotations();

            for (int j = 0; j < anns.size(); j++) {
                ann2 = (Annotation) anns.get(j);

                if (ann2.getEndTimeBoundary() <= bt) {
                    ann1 = ann2;

                    continue;
                }

                // check begintime
                if (ann2.getBeginTimeBoundary() <= bt) {
                    annotations.add(new AnnotationDataRecord(null,
                            ann2.getValue(),
                            Math.max(bt, ann2.getBeginTimeBoundary()),
                            Math.min(et, ann2.getEndTimeBoundary())));

                    if (ann2.getEndTimeBoundary() >= et) {
                        break;
                    }
                } else { //ann2 begin > bt

                    if (ann1 != null) {
                        // fill gap if any
                        if (ann1.getEndTimeBoundary() < ann2.getBeginTimeBoundary()) {
                            annotations.add(new AnnotationDataRecord(null, "",
                                    Math.max(bt, ann1.getEndTimeBoundary()),
                                    Math.min(et, ann2.getBeginTimeBoundary())));
                        }
                    } else { // first annotation begins > bt, fill gap
                        annotations.add(new AnnotationDataRecord(null, "", bt,
                                Math.min(et, ann2.getBeginTimeBoundary())));
                    }

                    if (ann2.getBeginTimeBoundary() >= et) {
                        break;
                    } else {
                        // add ann 2
                        annotations.add(new AnnotationDataRecord(null,
                                ann2.getValue(), ann2.getBeginTimeBoundary(),
                                Math.min(et, ann2.getEndTimeBoundary())));

                        if (ann2.getEndTimeBoundary() >= et) {
                            break;
                        }
                    }
                }

                ann1 = ann2;
            }

            // the intervals list is filled, the size is known now
            out.write(indent2);
            out.write("intervals: size = " + annotations.size());
            out.write(NEW_LINE);

            AnnotationDataRecord record = null;

            for (int j = 0; j < annotations.size(); j++) {
                record = (AnnotationDataRecord) annotations.get(j);
                out.write(indent2);
                out.write(intervals + (j + 1) + "]");
                out.write(NEW_LINE);
                out.write(indent3);
                out.write(xmin);
                out.write(String.valueOf(record.getBeginTime() / 1000d));
                out.write(NEW_LINE);
                out.write(indent3);
                out.write(xmax);
                out.write(String.valueOf(record.getEndTime() / 1000d));
                out.write(NEW_LINE);
                out.write(indent3);
                out.write(tx);
                out.write("\"" + record.getValue() + "\"");
                out.write(NEW_LINE);
            }
        }
    }
}
