/*
 * File:     ToolboxEncoderInfo.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.server.corpora.clomimpl.shoebox;

import mpi.eudico.server.corpora.clom.EncoderInfo;

import mpi.eudico.server.corpora.clomimpl.shoebox.interlinear.Interlinearizer;

import java.util.List;


/**
 * Toolbox specific EncoderInfo implementation may 06 (HS): added boolean for
 * two export options:<br>
 * - whether or not to recalculate begin and end times of annotations based on
 * the master media offset<br>
 * dec 2006 (HS): added a boolean to indicate that all tiers should be
 * exported using UTF-8 encoding
 *
 * @author hennie
 */
public class ToolboxEncoderInfo implements EncoderInfo {
    /** export with .typ file */
    public static final int TYPFILE = 0;

    /** export with marker file */
    public static final int DEFINED_MARKERS = 1;

    /** export using tier names */
    public static final int TIERNAMES = 2;
    private int pageWidth;
    private int markerSource;
    private int timeFormat;

    // may 2006: add db type to the encoder object instead of a static field in ShoeboxTypFile
    private String databaseType = "";

    // may 2006: add markers to the encoder object instead of a static field in ShoeboxTypFile
    private List markers;
    private boolean correctAnnotationTimes = true;

    // flag to indicate that all markers are in Unicode (UTF-8)
    private boolean allUnicode = false;

    // oct 2007 new fields for extra options for (utf-8 only) Toolbox export
    private boolean wrapLines = false;
    private int lineWrapStyle;
    private boolean includeEmptyMarkers = true;
    private String recordMarker;
    private List orderedVisibleTiers;
    private List markersWithBlankLines;
    private long timeOffset = 0L;

    /**
     * Creates a new ToolboxEncoderInfo instance
     *
     * @param pageWidth the pagewidth (in number of characters)
     * @param markerSource the type of marker source (.typ file, marker files
     *        etc)
     */
    public ToolboxEncoderInfo(int pageWidth, int markerSource) {
        this(pageWidth, markerSource, Interlinearizer.SSMS);
    }

    /**
     * Creates a new ToolboxEncoderInfo instance
     *
     * @param pageWidth the pagewidth (in number of characters)
     * @param markerSource the type of marker source (.typ file, marker files
     *        etc)
     * @param timeFormat the format of time values
     */
    public ToolboxEncoderInfo(int pageWidth, int markerSource, int timeFormat) {
        this.pageWidth = pageWidth;
        this.markerSource = markerSource;
        this.timeFormat = timeFormat;
    }

    /**
     * Returns the page width (number of characters)
     *
     * @return the page width
     */
    public int getPageWidth() {
        return pageWidth;
    }

    /**
     * Returns the marker source type.
     *
     * @return the marker source type
     */
    public int getMarkerSource() {
        return markerSource;
    }

    /**
     * Returns the time format value.
     *
     * @return the time format
     */
    public int getTimeFormat() {
        return timeFormat;
    }

    /**
     * Returns the flag whether annotation time values should be recalculated.
     *
     * @return the correct time values flag
     */
    public boolean getCorrectAnnotationTimes() {
        return correctAnnotationTimes;
    }

    /**
     * Sets whether the time values should be recalculated.
     *
     * @param correctAnnotationTimes if true the master media offset is added
     *        to all time values
     */
    public void setCorrectAnnotationTimes(boolean correctAnnotationTimes) {
        this.correctAnnotationTimes = correctAnnotationTimes;
    }

    /**
     * Returns the database type for the Toolbox header
     *
     * @return Returns the databaseType.
     */
    public String getDatabaseType() {
        return databaseType;
    }

    /**
     * Sets the database type for the Toolbox header
     *
     * @param databaseType The databaseType to set.
     */
    public void setDatabaseType(String databaseType) {
        this.databaseType = databaseType;
    }

    /**
     * Returns the list of marker objects
     *
     * @return Returns the markers.
     */
    public List getMarkers() {
        return markers;
    }

    /**
     * Sets the list of marker objects.
     *
     * @param markers The markers to set.
     */
    public void setMarkers(List markers) {
        this.markers = markers;
    }

    /**
     * Returns whether all annotations are to be exported in Unicode
     *
     * @return whether all annotations are to be exported in Unicode
     */
    public boolean isAllUnicode() {
        return allUnicode;
    }

    /**
     * Sets whether all annotations are to be exported in Unicode.
     *
     * @param allUnicode if true all tiers are exported with UTF-8 encoding,
     *        otherwise they  will be encoded in ISO-Latin or a mixture of
     *        encodings
     */
    public void setAllUnicode(boolean allUnicode) {
        this.allUnicode = allUnicode;
    }

    /**
     * Returns whether or not empty markers should be part of the output or be
     * skipped
     *
     * @return the includeEmptyMarkers flag
     */
    public boolean isIncludeEmptyMarkers() {
        return includeEmptyMarkers;
    }

    /**
     * Sets whether empty markers should be included in the output.
     *
     * @param includeEmptyMarkers the includeEmptyMarkers to set
     */
    public void setIncludeEmptyMarkers(boolean includeEmptyMarkers) {
        this.includeEmptyMarkers = includeEmptyMarkers;
    }

    /**
     * Returns the line wrap style (next line or end of block)
     *
     * @return the lineWrapStyle
     */
    public int getLineWrapStyle() {
        return lineWrapStyle;
    }

    /**
     * Sets the line wrap style (next line or end of block)
     *
     * @param lineWrapStyle the lineWrapStyle to set
     */
    public void setLineWrapStyle(int lineWrapStyle) {
        this.lineWrapStyle = lineWrapStyle;
    }

    /**
     * Returns a list of markers that should be followed by a blank line
     *
     * @return the list of markers followed by a blank line
     */
    public List getMarkersWithBlankLines() {
        return markersWithBlankLines;
    }

    /**
     * Sets the list of markers that should be followed by a blank line
     *
     * @param markersWithBlankLines the markersWithBlankLines to set
     */
    public void setMarkersWithBlankLines(List markersWithBlankLines) {
        this.markersWithBlankLines = markersWithBlankLines;
    }

    /**
     * Returns the record marker.
     *
     * @return the recordMarker
     */
    public String getRecordMarker() {
        return recordMarker;
    }

    /**
     * Sets the record marker
     *
     * @param recordMarker the recordMarker to set
     */
    public void setRecordMarker(String recordMarker) {
        this.recordMarker = recordMarker;
    }

    /**
     * Returns whether or not line wrapping should be applied
     *
     * @return the wrapLines flag
     */
    public boolean isWrapLines() {
        return wrapLines;
    }

    /**
     * Sets the linewrapping flag.
     *
     * @param wrapLines the wrapLines to set
     */
    public void setWrapLines(boolean wrapLines) {
        this.wrapLines = wrapLines;
    }

    /**
     * Returns the (master media) time offset; the number of ms to add to all
     * time values.
     *
     * @return the media time offset
     */
    public long getTimeOffset() {
        return timeOffset;
    }

    /**
     * Sets the time offset, the number of ms to add to the time values of
     * annotations.
     *
     * @param timeOffset the time offset
     */
    public void setTimeOffset(long timeOffset) {
        this.timeOffset = timeOffset;
    }

    /**
     * Returns the visible tiers, in the right order
     *
     * @return the visible tiers (names)
     */
    public List getOrderedVisibleTiers() {
        return orderedVisibleTiers;
    }

    /**
     * Sets the visible tiers, in the right order
     *
     * @param orderedVisibleTiers the ordered visible tiers (names)
     */
    public void setOrderedVisibleTiers(List orderedVisibleTiers) {
        this.orderedVisibleTiers = orderedVisibleTiers;
    }
}
