/*
 * File:     AbstractConstraintPanel.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package mpi.search.content.query.viewer;

import mpi.search.SearchLocale;

import mpi.search.content.model.CorpusType;

import mpi.search.content.query.model.Constraint;
import mpi.search.content.query.model.DependentConstraint;
import mpi.search.content.query.model.RestrictedAnchorConstraint;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;

import java.util.HashMap;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.KeyStroke;
import javax.swing.border.Border;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.LineBorder;
import javax.swing.tree.DefaultTreeModel;


/**
 * The ConstraintPanel is the GUI for editing one single Constraint
 * ConstraintPanels are meant to be included in the SearchConfigPanel
 *
 * $Id: AbstractConstraintPanel.java 11147 2007-12-05 21:24:22Z klasal $
 * $Author$
 */
public abstract class AbstractConstraintPanel extends JPanel
    implements ItemListener {
    /** Holds value of property DOCUMENT ME! */
    protected static int tierComboBoxWidth = 0;

    /** Holds value of property DOCUMENT ME! */
    protected final Action startAction;

    /** Holds value of property DOCUMENT ME! */
    protected final CardLayout framedPanelLayout = new CardLayout();

    /** Holds value of property DOCUMENT ME! */
    protected final Constraint constraint;

    /** Holds value of property DOCUMENT ME! */
    protected final CorpusType type;

    /** Holds value of property DOCUMENT ME! */
    protected final DefaultTreeModel treeModel;

    /** Holds value of property DOCUMENT ME! */
    protected final JCheckBox caseCheckBox = new JCheckBox(SearchLocale.getString(
                "Search.Constraint.CaseSensitive"), false);

    /** Holds value of property DOCUMENT ME! */
    protected final JCheckBox regExCheckBox = new JCheckBox(SearchLocale.getString(
                "Search.Constraint.RegularExpression"), false);

    /** Holds value of property DOCUMENT ME! */
    protected AttributeConstraintPanel attributePanel;

    /** Holds value of property DOCUMENT ME! */
    protected final JPanel framedPanel = new JPanel(framedPanelLayout);

    /** Holds value of property DOCUMENT ME! */
    protected final JPanel optionPanel = new JPanel(new BorderLayout());

    /** Holds value of property DOCUMENT ME! */
    protected JComboBox tierComboBox;

    /** Holds value of property DOCUMENT ME! */
    protected JPanel titleComponent = new JPanel(new FlowLayout(
                FlowLayout.LEFT, 5, 1));

    /** Holds value of property DOCUMENT ME! */
    protected PatternPanel patternPanel;

    /** Holds value of property DOCUMENT ME! */
    protected RelationPanel relationPanel;

    /** Holds value of property DOCUMENT ME! */
    private final Border blueBorder = new CompoundBorder(new EmptyBorder(1, 0,
                1, 0),
            new CompoundBorder(new LineBorder(Color.BLUE),
                new EmptyBorder(0, 3, 0, 0)));

    /**
     * Creates a new AbstractConstraintPanel object.
     *
     * @param constraint DOCUMENT ME!
     * @param treeModel DOCUMENT ME!
     * @param type DOCUMENT ME!
     * @param startAction DOCUMENT ME!
     */
    public AbstractConstraintPanel(Constraint constraint,
        DefaultTreeModel treeModel, CorpusType type, Action startAction) {
        this.type = type;
        this.constraint = constraint;
        this.treeModel = treeModel;
        this.startAction = startAction;
    }

    /**
     * DOCUMENT ME!
     *
     * @param c DOCUMENT ME!
     */
    public void setConstraint(Constraint c) {
        setRegEx(c.isRegEx());
        setCaseSensitive(c.isCaseSensitive());
        setPattern(c.getPattern());
        relationPanel.setLowerBoundary(c.getLowerBoundary());
        relationPanel.setUpperBoundary(c.getUpperBoundary());
        relationPanel.setUnit(c.getUnit());
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public boolean isRegEx() {
        return regExCheckBox.isSelected();
    }

    /**
     *
     *
     * @return DOCUMENT ME!
     */
    public String getTierName() {
        return getTierNames()[0];
    }

    //implementation of ItemListener
    public void itemStateChanged(ItemEvent e) {
        if (e.getStateChange() == ItemEvent.SELECTED) {
            if (e.getSource() == tierComboBox) {
                if (type.isClosedVoc((String) e.getItem())) {
                    regExCheckBox.setEnabled(false);
                    setRegEx(false);
                    setCaseSensitive(false);

                    if (type.getClosedVoc((String) e.getItem()).isEmpty()) {
                        startAction.setEnabled(false);
                    } else {
                        startAction.setEnabled(true);
                    }
                } else {
                    startAction.setEnabled(true);
                    regExCheckBox.setEnabled(true);
                }

                if (type.strictCaseSensitive((String) e.getItem())) {
                    setCaseSensitive(true);
                }

                boolean caseSensitiveFixed = type.isClosedVoc((String) e.getItem()) ||
                    type.strictCaseSensitive((String) e.getItem());

                if (!caseSensitiveFixed) {
                    if (!caseCheckBox.isEnabled()) {
                        setCaseSensitive(false);
                    }

                    // change from fixed to variable -> set default
                }

                caseCheckBox.setEnabled(!caseSensitiveFixed);

                if (attributePanel != null) {
                    attributePanel.setTier((String) e.getItem());
                }
            }
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public String toString() {
        String s = super.toString();
        System.out.println("string " + s);

        return s;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    protected HashMap getAttributes() {
        HashMap attributes = new HashMap();

        if (attributePanel != null) {
            String[] attributeNames = type.getAttributeNames(getTierName());

            for (int i = 0; i < attributeNames.length; i++) {
                attributes.put(attributeNames[i],
                    attributePanel.getAttributeValue(attributeNames[i]));
            }
        }

        return attributes;
    }

    /**
     * DOCUMENT ME!
     *
     * @param sensitiv DOCUMENT ME!
     */
    protected void setCaseSensitive(boolean sensitiv) {
        caseCheckBox.setSelected(sensitiv);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    protected boolean isCaseSensitive() {
        return caseCheckBox.isSelected();
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    protected Constraint getConstraint() {
        updateNode();

        return constraint;
    }

    /**
     * DOCUMENT ME!
     *
     * @param pattern DOCUMENT ME!
     */
    protected void setPattern(String pattern) {
        patternPanel.setPattern(pattern);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    protected String getPattern() {
        return patternPanel.getPattern();
    }

    /**
     * DOCUMENT ME!
     *
     * @param regEx DOCUMENT ME!
     */
    protected void setRegEx(boolean regEx) {
        regExCheckBox.setSelected(regEx);
    }

    /**
     * DOCUMENT ME!
     *
     * @param tierName DOCUMENT ME!
     */
    protected void setTierName(String tierName) {
        tierComboBox.setSelectedItem(tierName);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    protected String[] getTierNames() {
        String[] tierNames;
        Object[] selectedObjects = tierComboBox.getSelectedObjects();

        if ((selectedObjects == null) || (selectedObjects.length == 0)) {
            return new String[] { Constraint.ALL_TIERS };
        }

        tierNames = new String[selectedObjects.length];

        for (int i = 0; i < selectedObjects.length; i++) {
            tierNames[i] = (String) selectedObjects[i];
        }

        return tierNames;
    }

    /**
     * DOCUMENT ME!
     */
    protected void makeLayout() {
        //RegExPanel
        patternPanel = new PatternPanel(type, tierComboBox, regExCheckBox,
                constraint, startAction);

        //RelationPanel
        relationPanel = new RelationPanel(type, constraint);

        //OptionPanel
        JPanel checkBoxPanel = new JPanel(new GridLayout(2, 1));
        regExCheckBox.setFont(getFont().deriveFont(Font.PLAIN, 9f));
        caseCheckBox.setFont(getFont().deriveFont(Font.PLAIN, 9f));
        checkBoxPanel.add(regExCheckBox);
        checkBoxPanel.add(caseCheckBox);
        optionPanel.add(checkBoxPanel, BorderLayout.WEST);

        //InputPanel
        JPanel inputPanel = new JPanel(new GridLayout(2, 1, 0, 1));
        inputPanel.add(patternPanel);
        inputPanel.add(relationPanel);

        //AttributePanel
        if (type.hasAttributes()) {
            attributePanel = new AttributeConstraintPanel(type);
            optionPanel.add(attributePanel, BorderLayout.CENTER);
            attributePanel.setTier(getTierName());
        }

        //FramedPanel
        JPanel specificationPanel = new JPanel(new FlowLayout(FlowLayout.LEFT,
                    0, 1));
        specificationPanel.add(inputPanel);
        specificationPanel.add(optionPanel);
        framedPanel.add(specificationPanel, "");
        framedPanel.setBorder(blueBorder);
        framedPanelLayout.show(framedPanel, "");

        //this
        setLayout(new BorderLayout());
        add(titleComponent, BorderLayout.NORTH);
        add(framedPanel, BorderLayout.CENTER);

        tierComboBox.addItemListener(this);

        JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 5, 1));
        Action addConstraintAction = new AbstractAction(SearchLocale.getString(
                    "Search.Query.Add")) {
                public void actionPerformed(ActionEvent e) {
                    addConstraint();
                }
            };

        KeyStroke ks = KeyStroke.getKeyStroke(KeyEvent.VK_A,
                ActionEvent.CTRL_MASK);
        addConstraintAction.putValue(Action.ACCELERATOR_KEY, ks);

        JButton addButton = new JButton(addConstraintAction);
        addButton.setFont(getFont().deriveFont(11f));
        buttonPanel.add(addButton);

        if ((constraint.getParent() != null) &&
                !(constraint.getParent() instanceof RestrictedAnchorConstraint)) {
            Action deleteConstraintAction = new AbstractAction(SearchLocale.getString(
                        "Search.Query.Delete")) {
                    public void actionPerformed(ActionEvent e) {
                        deleteConstraint();
                    }
                };

            ks = KeyStroke.getKeyStroke(KeyEvent.VK_D, ActionEvent.CTRL_MASK);
            deleteConstraintAction.putValue(Action.ACCELERATOR_KEY, ks);

            JButton deleteButton = new JButton(deleteConstraintAction);
            deleteButton.setFont(getFont().deriveFont(11f));
            buttonPanel.add(deleteButton);
        }

        add(buttonPanel, BorderLayout.SOUTH);

        try {
            Class popupMenu = type.getInputMethodClass();
            popupMenu.getConstructor(new Class[] {
                    Component.class, AbstractConstraintPanel.class
                }).newInstance(new Object[] {
                    patternPanel.getDefaultInputComponent(),
                    AbstractConstraintPanel.this
                });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param comboBox DOCUMENT ME!
     * @param units DOCUMENT ME!
     */
    protected static void updateComboBox(JComboBox comboBox, String[] units) {
        //don't refill if equal strings
        if (units.length == comboBox.getItemCount()) {
            boolean equal = true;

            for (int i = 0; i < units.length; i++) {
                if (!units[i].equals(comboBox.getItemAt(i))) {
                    equal = false;
                }
            }

            if (equal) {
                return;
            }
        }

        Object oldItem = comboBox.getSelectedItem();

        comboBox.removeAllItems();

        for (int i = 0; i < units.length; i++) {
            comboBox.addItem(units[i]);
        }

        //select old item
        comboBox.setSelectedItem(oldItem);

        //if oldItem not in Box, select smallest possible linguistic unit
        if (comboBox.getSelectedItem() == null) {
            comboBox.setSelectedIndex(0);
        }
    }

    /**
     * DOCUMENT ME!
     */
    protected void updateNode() {
        try {
            constraint.setTierNames(getTierNames());
            constraint.setPattern(getPattern());
            constraint.setLowerBoundary(relationPanel.getLowerBoundary());
            constraint.setUpperBoundary(relationPanel.getUpperBoundary());
            constraint.setUnit(relationPanel.getUnit());
            constraint.setRegEx(isRegEx());
            constraint.setCaseSensitive(isCaseSensitive());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void addConstraint() {
        treeModel.insertNodeInto(new DependentConstraint(getTierNames()),
            constraint, constraint.getChildCount());
    }

    private void deleteConstraint() {
        treeModel.removeNodeFromParent(constraint);
    }
}
