/*
 * File:     LogUtil.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * Created on Mar 30, 2004
 * $Id: LogUtil.java 8222 2007-02-22 10:20:51Z klasal $
 */
package mpi.util;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;

import java.util.logging.LogManager;


/**
 * Convenience methods for logging configuration and formatting.
 *
 * @author Wouter Huijnink
 */
public final class LogUtil {
    /**
     * Creates a new LogUtil instance. Only present to allow of configuring the
     * logging mechanism:
     *
     * <p>
     * If the "java.util.logging.config.class" property is set, then the
     * property value is treated as a class name. The given class will be
     * loaded, an object will be instantiated, and that object's constructor
     * is responsible for reading in the initial configuration. (That object
     * may use other system properties to control its configuration.) The
     * alternate configuration class can use readConfiguration(InputStream) to
     * define properties in the LogManager. (<code>LogManager</code>
     * javadocs).
     * </p>
     *
     * <p>
     * Usage: <code>java
     * -Djava.util.logging.config.class=mpi.library.util.LogUtil
     * -Dlogging.config.file=LOGGING_CONFIG_FILE</code>
     * </p>
     */
    public LogUtil() {
        try {
            System.out.println("Initializing logging configuration...");

            String configFile = System.getProperty("logging.config.file");

            if (configFile != null) {
                LogManager.getLogManager().readConfiguration(ClassLoader.getSystemResourceAsStream(
                        configFile));
            } else {
                System.err.println(
                    "Logging could not be configured (system property logging.config.file not found) - logging will not be available.");
            }
        } catch (SecurityException e1) {
            System.err.println(
                "Logging could not be configured - logging will not be available.");
            e1.printStackTrace(System.err);
        } catch (IOException e1) {
            System.err.println(
                "Logging could not be configured - logging will not be available.");
            e1.printStackTrace(System.err);
        }
    }

    /**
     * Returns a String representation of a Throwable's stack trace.
     *
     * @param t The Throwable.
     *
     * @return A String representation of the Throwable's stack trace.
     */
    public static String formatStackTrace(Throwable t) {
        StringWriter sw = new StringWriter();
        t.printStackTrace(new PrintWriter(sw));

        return sw.toString();
    }
}
