/*
 * File:     URIUtil.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.util;

import java.io.File;
import java.io.IOException;

import java.net.URI;


/**
 * DOCUMENT ME!
 * $Id: jalopy_gnu_src_dist.xml 11222 2007-12-11 15:57:47Z hasloe $
 * @author $Author$
 * @version $Revision$
 */
public class URIUtil {
    /** Holds value of property DOCUMENT ME! */
    final private static String[] hex = {
        "%00", "%01", "%02", "%03", "%04", "%05", "%06", "%07", "%08", "%09",
        "%0a", "%0b", "%0c", "%0d", "%0e", "%0f", "%10", "%11", "%12", "%13",
        "%14", "%15", "%16", "%17", "%18", "%19", "%1a", "%1b", "%1c", "%1d",
        "%1e", "%1f", "%20", "%21", "%22", "%23", "%24", "%25", "%26", "%27",
        "%28", "%29", "%2a", "%2b", "%2c", "%2d", "%2e", "%2f", "%30", "%31",
        "%32", "%33", "%34", "%35", "%36", "%37", "%38", "%39", "%3a", "%3b",
        "%3c", "%3d", "%3e", "%3f", "%40", "%41", "%42", "%43", "%44", "%45",
        "%46", "%47", "%48", "%49", "%4a", "%4b", "%4c", "%4d", "%4e", "%4f",
        "%50", "%51", "%52", "%53", "%54", "%55", "%56", "%57", "%58", "%59",
        "%5a", "%5b", "%5c", "%5d", "%5e", "%5f", "%60", "%61", "%62", "%63",
        "%64", "%65", "%66", "%67", "%68", "%69", "%6a", "%6b", "%6c", "%6d",
        "%6e", "%6f", "%70", "%71", "%72", "%73", "%74", "%75", "%76", "%77",
        "%78", "%79", "%7a", "%7b", "%7c", "%7d", "%7e", "%7f", "%80", "%81",
        "%82", "%83", "%84", "%85", "%86", "%87", "%88", "%89", "%8a", "%8b",
        "%8c", "%8d", "%8e", "%8f", "%90", "%91", "%92", "%93", "%94", "%95",
        "%96", "%97", "%98", "%99", "%9a", "%9b", "%9c", "%9d", "%9e", "%9f",
        "%a0", "%a1", "%a2", "%a3", "%a4", "%a5", "%a6", "%a7", "%a8", "%a9",
        "%aa", "%ab", "%ac", "%ad", "%ae", "%af", "%b0", "%b1", "%b2", "%b3",
        "%b4", "%b5", "%b6", "%b7", "%b8", "%b9", "%ba", "%bb", "%bc", "%bd",
        "%be", "%bf", "%c0", "%c1", "%c2", "%c3", "%c4", "%c5", "%c6", "%c7",
        "%c8", "%c9", "%ca", "%cb", "%cc", "%cd", "%ce", "%cf", "%d0", "%d1",
        "%d2", "%d3", "%d4", "%d5", "%d6", "%d7", "%d8", "%d9", "%da", "%db",
        "%dc", "%dd", "%de", "%df", "%e0", "%e1", "%e2", "%e3", "%e4", "%e5",
        "%e6", "%e7", "%e8", "%e9", "%ea", "%eb", "%ec", "%ed", "%ee", "%ef",
        "%f0", "%f1", "%f2", "%f3", "%f4", "%f5", "%f6", "%f7", "%f8", "%f9",
        "%fa", "%fb", "%fc", "%fd", "%fe", "%ff"
    };

    /**
     * DOCUMENT ME!
     *
     * @param root DOCUMENT ME!
     * @param uri DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public static URI resolveURI(URI root, URI uri) {
        if (uri.isAbsolute()) {
            return uri;
        }

        return root.resolve(uri);
    }

    /**
     * DOCUMENT ME!
     *
     * @param desc DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     *
     * @throws Exception DOCUMENT ME!
     */
    public static URI newURI(String desc) throws Exception {
        URI uri;
        uri = new URI(encode(desc));

        return uri.normalize();
    }

    /**
     * DOCUMENT ME!
     *
     * @param uri DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public static String getURIBaseName(URI uri) {
        String path = uri.getPath();
        int pos = path.lastIndexOf('/');

        return path.substring(pos + 1);
    }

    /**
     * DOCUMENT ME!
     *
     * @param uri DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public static URI getURIParent(URI uri) {
        String path = uri.getPath();
        int pos = path.lastIndexOf('/', path.length() - 2);

        if (pos == -1) {
            pos = 0;
        }

        String parentPath = decode(path.substring(0, pos + 1));
        URI newURI;

        try {
            newURI = new URI(uri.getScheme(), uri.getAuthority(), parentPath,
                    null, null);
        } catch (Throwable t) {
            throw new Error("Failed to get the parent URI of " +
                uri.toString() + "\n\n:" + t);
        }

        return newURI;
    }

    /**
     * DOCUMENT ME!
     *
     * @param uriPath DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public static String uriPathToFilePath(String uriPath) {
        return decode(uriPath).replace('/', File.separatorChar);
    }

    /**
     * DOCUMENT ME!
     *
     * @param a DOCUMENT ME!
     * @param b DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public static boolean fromSameServer(URI a, URI b) {
        String aAuth = a.getAuthority();
        String bAuth = b.getAuthority();

        if ((aAuth != null) && (bAuth != null)) {
            return (aAuth.equals(bAuth));
        }

        return ((aAuth == null) && (bAuth == null));
    }

    /**
     * DOCUMENT ME!
     *
     * @param uri DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public static String getURIExtension(URI uri) {
        String path = uri.getPath();
        int pos = path.lastIndexOf('.');

        if (pos == -1) {
            return "";
        }

        return path.substring(pos + 1);
    }

    // removed: public static RFile uriToRFile (URI uri) 
    // removed: public static int getContents (URI uri, StringBuffer buf, String enc) throws IOException

    /** if base is absolute and file is not, file will be treated
     *  as relative to base, not to as relative to the current dir,
     *  as the caller is assumed to use the base as reference. If
     *  both base and file are relative, they are treated as both
     *  being relative to the current directory.
     */
    public static String calculateRelativeURI(File base, File file) {
        // 1. Find common parent, and count how many steps up it is
        File commonParent = null;

        try { // resolve all "./" and "../" parts, make path absolute, etc
            commonParent = base.getCanonicalFile().getParentFile();

            // if base already is a directory: use the base itself, NOT its parent
            // this case was not supported in older versions
            if (base.isDirectory()) {
                commonParent = base.getCanonicalFile();
            }
        } catch (IOException ioe) { // maybe symlinks can cause this exception?
            commonParent = base.getAbsoluteFile().getParentFile();

            if (base.isDirectory()) {
                commonParent = base.getAbsoluteFile();
            }
        }

        if (!file.isAbsolute() && base.isAbsolute()) { // then make it absolute relative to the base!
            file = new File(commonParent, file.getPath());
        }

        File absFile = null;

        try {
            absFile = file.getCanonicalFile();
        } catch (IOException ioe) {
            absFile = file.getAbsoluteFile();
        }

        int stepsUp = 0;

        while (commonParent != null) { // was: while true

            File nextParent = commonParent.getParentFile();

            if (nextParent == null) { // older versions failed for top directory

                break; // reached top level: special handling as this includes File.separator already
            }

            if (absFile.toString().startsWith(commonParent.toString() +
                        File.separator)) {
                break; // found common parent directory
            }

            commonParent = nextParent;
            stepsUp++;
        }

        // 2. Get the folders we need to enter to get to @file
        File fileParent = absFile.getParentFile();
        String append = "";

        while (fileParent != null) { // was: while true

            if (fileParent.equals(commonParent)) {
                break;
            }

            append = fileParent.getName() + "/" + append;
            fileParent = fileParent.getParentFile();

            if (fileParent == null) {
                System.out.println(
                    "URIUtil.calculateRelativeURI ran out of file parents?");
            }
        }

        // 3. Combine it all
        String ret = "";
        int i;

        for (i = 0; i < stepsUp; i++)
            ret += "../";

        ret += append;
        ret += absFile.getName();

        return ret;
    }

    // This is not 100% correct, but should do in most of the cases.
    // Having invalid URIs in the XML is broken anyway
    private static String encode(String s) {
        StringBuffer sbuf = new StringBuffer();
        int len = s.length();

        // hadAuthority is only looked at in case '://'
        // does not occur in the string, so this is
        // in case of relative and file: URIs.
        // See also comments later on for clarification.
        boolean hadAuthority = (s.indexOf("file:") < 0);
        boolean countSlashes = (s.indexOf("://") > 0);
        int slashDepth = 0;

        for (int i = 0; i < len; i++) {
            int ch = s.charAt(i);

            if (('A' <= ch) && (ch <= 'Z')) { // 'A'..'Z'
                sbuf.append((char) ch);
            } else if (('a' <= ch) && (ch <= 'z')) { // 'a'..'z'
                sbuf.append((char) ch);
            } else if (('0' <= ch) && (ch <= '9')) { // '0'..'9'
                sbuf.append((char) ch);
            } else if ((ch == '-') || (ch == '_') // unreserved
                     ||(ch == '.') || (ch == '!') || (ch == '~') ||
                    (ch == '*') || (ch == '\'') || (ch == '(') || (ch == ')') ||
                    (ch == '/') ||
                    ((countSlashes ? (slashDepth < 3) : (!hadAuthority)) &&
                    (ch == ':')) ||
                    (countSlashes && (slashDepth < 3) && (ch == '@'))) {
                sbuf.append((char) ch);

                if (ch == '/') {
                    slashDepth++;
                } else if (ch == ':') {
                    hadAuthority = true;
                }
            } else if (ch <= 0x007f) { // other ASCII
                sbuf.append(hex[ch]);
            } else if (ch <= 0x07FF) { // non-ASCII <= 0x7FF
                sbuf.append(hex[0xc0 | (ch >> 6)]);
                sbuf.append(hex[0x80 | (ch & 0x3F)]);
            } else { // 0x7FF < ch <= 0xFFFF
                sbuf.append(hex[0xe0 | (ch >> 12)]);
                sbuf.append(hex[0x80 | ((ch >> 6) & 0x3F)]);
                sbuf.append(hex[0x80 | (ch & 0x3F)]);
            }
        }

        return sbuf.toString();
    }

    private static String decode(String s) {
        StringBuffer sbuf = new StringBuffer();
        int l = s.length();
        int ch = -1;
        int b;
        int sumb = 0;

        for (int i = 0, more = -1; i < l; i++) {
            /* Get next byte b from URL segment s */
            switch (ch = s.charAt(i)) {
            case '%':
                ch = s.charAt(++i);

                int hb = (Character.isDigit((char) ch) ? (ch - '0')
                                                       : ((10 +
                    Character.toLowerCase((char) ch)) - 'a')) & 0xF;
                ch = s.charAt(++i);

                int lb = (Character.isDigit((char) ch) ? (ch - '0')
                                                       : ((10 +
                    Character.toLowerCase((char) ch)) - 'a')) & 0xF;
                b = (hb << 4) | lb;

                break;

            default:
                b = ch;
            }

            /* Decode byte b as UTF-8, sumb collects incomplete chars */
            if ((b & 0xc0) == 0x80) { // 10xxxxxx (continuation byte)
                sumb = (sumb << 6) | (b & 0x3f); // Add 6 bits to sumb

                if (--more == 0) {
                    sbuf.append((char) sumb); // Add char to sbuf
                }
            } else if ((b & 0x80) == 0x00) { // 0xxxxxxx (yields 7 bits)
                sbuf.append((char) b); // Store in sbuf
            } else if ((b & 0xe0) == 0xc0) { // 110xxxxx (yields 5 bits)
                sumb = b & 0x1f;
                more = 1; // Expect 1 more byte
            } else if ((b & 0xf0) == 0xe0) { // 1110xxxx (yields 4 bits)
                sumb = b & 0x0f;
                more = 2; // Expect 2 more bytes
            } else if ((b & 0xf8) == 0xf0) { // 11110xxx (yields 3 bits)
                sumb = b & 0x07;
                more = 3; // Expect 3 more bytes
            } else if ((b & 0xfc) == 0xf8) { // 111110xx (yields 2 bits)
                sumb = b & 0x03;
                more = 4; // Expect 4 more bytes
            } else /*if ((b & 0xfe) == 0xfc)*/
             { // 1111110x (yields 1 bit)
                sumb = b & 0x01;
                more = 5; // Expect 5 more bytes
            }

            /* We don't test if the UTF-8 encoding is well-formed */
        }

        return sbuf.toString();
    }
}
