/*
 * File:     RestrictedComboBoxEditor.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.util.gui;

import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.plaf.basic.BasicComboBoxEditor;


/**
 * This class represents an Editor for a JCombobox It displays only the
 * back-part of the Items (e.g. string after the splitChar) All keyEvents are
 * blocked except cursor keys and the delete key. Pressing the delete key
 * replaces the part of the string from cursor position to the end of the
 * string with the wildkey + end character
 */
public class RestrictedComboBoxEditor extends BasicComboBoxEditor
    implements KeyListener {
    /** Holds value of property DOCUMENT ME! */
    final private char splitChar;

    /** Holds value of property DOCUMENT ME! */
    final private char sepChar;

    /** Holds value of property DOCUMENT ME! */
    final private char endChar;

    /** Holds value of property DOCUMENT ME! */
    final private char wildkeyChar;
    private String oldString;
    private String prefix;
    private Object oldValue;

    /**
     * Creates a new RestrictedComboBoxEditor instance
     *
     * @param splitChar DOCUMENT ME!
     * @param sepChar DOCUMENT ME!
     * @param endChar DOCUMENT ME!
     * @param wildkeyChar DOCUMENT ME!
     */
    public RestrictedComboBoxEditor(char splitChar, char sepChar, char endChar,
        char wildkeyChar) {
        this.splitChar = splitChar;
        this.sepChar = sepChar;
        this.endChar = endChar;
        this.wildkeyChar = wildkeyChar;
        editor.addKeyListener(this);
    }

    /**
     * Creates a new RestrictedComboBoxEditor instance
     */
    public RestrictedComboBoxEditor() {
        this('(', ',', ')', '*');
    }

    /**
     * Sets the item that should be edited. Only the back part of the string is
     * shown.
     *
     * @param anObject the displayed value of the editor
     */
    public void setItem(Object anObject) {
        if (anObject != null) {
            String string = anObject.toString();
            int splitindex = string.indexOf(splitChar);

            if (splitindex >= 0) {
                prefix = string.substring(0, splitindex);
                string = string.substring(splitindex);
            }

            editor.setText(string);

            oldValue = anObject;
            oldString = string;
            editor.setBackground(java.awt.Color.white);
        } else {
            editor.setText("");
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public Object getItem() {
        Object newValue = editor.getText();

        if (oldValue != null) {
            // The original value is not a string. Should return the value in it's
            // original type.
            if (newValue.equals(oldString)) {
                return oldValue;
            } else {
                return prefix + newValue;
            }
        }

        return newValue;
    }

    // implementation of java.awt.event.KeyListener interface

    /**
     * DOCUMENT ME!
     *
     * @param ke Does nothing.
     */
    public void keyTyped(KeyEvent ke) {
        ke.consume();
    }

    /**
     * Responds only to the cursor and delete key
     *
     * @param ke
     */
    public void keyPressed(KeyEvent ke) {
        if (ke.getKeyCode() == ke.VK_DELETE) {
            int position = editor.getCaretPosition();
            String string = editor.getText();

            if ((0 < position) && (position < (string.length() - 1))) {
                int prev = string.lastIndexOf(sepChar, position);

                if (prev == -1) {
                    prev = 0;
                }

                editor.setText(string.substring(0, prev + 1) + wildkeyChar +
                    endChar);
            }
        }

        if ((ke.getKeyCode() != ke.VK_LEFT) &&
                (ke.getKeyCode() != ke.VK_RIGHT)) {
            ke.consume();
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param ke Does nothing.
     */
    public void keyReleased(KeyEvent ke) {
        ke.consume();
    }
}
