/*
 * Decompiled with CFR 0.152.
 */
package mpi.annex.data;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.TreeMap;
import java.util.Vector;
import mpi.annex.data.AnnexAnnotation;
import mpi.annex.data.AnnexTier;
import mpi.annex.data.AnnexTranscription;
import mpi.annex.data.FlexEncReader;
import mpi.annex.data.ShoeboxRecord;
import mpi.annex.data.ShoeboxTiers;
import mpi.annex.data.TierType;
import org.apache.log4j.Logger;

public class ShoeboxParser {
    private static Logger _logger = Logger.getLogger(ShoeboxParser.class.getName());
    public String REF_TAG = "ref";
    private ShoeboxTiers tiers;
    private File file;
    private boolean knownTiers;

    public ShoeboxParser(File shoeboxFile) throws IOException {
        this.file = shoeboxFile;
        this.tiers = new ShoeboxTiers();
        this.knownTiers = false;
        this.guessRefTag();
    }

    public ShoeboxParser(File shoeboxFile, File metaFile) {
        this.knownTiers = true;
        this.file = shoeboxFile;
        if (metaFile.getName().endsWith(".mkr")) {
            this.readMarkerFile(metaFile);
        } else {
            this.readMetaFiles(metaFile, null);
        }
    }

    public void readMarkerFile(File markerFile) {
        _logger.debug("Reading marker file " + markerFile);
        this.tiers = new ShoeboxTiers();
        try {
            BufferedReader read = new BufferedReader(new FileReader(markerFile));
            String line = null;
            String marker = null;
            String charset = null;
            String stereotype = null;
            String parent = null;
            while ((line = read.readLine()) != null) {
                if (line.startsWith("marker:")) {
                    marker = line.substring(7, line.length()).trim();
                    continue;
                }
                if (line.startsWith("parent:")) {
                    parent = line.substring(7, line.length()).trim();
                    continue;
                }
                if (line.startsWith("stereotype:")) {
                    stereotype = line.substring(11, line.length()).trim();
                    continue;
                }
                if (!line.startsWith("charset:")) continue;
                charset = line.substring(8, line.length()).trim();
                if ("null".equals(parent)) {
                    this.REF_TAG = marker;
                    this.tiers.put("ref", TierType.REF);
                    continue;
                }
                TierType type = TierType.UNKNOWN;
                if ("Symbolic Subdivision".equals(stereotype)) {
                    type = TierType.ALIGNED;
                } else if ("Time".equals(stereotype)) {
                    type = TierType.TIME;
                }
                if ("UTF-8".equals(charset)) {
                    this.tiers.put(marker, type, "UTF-8");
                } else {
                    this.tiers.put(marker, type);
                }
                this.tiers.setParent(marker, parent);
            }
            read.close();
        }
        catch (IOException e) {
            _logger.error("IOException in readMarkerFile: " + e + " for: " + markerFile);
        }
        _logger.info("typ file tier structure: " + this.tiers + " for: " + markerFile);
    }

    public void readMetaFiles(File typFile, File langFile) {
        this.tiers = new ShoeboxTiers();
        try {
            BufferedReader typ = new BufferedReader(new FileReader(typFile));
            String line = null;
            String marker = null;
            while ((line = typ.readLine()) != null) {
                if (line.startsWith("\\mkrRecord ")) {
                    this.REF_TAG = line.substring(11, line.length()).trim();
                    this.tiers.put("ref", TierType.REF);
                }
                if (line.startsWith("\\+mkr ")) {
                    marker = line.substring(6, line.length()).trim();
                    this.tiers.put(marker, TierType.UNKNOWN);
                    continue;
                }
                if (!line.startsWith("\\mkrOverThis ") || marker == null) continue;
                String parent = line.substring(13, line.length()).trim();
                this.tiers.setParent(marker, parent);
                marker = null;
            }
            typ.close();
        }
        catch (IOException e) {
            _logger.error("IOException in readMetaFiles: " + e + " for: " + typFile);
        }
        _logger.info("typ file tier structure: " + this.tiers + " for: " + typFile);
    }

    private void guessRefTag() throws IOException {
        try {
            BufferedReader read = new BufferedReader(new FileReader(this.file));
            String line = null;
            int countR = 0;
            int countRef = 0;
            while ((line = read.readLine()) != null) {
                if (line.startsWith("\\ref ")) {
                    ++countRef;
                }
                if (!line.startsWith("\\r ")) continue;
                ++countR;
            }
            read.close();
            if (countRef > 0) {
                this.REF_TAG = "ref";
            }
            if (countRef == 0 && countR > 0) {
                this.REF_TAG = "r";
            }
            if (countRef == 0 && countR == 0) {
                _logger.warn("Neither ref nor r record marker found in: " + this.file);
            }
        }
        catch (IOException ioe) {
            _logger.error("IOException in guessRefTag: " + ioe + " for: " + this.file);
            throw ioe;
        }
        if (!"ref".equals(this.REF_TAG)) {
            _logger.info("Assuming record marker " + this.REF_TAG + " for: " + this.file);
        }
    }

    public void parse(AnnexTranscription transcription) throws IOException {
        try {
            FlexEncReader reader = null;
            reader = this.knownTiers ? new FlexEncReader(new FileInputStream(this.file), this.tiers.getEncodingHash()) : new FlexEncReader(this.file);
            ShoeboxRecord record = null;
            ShoeboxRecord lastRecord = null;
            boolean state = false;
            int recCount = 0;
            String line = null;
            int lineNumber = 0;
            StringBuilder header = new StringBuilder("Shoebox header:\n");
            int emptyCount = 0;
            int emptyLen = 0;
            while ((line = reader.readLine()) != null) {
                ++lineNumber;
                if (line.length() == 0) continue;
                if (line.trim().length() == 0) {
                    ++emptyCount;
                    emptyLen += line.length();
                }
                if (this.REF_TAG.equals(ShoeboxRecord.getTierTag(line))) {
                    if (lastRecord != null) {
                        lastRecord.provideAlternativeTime(record, recCount++);
                        this.createTranscription(lastRecord, transcription);
                    }
                    lastRecord = record;
                    record = new ShoeboxRecord(line, this.tiers, this.knownTiers);
                    state = true;
                    transcription.setInfo(header.toString());
                    continue;
                }
                if (!state) {
                    header.append(line.trim());
                    header.append("\n");
                    continue;
                }
                record.put(line);
            }
            if (lastRecord != null) {
                lastRecord.provideAlternativeTime(record, recCount++);
                this.createTranscription(lastRecord, transcription);
            }
            if (record != null) {
                record.provideAlternativeTime(null, recCount++);
                this.createTranscription(record, transcription);
            }
            reader.close();
            if (emptyCount > 0) {
                _logger.info("Whitespace-only lines: " + emptyCount + " (spaces: " + emptyLen + ") in: " + this.file);
            }
        }
        catch (IOException e) {
            _logger.error("IOException in parse: " + e + " for: " + this.file);
            throw e;
        }
        String assoc = this.createTierAssociations(transcription);
        if (assoc.length() == 0) {
            _logger.error("ShoeboxParser: No tiers found in: " + this.file);
            throw new IOException("File contains no Shoebox tiers at all");
        }
        _logger.debug("ShoeboxParser: Tiers: " + assoc + " in file: " + this.file);
    }

    public void tokenizeTier(AnnexTier tier, String content, int longest, long startTime, long endTime) {
        TreeMap annotations = new TreeMap();
        int position = 0;
        int lastPosition = 0;
        while (content.length() > 0) {
            String token = null;
            int firstSpace = content.indexOf(" ");
            if (firstSpace == -1) {
                token = content;
                content = "";
                lastPosition = position;
                position = longest;
            } else {
                lastPosition = position;
                token = content.substring(0, firstSpace);
                position += token.length();
                content = content.substring(firstSpace);
                while (content.length() > 0 && content.charAt(0) == ' ') {
                    content = content.substring(1);
                    ++position;
                }
            }
            AnnexAnnotation annexAnnot = new AnnexAnnotation(token, this.getAlignTime(lastPosition, longest, startTime, endTime), this.getAlignTime(position, longest, startTime, endTime), true, null);
            tier.addAnnotation(annexAnnot);
        }
    }

    private void createTranscription(ShoeboxRecord record, AnnexTranscription transcription) {
        for (String tag : record.tags()) {
            if (record.getContent(tag) == null) continue;
            switch (record.getTierType(tag)) {
                case TIME: {
                    break;
                }
                case ALIGNED: {
                    this.tokenizeTier(this.getOrCreateTier(transcription, tag), record.getContent(tag), record.getLongestAlignedTier(), record.getStartTime(), record.getEndTime());
                    break;
                }
                case UNKNOWN: 
                case BLOCK: 
                case REF: {
                    this.getOrCreateTier(transcription, tag).addAnnotation(new AnnexAnnotation(record.getContent(tag), record.getStartTime(), record.getEndTime(), true, null));
                }
            }
        }
    }

    private String createTierAssociations(AnnexTranscription transcription) {
        StringBuffer assocLog = new StringBuffer();
        for (String tierName : this.tiers.names(false)) {
            AnnexTier parentTier;
            if (this.tiers.getType(tierName) == TierType.ALIGNED) {
                assocLog.append("A:");
            }
            if (this.tiers.getParent(tierName) == null) {
                assocLog.append("[" + tierName + "]");
            } else if ("ref".equals(this.tiers.getParent(tierName))) {
                assocLog.append(tierName);
            } else {
                assocLog.append(tierName + "<" + this.tiers.getParent(tierName));
            }
            AnnexTier childTier = null;
            try {
                childTier = this.getTier(transcription, tierName);
            }
            catch (IllegalArgumentException e) {
                _logger.warn("Warning (createTierAssociations): tier not found: " + tierName);
                assocLog.append("? ");
                continue;
            }
            if (childTier.annotations.size() == 1) {
                AnnexAnnotation anno = (AnnexAnnotation)childTier.annotations.get(0);
                _logger.debug("One-annotation-tier: " + tierName + "[" + anno.beginTime + "-" + anno.endTime + "] <" + anno.value + ">");
                assocLog.append('@');
            }
            assocLog.append(' ');
            if (this.tiers.getParent(tierName) == null) {
                childTier.parentTier = null;
                continue;
            }
            childTier.parentTier = parentTier = this.getTier(transcription, this.tiers.getParent(tierName));
            if (this.tiers.getType(tierName) != TierType.ALIGNED) continue;
            childTier.type = "ALIGNED";
            for (int i = 0; i < childTier.annotations.size(); ++i) {
                AnnexAnnotation annot = (AnnexAnnotation)childTier.annotations.get(i);
                AnnexAnnotation mother = null;
                for (int j = 0; j < parentTier.annotations.size(); ++j) {
                    AnnexAnnotation next = (AnnexAnnotation)parentTier.annotations.get(j);
                    if (next.beginTime > annot.beginTime) break;
                    mother = next;
                }
                if (mother == null) {
                    _logger.warn("Warning: could not find mother for element " + tierName + "." + annot.value + " on level " + this.tiers.getParent(tierName));
                }
                annot.refAnnotation = mother;
            }
        }
        return assocLog.toString().trim();
    }

    private AnnexTier getOrCreateTier(AnnexTranscription transcription, String tierName) {
        try {
            return this.getTier(transcription, tierName);
        }
        catch (IllegalArgumentException e) {
            AnnexTier tier = new AnnexTier(tierName, "text");
            transcription.getTiers().add(tier);
            return tier;
        }
    }

    private AnnexTier getTier(AnnexTranscription transcription, String tierName) throws IllegalArgumentException {
        AnnexTier tier = null;
        for (int i = 0; i < transcription.getTiers().size(); ++i) {
            AnnexTier iterate = (AnnexTier)transcription.getTiers().get(i);
            if (!iterate.name.equals(tierName)) continue;
            tier = iterate;
            break;
        }
        if (tier == null) {
            throw new IllegalArgumentException("tier not found: " + tierName);
        }
        return tier;
    }

    private long getAlignTime(int i, int size, long startTime, long endTime) {
        double position = (double)i / (double)size;
        long duration = endTime - startTime;
        return (long)(position * (double)duration) + startTime;
    }

    private void printAnnexTranscription(AnnexTranscription transcription) {
        for (Object tier : transcription.getTiers()) {
            _logger.info("| " + ((AnnexTier)tier).name + ": " + ((AnnexTier)tier).annotations.size());
        }
    }

    public static void main(String[] args) {
        if (args.length == 0 || args.length > 2) {
            System.out.println("ShoeboxParser sbxfile");
            System.out.println("ShoeboxParser sbxfile typfile");
            System.out.println("ShoeboxParser -d directory");
            System.out.println("[typfile is of format .typ or .mkr]");
            return;
        }
        if ("-d".equals(args[0])) {
            Vector<String> errors = new Vector<String>();
            File dir = new File(args[1]);
            File[] files = dir.listFiles();
            for (int i = 0; i < files.length; ++i) {
                try {
                    System.out.println("Parsing " + files[i].getPath());
                    new AnnexTranscription("test", 2, files[i]);
                    continue;
                }
                catch (RuntimeException e) {
                    System.out.println("Error in: " + files[i].getPath() + ": " + e);
                    _logger.error("RuntimeException in main for: " + files[i].getPath() + ": " + e, e);
                    errors.add(files[i].getPath());
                }
            }
            System.out.println("Parsed " + files.length + " files.");
            System.out.println("Exceptions in: " + errors);
        } else if (args.length == 1) {
            new AnnexTranscription("test", 2, new File(args[0]));
        } else if (args.length == 2) {
            new AnnexTranscription("test", new File(args[0]), new File(args[1]));
        }
    }
}

