/*
 * File:     DCSmall.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.dcr;

import mpi.dcr.isocat.ISOCatConstants;
import mpi.dcr.isocat.Profile;


/**
 * An object to store a summary of the information available for a data
 * category. The id, (camelcase) identifier, a description, the profiles and
 * the broader concept generic are stored.
 *
 * @author Han Sloetjes
 * @version 1.0
 * @version 2.0 July 2009 some changes to be up-to-date with the changes
 * in the dcr model. The signature of getProfiles() and setProfiles() have
 * changed; they now use Profile objects instead of Strings
 */
public class DCSmall {
    private String id;
    private Integer idAsInteger;
    private String identifier;
    private String name;
    private String desc;
    private DCSmall broaderDC; // or just the id?
    private String broaderDCId;
    private Profile[] profiles;
    private boolean loaded = false;

    /**
     * Creates a new DCSmall instance
     *
     * @param profile the profile
     * @param id the unique id
     * @param identifier the textual identifier
     */
    public DCSmall(Profile profile, String id, String identifier) {
        super();

        // throw exception if any param is null
        profiles = new Profile[] { profile };
        this.id = id;
        this.identifier = identifier;
        createIdAsInt(id);
    }

    /**
     * Creates a new DCSmall instance without profile.
     *
     * @param id the unique id
     * @param identifier the textual identifier
     */
    public DCSmall(String id, String identifier) {
        super();

        // throw exception if any param is null
        profiles = new Profile[] {  };
        this.id = id;
        this.identifier = identifier;
        createIdAsInt(id);
    }

    /**
     * Returns the identifier field, a unique string representation.  This way
     * DCSmall can be used as user object in trees etc.
     *
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return identifier;
    }

    private void createIdAsInt(String id) {
        if (id == null) {
            idAsInteger = new Integer(-1);
        }

        try {
            if (id.startsWith(ISOCatConstants.PID_PREFIX)) {
                idAsInteger = Integer.valueOf(id.substring(
                            ISOCatConstants.PID_PREFIX.length()));
            } else {
                idAsInteger = Integer.valueOf(id);
            }
        } catch (NumberFormatException nfe) {
        }
    }

    /**
     * Returns the broader concept generic as a DCSmall object.
     *
     * @return Returns the broaderDC, can be null
     *
     * @see #getBroaderDCId()
     */
    public DCSmall getBroaderDC() {
        return broaderDC;
    }

    /**
     * Sets the broader concept generic as a DCSmall object.
     *
     * @param broaderDC the broaderDC
     */
    public void setBroaderDC(DCSmall broaderDC) {
        this.broaderDC = broaderDC;
    }

    /**
     * Returns the id of the broader concept generic data category.
     *
     * @return Returns the id of the broader DC as a String, or null
     */
    public String getBroaderDCId() {
        return broaderDCId;
    }

    /**
     * Set the id of the broader concept generic data category.
     *
     * @param broaderDCId the id of the broader DC as a String
     */
    public void setBroaderDCId(String broaderDCId) {
        this.broaderDCId = broaderDCId;
    }

    /**
     * Returns a description of the category (in English)
     *
     * @return Returns the description
     */
    public String getDesc() {
        return desc;
    }

    /**
     * Sets the description of the category.
     *
     * @param desc the description
     */
    public void setDesc(String desc) {
        this.desc = desc;
    }

    /**
     * Returns the unique id as a String
     *
     * @return Returns the id as a String
     */
    public String getId() {
        return id;
    }

    /**
     * Sets the id as a String
     *
     * @param id the id as a String
     */
    public void setId(String id) {
        this.id = id;
        createIdAsInt(id);
    }

    /**
     * Returns the (camelcase) textual identifier of the category, e.g.
     * "commonNoun"
     *
     * @return Returns the textual identifier
     */
    public String getIdentifier() {
        return identifier;
    }

    /**
     * Sets the (camelcase) textual identifier of the category.
     *
     * @param identifier the identifier.
     */
    public void setIdentifier(String identifier) {
        this.identifier = identifier;
    }

    /**
     * Returns an array of profiles this category belongs to.
     *
     * @return Returns the profiles
     */
    public Profile[] getProfiles() {
        return profiles;
    }

    /**
     * Sets the array of profiles this category belongs to.
     *
     * @param profiles the profiles
     */
    public void setProfiles(Profile[] profiles) {
        this.profiles = profiles;
    }

    /**
     * Returns the unique id as an Integer.
     *
     * @return Returns the id as an Integer
     */
    public Integer getIdAsInteger() {
        return idAsInteger;
    }

    /**
     * Sets the id as an Integer, the String representation of the id is
     * updated as well.
     *
     * @param idAsInteger the id as an Integer
     */
    public void setIdAsInteger(Integer idAsInteger) {
        this.idAsInteger = idAsInteger;

        if (idAsInteger != null) {
            id = idAsInteger.toString();
        }
    }

    /**
     * Returns whether all information of this summary is loaded.  By default
     * only the id and identifier are loaded (for performance reasons).
     *
     * @return Returns whether this record is fully loaded or only the id and
     *         identifier.
     */
    public boolean isLoaded() {
        return loaded;
    }

    /**
     * Sets the loaded flag.
     *
     * @param loaded set to true when additional information has been loaded
     *        like description , broader concept generic, profiles.
     *
     * @see #isLoaded()
     */
    public void setLoaded(boolean loaded) {
        this.loaded = loaded;
    }

    /**
     * Returns the name of the dc
     *
     * @return the name, can be null
     */
    public String getName() {
        return name;
    }

    /**
     * Sets the name of the dc
     * @param name the name of the dc
     */
    public void setName(String name) {
        this.name = name;
    }
}
