/*
 * File:     IDCRConnector.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.dcr;

import mpi.dcr.ISO12620.DataCategory;
import mpi.dcr.ISO12620.DataCategorySelection;
import mpi.dcr.ISO12620.ProfileList;

import java.util.List;


/**
 * This interface describes the basic interface for a DCR Connector as part of
 * the Lirics project
 *
 * @author markem
 * @version 1.0
 */
public interface IDCRConnector {
    /**
     * Returns the name of the DCR.  NOTE: this method is not part of the
     * interface declaration. It i merely a convenience method for the user to
     * identify the DCR.
     *
     * @return the name of the DCR
     */
    public String getName();

    /**
     * Returns the List of data categories from the specified profile.  The
     * registration status is optional. If specified, the List of
     * Datacategories  will only contain those with the specifed registration
     * status.
     *
     * @param a_profile
     * @param a_registrationStatus
     *
     * @return the List of data categories from the specified profile
     *
     * @throws DCRConnectorException
     */
    public DataCategorySelection getDataCategories(String a_profile,
        String a_registrationStatus) throws DCRConnectorException;

    /**
     * Returns the data category identfied by the specified urid
     *
     * @param a_urid
     *
     * @return the data category identfied by the specified urid
     *
     * @throws DCRConnectorException
     */
    public DataCategory getDataCategory(String a_urid)
        throws DCRConnectorException;

    /**
     * Returns the List of profiles registered in the DCR
     *
     * @return the List of profiles registered in the DCR
     *
     * @throws DCRConnectorException
     */
    public ProfileList getProfiles() throws DCRConnectorException;

    /**
     * Searches the DCR for the list of keywords in the specified list of
     * fields. The search can be further refined by specifying the profile or
     * registration status of the data category.
     *
     * @param a_listOfKeywords
     * @param a_listOfFields
     * @param a_profile
     * @param a_registrationStatus
     *
     * @return the search results
     *
     * @throws DCRConnectorException
     */
    public DataCategorySelection searchDataCategories(List a_listOfKeywords,
        List a_listOfFields, String a_profile, String a_registrationStatus)
        throws DCRConnectorException;
}
