/*
 * File:     LATDCRConnector.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.dcr;

import mpi.dcr.ISO12620.ConceptualDomain;
import mpi.dcr.ISO12620.DCR;
import mpi.dcr.ISO12620.DataCategory;
import mpi.dcr.ISO12620.DataCategorySelection;
import mpi.dcr.ISO12620.DataCategorySummary;
import mpi.dcr.ISO12620.Definition;
import mpi.dcr.ISO12620.DefinitionBrack;
import mpi.dcr.ISO12620.Profile;
import mpi.dcr.ISO12620.ProfileList;
import mpi.dcr.ISO12620.RegistrationStatus;

import org.xml.sax.SAXException;

import java.io.IOException;
import java.io.InputStream;

import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.xml.parsers.ParserConfigurationException;


/**
 * Implementation of a DCR connector that can connect to a remote DCR server.
 * Currently the Syntax server and Isocat are "known" and supported.   To do:
 * reintroduce logging?? To do: reintroduce afterPropertiesSet and/or an
 * alternative to retrieve the DCRLocation, proxy etc.
 */
public class LATDCRConnector /*implements ILATDCRConnector*/ {
    /** Refers to the URL for the DCR server */
    private String DCRLocation = "To be filled in using applicationContext file";

    // http://syntax.inist.fr/mod_webservice/call.php
    private String dcrName = "To be filled in using applicationContext file";

    /** Refers to the proxy server. Some organizations require use of a proxy */
    private String proxy = null;

    /** Refers to the proxy port. Used in conjunction with the proxy. */
    private int proxyPort = -1;

    /**
     * Refers to the name for the DCR Connector. To be set by the application
     * context, default is LAT DCR Connector.
     */
    private String name = "LAT DCR Connector";
    private String syntaxLocation = "http://syntax.inist.fr/mod_webservice/call.php";
    private String syntaxName = "SYNTAX ISO12620";
    private String isocatLocation = "http://lux12.mpi.nl/isocat/rpc/syntax";
    private String isocatName = "ISOCAT ISO12620";

    /**
     * Constructor.
     */
    public LATDCRConnector() {
        super();

        // sets defaults
        //DCRLocation = isocatLocation;
        //dcrName = isocatName;
        DCRLocation = syntaxLocation;
        dcrName = syntaxName;

        String locProperty = System.getProperty("DCRLocation");

        if (isocatLocation.equals(locProperty)) {
            DCRLocation = isocatLocation;
            dcrName = isocatName;
        } else if (syntaxLocation.equals(locProperty)) {
            DCRLocation = syntaxLocation;
            dcrName = syntaxName;
        } else if (locProperty != null) {
            DCRLocation = locProperty;
            dcrName = DCRLocation;
        }
    }

    /**
     * REturns the location of the DCR.
     *
     * @return Returns the dCRLocation.
     */
    public String getDCRLocation() {
        return DCRLocation;
    }

    /**
     * Sets the location of the DCR.
     *
     * @param location The dCRLocation to set.
     */
    public void setDCRLocation(String location) {
        DCRLocation = location;
    }

    /**
     * Returns the proxy value.
     *
     * @return Returns the proxy.
     */
    public String getProxy() {
        return proxy;
    }

    /**
     * Sets the proxy value.
     *
     * @param proxy The proxy to set.
     */
    public void setProxy(String proxy) {
        this.proxy = proxy;
    }

    /**
     * Retruns the proxy port.
     *
     * @return Returns the proxyPort.
     */
    public int getProxyPort() {
        return proxyPort;
    }

    /**
     * Sets the proxy port
     *
     * @param proxyPort The proxyPort to set.
     */
    public void setProxyPort(int proxyPort) {
        this.proxyPort = proxyPort;
    }

    /**
     * @see org.mpi.DCR.IDCRConnector#getName()
     */
    public String getName() {
        return name; // return dcrName?
    }

    /**
     * Sets the name of this connector.
     *
     * @param name The name to set.
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Returns a list of the data categories belonging to the specified profile
     * in the form of summary objects, containing only the id, identifier and
     * profile.
     *
     * @param a_profile the profile name
     * @param a_registrationStatus the registration status
     *
     * @return a list of DCSmall objects
     *
     * @throws DCRConnectorException when any error occurs in connecting to the
     *         dcr server.
     */
    public List getDCSmallList(String a_profile, String a_registrationStatus)
        throws DCRConnectorException {
        DataCategorySelection dcs = getDataCategories(a_profile,
                a_registrationStatus);
        List dcList = new ArrayList();
        DCSmall tinyDC = null;

        DataCategorySummary[] summs = dcs.getDataCategorySummaries();

        for (int j = 0; j < summs.length; j++) {
            tinyDC = new DCSmall(new mpi.dcr.isocat.Profile(null, a_profile),
                    summs[j].getIdAsString(),
                    summs[j].getIdentifier().getContentAsString());
            dcList.add(tinyDC);
        }

        return dcList;
    }

    /**
     * Returns a list of the data categories belonging to the specified profile
     * in the form of summary objects, containing the id, identifier,
     * profile, a description and the broader concept value.
     *
     * @param a_profile the profile name
     * @param a_registrationStatus the registration status
     *
     * @return a list of DCSmall objects
     *
     * @throws DCRConnectorException when any error occurs in connecting to the
     *         dcr server.
     */
    public List getDCSmallList2(String a_profile, String a_registrationStatus)
        throws DCRConnectorException {
        DataCategorySelection dcs = getDataCategories(a_profile,
                a_registrationStatus);
        List dcList = new ArrayList();
        DCSmall tinyDC = null;

        DataCategorySummary[] summs = dcs.getDataCategorySummaries();
        DataCategory dc;
        Object defObj;
        DefinitionBrack dBrack;
        Definition defin;

        for (int j = 0; j < summs.length; j++) {
            try {
                dc = getDataCategory(summs[j].getIdAsString());

                //System.out.println("\t" + summs[j].getIdentifier().getContentAsString() + " " + dc.getId());
                //System.out.println("\t\t" + dc.getDescription());
                tinyDC = new DCSmall(new mpi.dcr.isocat.Profile(null, a_profile),
                        dc.getId(),
                        summs[j].getIdentifier().getContentAsString());

                if ((dc.getDescription() != null) &&
                        (dc.getDescription().getDefinitions() != null)) {
                    defObj = dc.getDescription().getDefinition(0);

                    if (defObj instanceof DefinitionBrack) {
                        dBrack = (DefinitionBrack) defObj;
                        defin = dBrack.getDefinition();
                    } else if (defObj instanceof Definition) {
                        defin = (Definition) defObj;
                    } else {
                        defin = null;
                    }

                    if (defin != null) {
                        tinyDC.setDesc(defin.getContentAsString());

                        //System.out.println("\t\t" + defin.getContentAsString());
                    } else {
                        //System.out.println("Definition is null...");
                    }
                } else {
                    if (dc.getDescription() == null) {
                        //System.out.println("Description is null...");
                    } else {
                        tinyDC.setDesc(dc.getDescription().getTypeAsString()); //??
                                                                               //System.out.println("Definitions is null...");
                    }
                }

                if ((dc.getDescription() != null) &&
                        (dc.getDescription().getBroaderConceptGeneric() != null)) {
                    //System.out.println("\t\t" + dc.getDescription().getBroaderConceptGeneric().getContentAsString());
                    tinyDC.setBroaderDCId(dc.getDescription()
                                            .getBroaderConceptGeneric()
                                            .getContentAsString());
                }

                if ((dc.getDescription() != null) &&
                        (dc.getDescription().getProfiles() != null)) {
                    Profile[] profs = dc.getDescription().getProfiles();

                    //String[] prNames = new String[profs.length];
                    mpi.dcr.isocat.Profile[] newProfs = new mpi.dcr.isocat.Profile[profs.length];

                    for (int i = 0; i < profs.length; i++) {
                        //prNames[i] = profs[i].getContentAsString();
                        newProfs[i] = new mpi.dcr.isocat.Profile(null,
                                profs[i].getContentAsString());
                    }

                    tinyDC.setProfiles(newProfs);
                }

                if ((dc.getDescription() != null) &&
                        (dc.getDescription().getConceptualDomains() != null)) {
                    ConceptualDomain[] doms = dc.getDescription()
                                                .getConceptualDomains();

                    for (int i = 0; i < doms.length; i++) {
                        //System.out.println("\t\t\tConceptual Domain: " + doms[i].getContentAsString());
                    }
                }

                dcList.add(tinyDC);
            } catch (DCRConnectorException dce) {
                System.out.println(dce.getMessage());
            }
        }

        return dcList;
    }

    /**
     * Retrieves additional information, description (English), broader concept
     * generic value  and all profiles the specified category belongs to.
     *
     * @param dcSmall the category summary object containing only the id and
     *        identifier
     */
    public void getMoreInfo(DCSmall dcSmall) {
        if ((dcSmall == null) || dcSmall.isLoaded()) {
            return;
        }

        try {
            DataCategory dc = getDataCategory(dcSmall.getId());
            Object defObj;
            DefinitionBrack dBrack;
            Definition defin;

            if ((dc.getDescription() != null) &&
                    (dc.getDescription().getDefinitions() != null)) {
                defObj = dc.getDescription().getDefinition(0);

                if (defObj instanceof DefinitionBrack) {
                    dBrack = (DefinitionBrack) defObj;
                    defin = dBrack.getDefinition();
                } else if (defObj instanceof Definition) {
                    defin = (Definition) defObj;
                } else {
                    defin = null;
                }

                if (defin != null) {
                    dcSmall.setDesc(defin.getContentAsString());

                    //System.out.println("\t\t" + defin.getContentAsString());
                } else {
                    //System.out.println("Definition is null...");
                }
            } else {
                if (dc.getDescription() == null) {
                    //System.out.println("Description is null...");
                } else {
                    dcSmall.setDesc(dc.getDescription().getTypeAsString()); //??
                                                                            //System.out.println("Definitions is null...");
                }
            }

            if ((dc.getDescription() != null) &&
                    (dc.getDescription().getBroaderConceptGeneric() != null)) {
                //System.out.println("\t\t" + dc.getDescription().getBroaderConceptGeneric().getContentAsString());
                dcSmall.setBroaderDCId(dc.getDescription()
                                         .getBroaderConceptGeneric()
                                         .getContentAsString());
            }

            if ((dc.getDescription() != null) &&
                    (dc.getDescription().getProfiles() != null)) {
                Profile[] profs = dc.getDescription().getProfiles();

                //String[] prNames = new String[profs.length];
                mpi.dcr.isocat.Profile[] newProfs = new mpi.dcr.isocat.Profile[profs.length];

                for (int i = 0; i < profs.length; i++) {
                    //prNames[i] = profs[i].getContentAsString();
                    newProfs[i] = new mpi.dcr.isocat.Profile(null,
                            profs[i].getContentAsString());
                }

                dcSmall.setProfiles(newProfs);
            }

            dcSmall.setLoaded(true);
        } catch (DCRConnectorException dce) {
            System.out.println(dce.getMessage());
        }
    }

    /**
     * Returns a data category selection containing summaries of the data
     * categories  of the specified profile.
     *
     * @param a_profile the profile name
     * @param a_registrationStatus the registration status
     *
     * @return the selection
     *
     * @throws DCRConnectorException if any connection error occurs
     * @throws IllegalArgumentException if the profile name is null
     */
    public DataCategorySelection getDataCategories(String a_profile,
        String a_registrationStatus) throws DCRConnectorException {
        if (a_profile == null) {
            throw new IllegalArgumentException(
                "Expect a valid profile (not null) here");
        }

        if (a_registrationStatus != null) {
            if (!RegistrationStatus.isValid(a_registrationStatus)) {
                throw new IllegalArgumentException(
                    "The specified registrationStatus [" +
                    a_registrationStatus + "]is not valid");
            }
        }

        DataCategorySelection dcs = null;

        URL url = null;

        try { // use dcrLocation

            String urlString = DCRLocation + "?fct=getDataCategories&profile=" +
                a_profile;

            if (a_registrationStatus != null) {
                urlString += ("&status=" + a_registrationStatus);
            }

            if ((this.proxy != null) && (this.proxyPort != -1)) { //why should the port != -1?
                url = new URL("http", proxy, proxyPort, urlString);
            } else {
                url = new URL(urlString);
            }
        } catch (MalformedURLException mue) {
            throw new DCRConnectorException("Unable to connect to DCR: " +
                mue.getMessage());
        }

        if (url != null) {
            HttpURLConnection conn = getConnection(url);

            try {
                Object cont = conn.getContent();

                if (cont instanceof InputStream) {
                    InputStream is = (InputStream) cont;

                    try {
                        dcs = new DataCategorySelection(is);
                    } catch (ParserConfigurationException e) {
                        throw new DCRConnectorException(
                            "Unable to parse stream DCR: " + e);
                    } catch (SAXException saxe) {
                        throw new DCRConnectorException(
                            "Unable to parse stream DCR: " + saxe.getMessage());
                    } catch (IOException e) {
                        throw new DCRConnectorException(e);
                    }
                }
            } catch (IOException ioee) {
                throw new DCRConnectorException(ioee);
            } finally {
                conn.disconnect();
            }
        }

        return dcs;
    }

    /**
     * Returns the data category object with the specified id
     *
     * @param a_urid the unique id
     *
     * @return a data category object
     *
     * @throws DCRConnectorException if any connection error occurs
     * @throws IllegalArgumentException if the urid is null
     */
    public DataCategory getDataCategory(String a_urid)
        throws DCRConnectorException {
        if (a_urid == null) {
            throw new IllegalArgumentException(
                "A valid URID(not null) is expected here");
        }

        DCR dcr = null;

        URL url = null;

        try { // use dcrLocation

            String urlString = DCRLocation + "?fct=getDataCategory&urid=" +
                a_urid;

            if ((this.proxy != null) && (this.proxyPort != -1)) { //why should the port != -1?
                url = new URL("http", proxy, proxyPort, urlString);
            } else {
                url = new URL(urlString);
            }
        } catch (MalformedURLException mue) {
            throw new DCRConnectorException("Unable to connect to DCR: " +
                mue.getMessage());
        }

        if (url != null) {
            HttpURLConnection conn = getConnection(url);

            try {
                Object cont = conn.getContent();

                if (cont instanceof InputStream) {
                    InputStream is = (InputStream) cont;

                    try {
                        dcr = new DCR(is);
                    } catch (ParserConfigurationException e) {
                        throw new DCRConnectorException(
                            "Unable to parse stream DCR: " + e);
                    } catch (SAXException saxe) {
                        throw new DCRConnectorException(
                            "Unable to parse stream DCR: " + saxe.getMessage());
                    } catch (IOException e) {
                        throw new DCRConnectorException(e);
                    }
                }
            } catch (IOException ioee) {
                throw new DCRConnectorException(ioee);
            } finally {
                conn.disconnect();
            }
        }

        DataCategory[] datcats = dcr.getDataCategories();

        if (datcats.length == 0) {
            throw new DCRConnectorException(
                "No dataCategories retrieved using the specified URID[" +
                a_urid + "]");
        }

        if (datcats.length > 1) {
            throw new DCRConnectorException(
                "Multiple dataCategories retrieved using the specified URID[" +
                a_urid + "]");
        }

        return datcats[0];
    }

    /**
     * Returns a list of profiles.
     *
     * @return list of profiles
     *
     * @throws DCRConnectorException if any connection error occurs
     */
    public ProfileList getProfiles() throws DCRConnectorException {
        ProfileList list = null;
        URL url = null;

        try { // use dcrLocation

            if ((this.proxy != null) && (this.proxyPort != -1)) { //why should the port != -1?
                url = new URL("http", proxy, proxyPort,
                        DCRLocation + "?fct=getProfiles"); // this is wrong? possible double http://??
            } else {
                url = new URL(DCRLocation + "?fct=getProfiles");
            }
        } catch (MalformedURLException mue) {
            throw new DCRConnectorException("Unable to connect to DCR: " +
                mue.getMessage());
        }

        if (url != null) {
            HttpURLConnection conn = getConnection(url);

            try {
                Object cont = conn.getContent();

                if (cont instanceof InputStream) {
                    InputStream is = (InputStream) cont;

                    try {
                        list = new ProfileList(is);
                    } catch (ParserConfigurationException e) {
                        throw new DCRConnectorException(
                            "Unable to parse stream DCR: " + e);
                    } catch (SAXException saxe) {
                        throw new DCRConnectorException(
                            "Unable to parse stream DCR: " + saxe.getMessage());
                    } catch (IOException e) {
                        throw new DCRConnectorException(e);
                    }
                }
            } catch (IOException ioee) {
                throw new DCRConnectorException(ioee);
            } finally {
                conn.disconnect();
            }
        }

        return list;
    }

    /**
     * @see org.mpi.DCR.IDCRConnector#searchDataCategories(java.util.List,
     *      java.util.List, java.lang.String, java.lang.String)
     */
    public DataCategorySelection searchDataCategories(List a_listOfKeywords,
        List a_listOfFields, String a_profile, String a_registrationStatus)
        throws DCRConnectorException {
        if (a_listOfKeywords == null) {
            throw new IllegalArgumentException(
                "Expect a valid List Of Keywords (not null) here");
        }

        if (a_listOfKeywords.size() == 0) {
            throw new IllegalArgumentException(
                "Expect at least one Keyword here");
        }

        if (a_profile != null) {
            if (!this.isValid(a_profile)) {
                throw new IllegalArgumentException("The specified profile[" +
                    a_profile +
                    "] is not registered in the DCR. You may check the available list of profiles using the getProfiles method");
            }
        }

        if (a_registrationStatus != null) {
            if (!RegistrationStatus.isValid(a_registrationStatus)) {
                throw new IllegalArgumentException(
                    "The specified registrationStatus [" +
                    a_registrationStatus + "]is not valid");
            }
        }

        URL url = null;

        try { // use dcrLocation

            StringBuffer urlBuf = new StringBuffer(DCRLocation +
                    "?fct=searchDataCategories");
            String keyword = null;
            Iterator it = a_listOfKeywords.iterator();
            String kw = "&keywords=";

            if (DCRLocation.equals(syntaxLocation)) {
                kw = "&keywords[]=";
            }

            while (it.hasNext()) {
                keyword = (String) it.next();
                urlBuf.append(kw); // syntax server: &keywords[]= ... isocat: &keywords=
                urlBuf.append(keyword);
            }

            if (a_listOfFields != null) {
                it = a_listOfFields.iterator();

                String fw = "&fields=";

                if (DCRLocation.equals(syntaxLocation)) {
                    fw = "&fields[]=";
                }

                while (it.hasNext()) {
                    keyword = (String) it.next();
                    urlBuf.append(fw); // syntax server: &fields[]= ... isocat: &fields=
                    urlBuf.append(keyword);
                }
            }

            if (a_profile != null) {
                urlBuf.append("&profile=");
                urlBuf.append(a_profile);
            }

            if (a_registrationStatus != null) {
                urlBuf.append("&status=");
                urlBuf.append(a_registrationStatus);
            }

            if ((this.proxy != null) && (this.proxyPort != -1)) { //why should the port != -1?
                url = new URL("http", proxy, proxyPort, urlBuf.toString());
            } else {
                url = new URL(urlBuf.toString());
            }
        } catch (MalformedURLException mue) {
            throw new DCRConnectorException("Unable to connect to DCR: " +
                mue.getMessage());
        }

        DataCategorySelection dcs = null;

        if (url != null) {
            HttpURLConnection conn = getConnection(url);

            try {
                Object cont = conn.getContent();

                if (cont instanceof InputStream) {
                    InputStream is = (InputStream) cont;

                    try {
                        dcs = new DataCategorySelection(is);
                    } catch (ParserConfigurationException e) {
                        throw new DCRConnectorException(
                            "Unable to parse stream DCR: " + e);
                    } catch (SAXException saxe) {
                        throw new DCRConnectorException(
                            "Unable to parse stream DCR: " + saxe.getMessage());
                    } catch (IOException e) {
                        throw new DCRConnectorException(e);
                    }
                }
            } catch (IOException ioee) {
                throw new DCRConnectorException(ioee);
            } finally {
                conn.disconnect();
            }
        }

        return dcs;
    }

    /**
     * Returns whether the specified profile String is a valid profile
     *
     * @param a_profile the name of a profile
     *
     * @return whether the specified profile String is a valid profile
     *
     * @throws DCRConnectorException if any connection error occurs
     */
    private boolean isValid(String a_profile) throws DCRConnectorException {
        if (a_profile == null) {
            return false;
        }

        ProfileList list = this.getProfiles();
        boolean isValid = false;

        for (int i = 0; i < list.sizeProfiles(); i++) {
            Profile profile = list.getProfiles(i);

            if ((profile.getContent() != null) &&
                    profile.getContent().equals(a_profile)) {
                isValid = true;

                break;
            }
        }

        return isValid;
    }

    /**
     * Creates and configures a HTTP Url connection.
     *
     * @param url the url (with parameters)
     *
     * @return a HttpURLConnection
     *
     * @throws DCRConnectorException if an IOException or any other exception
     *         occurs
     */
    private HttpURLConnection getConnection(URL url)
        throws DCRConnectorException {
        if (url == null) {
            return null;
        }

        HttpURLConnection conn = null;

        try {
            conn = (HttpURLConnection) url.openConnection();
            conn.setDefaultUseCaches(true);
            conn.setRequestMethod("GET");
            conn.connect();

            return conn;
        } catch (IOException ioe) {
            throw new DCRConnectorException("Unable to connect to DCR" +
                ioe.getMessage());
        } catch (Exception e) {
            throw new DCRConnectorException("Unable to connect to DCR" +
                e.getMessage());
        }
    }
}
