/*
 * File:     ActiveAnnotationEditCommand.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ViewerManager2;

import mpi.eudico.client.annotator.viewer.TimeLineViewer;

import mpi.eudico.server.corpora.clom.Annotation;

import mpi.eudico.server.corpora.clomimpl.abstr.AlignableAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.RefAnnotation;


/**
 * A command that jumps to the next or previous annotation and starts editing
 * at once.
 *
 * @author Han Sloetjes
 * @version Aug 2008
 */
public class ActiveAnnotationEditCommand implements Command {
    private String commandName;

    /**
     * Creates a new ActiveAnnotationEditCommand instance
     *
     * @param name the name
     */
    public ActiveAnnotationEditCommand(String name) {
        commandName = name;
    }

    /**
     * <b>Note: </b>it is assumed that the types and order of the arguments are
     * correct.
     *
     * @param receiver the ViewerManager
     * @param arguments the arguments:  <ul><li>arg[0] = the annotation to
     *        activate and edit (Annotation)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        ViewerManager2 vm = (ViewerManager2) receiver;
        Annotation annot = (Annotation) arguments[0];

        if (annot == null) {
            return;
        }

        vm.getActiveAnnotation().setAnnotation(annot);

        // should the selection be set??
        if (annot instanceof AlignableAnnotation) {
            vm.getSelection().setSelection(annot.getBeginTimeBoundary(),
                annot.getEndTimeBoundary());

            if (!vm.getMediaPlayerController().isBeginBoundaryActive()) {
                vm.getMediaPlayerController().toggleActiveSelectionBoundary();
            }
        } else if (annot instanceof RefAnnotation) {
            Annotation parent = annot;

            while (true) {
                parent = parent.getParentAnnotation();

                if ((parent == null) || parent instanceof AlignableAnnotation) {
                    break;
                }
            }

            if (parent instanceof AlignableAnnotation) {
                AlignableAnnotation aa = (AlignableAnnotation) parent;
                vm.getSelection().setSelection(aa.getBeginTimeBoundary(),
                    aa.getEndTimeBoundary());
            }
        }

        int mode = ELANCommandFactory.getLayoutManager(vm.getTranscription())
                                     .getMode();

        if (mode == ELANCommandFactory.getLayoutManager(vm.getTranscription()).NORMAL_MODE) {
            if (ELANCommandFactory.getLayoutManager(vm.getTranscription())
                                      .getVisibleMultiTierViewer() instanceof TimeLineViewer) {
                ELANCommandFactory.getLayoutManager(vm.getTranscription())
                                  .getTimeLineViewer().showEditBoxForAnnotation(annot);
            } else { // only 2 options now
                ELANCommandFactory.getLayoutManager(vm.getTranscription())
                                  .getInterlinearViewer()
                                  .showEditBoxForAnnotation(annot);
            }
        }
    }

    /**
     * Returns the name.
     *
     * @return the name
     */
    public String getName() {
        return commandName;
    }
}
