/*
 * File:     AddTypeCommand.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.type.Constraint;
import mpi.eudico.server.corpora.clomimpl.type.LinguisticType;


/**
 * Adds a new Linguistic Type to a Transcription.
 *
 * @author Han Sloetjes
 * @version jun 04 added the name of a linked Controlled Vocabulary
 * @version apr 08 added the id of a data category (ISO 12620, DCR); this value is added as last object
 * in the arguments array
 */
public class AddTypeCommand implements UndoableCommand {
    private String commandName;

    // receiver
    private Transcription transcription;

    // store the arguments for undo /redo, not used actually
    private String typeName;
    private Constraint constraint;
    private String cvName;
    private String dcId;
    private boolean timeAlignable;
    private boolean graphicsAllowed;

    // the new LinguisticType
    private LinguisticType newLT;

    /**
     * Creates a new AddTypeCommand instance
     *
     * @param name the name of the command
     */
    public AddTypeCommand(String name) {
        commandName = name;
    }

    /**
     * The undo action.
     */
    public void undo() {
        if (newLT != null) {
            transcription.removeLinguisticType(newLT);
        }
    }

    /**
     * The redo action.
     */
    public void redo() {
        if (newLT != null) {
            transcription.addLinguisticType(newLT);
        }
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver the Transcription
     * @param arguments the arguments: <ul><li>arg[0] = the name of the type
     *        (String)</li> <li>arg[1] = the the stereotype name for the type
     *        (ConstraintImpl)</li> <li>arg[2] = the name of the ControlledVocabulary
     *        (String)</li> <li>arg[3] = time alignable value (Boolean)</li>
     *        <li>arg[4] = graphics reference allowed value (Boolean)</li>
     *        <li>arg[5] = data category ID (String)</li>
     *        </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        transcription = (Transcription) receiver;
        typeName = (String) arguments[0];
        constraint = (Constraint) arguments[1];
        cvName = (String) arguments[2];
        timeAlignable = ((Boolean) arguments[3]).booleanValue();
        graphicsAllowed = ((Boolean) arguments[4]).booleanValue();

        if (arguments.length >= 6) {
            dcId = (String) arguments[5];
        }

        newLT = new LinguisticType(typeName);
        newLT.setControlledVocabularyName(cvName);
        newLT.setTimeAlignable(timeAlignable);
        newLT.setGraphicReferences(graphicsAllowed);

        if (constraint != null) {
            newLT.addConstraint(constraint);
        }

        if (dcId != null) {
            newLT.setDataCategory(dcId);
        }

        transcription.addLinguisticType(newLT);
    }

    /**
     * Returns the name of the command.
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }
}
