/*
 * File:     AnnotationBeforeCommand.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.Preferences;
import mpi.eudico.client.annotator.ViewerManager2;

import mpi.eudico.client.annotator.util.AnnotationDataRecord;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.AlignableAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;


/**
 * A command for creating a new annotation before a given annotation.
 *
 * @author Han Sloetjes
 */
public class AnnotationBeforeCommand implements UndoableCommand {
    private String commandName;
    private TierImpl tier;
    private Transcription transcription;
    private AnnotationDataRecord activeAnnRecord;
    private long aaMiddle;
    private long newAnnMiddle;

    /**
     * Creates a new AnnotationBeforeCommand instance
     *
     * @param name the name of the command
     */
    public AnnotationBeforeCommand(String name) {
        commandName = name;
    }

    /**
     * The undo action.
     */
    public void undo() {
        if (tier != null) {
            Annotation delAnn = tier.getAnnotationAtTime(newAnnMiddle);

            if (delAnn != null) {
                tier.removeAnnotation(delAnn);
            }

            if (tier.isTimeAlignable()) {
                AlignableAnnotation activeAnn = (AlignableAnnotation) (tier.getAnnotationAtTime(aaMiddle));

                if ((activeAnn != null) &&
                        ((activeAnn.getBegin().isTimeAligned() != activeAnnRecord.isBeginTimeAligned()) ||
                        (activeAnn.getEnd().isTimeAligned() != activeAnnRecord.isEndTimeAligned()))) {
                    int curPropMode = 0;

                    curPropMode = transcription.getTimeChangePropagationMode();

                    if (curPropMode != Transcription.NORMAL) {
                        transcription.setTimeChangePropagationMode(Transcription.NORMAL);
                    }

                    activeAnn.updateTimeInterval(activeAnnRecord.getBeginTime(),
                        activeAnnRecord.getEndTime());

                    // restore the time propagation mode
                    transcription.setTimeChangePropagationMode(curPropMode);
                }
            }
        }
    }

    /**
     * The redo action.
     */
    public void redo() {
        if (tier != null) {
            Annotation beforeAnn = tier.getAnnotationAtTime(aaMiddle);

            if (beforeAnn != null) {
                tier.createAnnotationBefore(beforeAnn);
            }
        }
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver the TierImpl
     * @param arguments the arguments:  <ul><li>arg[0] = the active annotation
     *        (Annotation)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        tier = (TierImpl) receiver;

        Annotation activeAnn = (Annotation) arguments[0];
        activeAnnRecord = new AnnotationDataRecord(activeAnn);
        aaMiddle = (activeAnnRecord.getBeginTime() +
            activeAnnRecord.getEndTime()) / 2;

        transcription = (Transcription) tier.getParent();

        Annotation newAnnotation = tier.createAnnotationBefore(activeAnn);

        if (newAnnotation != null) {
            newAnnMiddle = (newAnnotation.getBeginTimeBoundary() +
                newAnnotation.getEndTimeBoundary()) / 2;

            Object pref = Preferences.get("ClearSelectionAfterCreation", null);

            if (pref instanceof Boolean) {
                if (((Boolean) pref).booleanValue()) {
                    ViewerManager2 vm = ELANCommandFactory.getViewerManager(transcription);

                    if (vm.getMediaPlayerController().getSelectionMode() == true) {
                        vm.getSelection().setSelection(vm.getMediaPlayerController()
                                                         .getMediaTime(),
                            vm.getMediaPlayerController().getMediaTime());
                    } else {
                        vm.getSelection().setSelection(0, 0);
                    }
                }
            }
        }
    }

    /**
     * Returns the name of the command.
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }
}
