package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.Constants;

import mpi.eudico.client.annotator.util.AnnotationDataRecord;
import mpi.eudico.client.annotator.util.ClientLogger;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.Tier;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.AbstractAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.AlignableAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;
import mpi.eudico.server.corpora.clomimpl.type.LinguisticType;

import mpi.util.TimeFormatter;

import java.util.ArrayList;
import java.util.List;


/**
 * A command that creates annotations based on the gaps between annotations on
 * a given tier. The new annotations can either be inserted on the same tier
 * or created on an other tier.
 *
 * @author Han Sloetjes
 */
public class AnnotationsFromGapsCommand implements UndoableCommand {
    private String commandName;
    private TranscriptionImpl transcription;
    private long mediaDuration;
    private String sourceTier;
    private String destTierName;
    private String annValue;
    private String timeFormat;
    private List<AnnotationDataRecord> createdAnnos;

    /**
     * Constructor.
     *
     * @param name the name of the command
     */
    public AnnotationsFromGapsCommand(String name) {
        commandName = name;
    }

    /**
     * Recreates the new tier (if applicable) and the newly created
     * annotations.
     */
    public void redo() {
        if (transcription != null) {
            TierImpl dt = (TierImpl) transcription.getTierWithId(sourceTier);

            if (dt == null) {
                ClientLogger.LOG.severe(
                    "The source tier does not exist anymore");

                return;
            }

            if (destTierName != null) {
                if (transcription.getTierWithId(destTierName) != null) {
                    ClientLogger.LOG.severe(
                        "The destination tier already exists");

                    return;
                }

                LinguisticType lType = dt.getLinguisticType();

                if (lType.getConstraints() != null) {
                    // the source tier should be a toplevel tier
                    ClientLogger.LOG.severe(
                        "The source tier is not a root tier.");

                    return;
                }

                TierImpl dTier = new TierImpl(null, destTierName,
                        dt.getParticipant(), transcription, lType);
                dTier.setAnnotator(dt.getAnnotator());
                dTier.setDefaultLocale(dt.getDefaultLocale());
                transcription.addTier(dTier);
                dt = dTier;
            }

            if (dt == null) {
                ClientLogger.LOG.severe(
                    "Could not find the destination tier for redo");

                return;
            }

            if ((createdAnnos == null) && (createdAnnos.size() == 0)) {
                ClientLogger.LOG.info("No annotations to restore");

                return;
            }

            int curPropMode = 0;

            curPropMode = transcription.getTimeChangePropagationMode();

            if (curPropMode != Transcription.NORMAL) {
                transcription.setTimeChangePropagationMode(Transcription.NORMAL);
            }

            transcription.setNotifying(false);

            AnnotationDataRecord record;
            Annotation ann;

            for (int i = 0; i < createdAnnos.size(); i++) {
                record = (AnnotationDataRecord) createdAnnos.get(i);
                ann = dt.createAnnotation(record.getBeginTime(),
                        record.getEndTime());

                if ((ann != null) && (record.getValue() != null)) {
                    ann.setValue(record.getValue());
                }
            }

            transcription.setNotifying(true);
            // restore the time propagation mode
            transcription.setTimeChangePropagationMode(curPropMode);
        }
    }

    /**
     * Deletes the new annotations and/or the new tier.
     */
    public void undo() {
        if (transcription != null) {
            if (destTierName != null) {
                Tier dt = transcription.getTierWithId(destTierName);

                if (dt != null) {
                    transcription.removeTier(dt);
                }
            } else {
                TierImpl st = (TierImpl) transcription.getTierWithId(sourceTier);

                if (st != null) {
                    if ((createdAnnos != null) && (createdAnnos.size() > 0)) {
                        transcription.setNotifying(false);

                        AnnotationDataRecord record;
                        Annotation ann;

                        for (int i = 0; i < createdAnnos.size(); i++) {
                            record = (AnnotationDataRecord) createdAnnos.get(i);
                            ann = st.getAnnotationAtTime((record.getBeginTime() +
                                    record.getEndTime()) / 2);

                            if (ann != null) {
                                st.removeAnnotation(ann);
                            }
                        }

                        transcription.setNotifying(true);
                    }
                }
            }
        }
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are correct.<br>
     * If arg[1] is null it is assumed that the annotations have to be
     * inserted on the source tier.
     *
     * @param receiver the TranscriptionImpl
     * @param arguments the arguments: <ul><li>arg[0] = the selected tier
     *        (String)</li> <li>arg[1] = the new tier name. can be null
     *        (String)     </li> <li>arg[2] = the value for the new
     *        annotations, can be null(String)</li> <li>arg[3] = the time
     *        format in case the duration should be the value (can be null, if
     *        not null the duration should be used) (String) </li> <li>arg[4]
     *        = the total media duration (Long)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        transcription = (TranscriptionImpl) receiver;
        sourceTier = (String) arguments[0];
        destTierName = (String) arguments[1];
        annValue = (String) arguments[2];
        timeFormat = (String) arguments[3];
        mediaDuration = (Long) arguments[4];

        // to be sure
        if (timeFormat != null) {
            annValue = null;
        }

        createdAnnos = new ArrayList<AnnotationDataRecord>();
        createAnnotations();
    }

    /**
     * Returns the name of the command.
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }

    /**
     * Iterates over the annotations of the source tier and creates annotations
     * for the gaps.
     */
    private void createAnnotations() {
        TierImpl tier = (TierImpl) transcription.getTierWithId(sourceTier);

        if (tier == null) {
            ClientLogger.LOG.severe("The source tier was not found");

            // message
            return;
        }

        TierImpl dTier = null;

        if (destTierName == null) {
            dTier = tier;
        } else {
            if (transcription.getTierWithId(destTierName) != null) {
                ClientLogger.LOG.severe("The destination tier already exists");

                // message
                return;
            }

            LinguisticType lType = tier.getLinguisticType();

            if (lType.getConstraints() != null) {
                // the source tier should be a toplevel tier
                ClientLogger.LOG.severe("The source tier is not a root tier.");

                // message
                return;
            }

            dTier = new TierImpl(null, destTierName, tier.getParticipant(),
                    transcription, lType);
            dTier.setAnnotator(tier.getAnnotator());
            dTier.setDefaultLocale(tier.getDefaultLocale());
            transcription.addTier(dTier);
        }

        int curPropMode = transcription.getTimeChangePropagationMode();
        transcription.setNotifying(false);

        if (curPropMode != Transcription.NORMAL) {
            transcription.setTimeChangePropagationMode(Transcription.NORMAL);
        }

        List annos = tier.getAnnotations();
        int numCreated = 0;
        AlignableAnnotation aa = null;

        if (annos.size() == 0) {
            aa = (AlignableAnnotation) dTier.createAnnotation(0, mediaDuration);

            if (aa != null) {
                numCreated++;

                if (timeFormat != null) {
                    aa.setValue(getTimeString(mediaDuration));
                } else if (annValue != null) {
                    aa.setValue(annValue);
                }

                createdAnnos.add(new AnnotationDataRecord(aa));
            } else {
                ClientLogger.LOG.warning("Annotation could not be created " +
                    0 + "-" + mediaDuration);
            }
        } else {
            long curAdvance = 0;
            long bt;
            long et;
            AbstractAnnotation ann = null;

            for (int i = 0; i < annos.size(); i++) {
                ann = (AbstractAnnotation) annos.get(i);
                bt = ann.getBeginTimeBoundary();
                et = ann.getEndTimeBoundary();

                if (bt > curAdvance) { // a gap
                    aa = (AlignableAnnotation) dTier.createAnnotation(curAdvance,
                            bt);

                    if (aa != null) {
                        numCreated++;

                        if (timeFormat != null) {
                            aa.setValue(getTimeString(bt - curAdvance));
                        } else if (annValue != null) {
                            aa.setValue(annValue);
                        }

                        createdAnnos.add(new AnnotationDataRecord(aa));
                    } else {
                        ClientLogger.LOG.warning(
                            "Annotation could not be created " + curAdvance +
                            "-" + bt);
                    }
                }

                curAdvance = et;
            }

            // check after last annotation
            if (curAdvance < mediaDuration) {
                aa = (AlignableAnnotation) dTier.createAnnotation(curAdvance,
                        mediaDuration);

                if (aa != null) {
                    numCreated++;

                    if (timeFormat != null) {
                        aa.setValue(getTimeString(mediaDuration - curAdvance));
                    } else if (annValue != null) {
                        aa.setValue(annValue);
                    }

                    createdAnnos.add(new AnnotationDataRecord(aa));
                } else {
                    ClientLogger.LOG.warning("Annotation could not be created " +
                        curAdvance + "-" + mediaDuration);
                }
            }
        }

        ClientLogger.LOG.info("Number of annotations created: " + numCreated);

        if (transcription.getTimeChangePropagationMode() != curPropMode) {
            transcription.setTimeChangePropagationMode(curPropMode);
        }

        transcription.setNotifying(true);
    }

    /**
     * COnverts a numeric time value to a (formatted) string.
     *
     * @param time
     *
     * @return a string representation
     */
    private String getTimeString(long time) {
        if (Constants.HHMMSSMS_STRING.equals(timeFormat)) {
            return TimeFormatter.toString(time);
        } else if (Constants.SSMS_STRING.equals(timeFormat)) {
            return TimeFormatter.toSSMSString(time);
        } else {
            return String.valueOf(time);
        }
    }
}
