/*
 * File:     CopyAnnotationCommand.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.util.AnnotationDataRecord;
import mpi.eudico.client.annotator.util.ClientLogger;
import mpi.eudico.client.annotator.util.TransferableAnnotation;

import mpi.eudico.server.corpora.clom.Annotation;

import java.awt.Toolkit;


/**
 * A Command to copy an annotation (i.e. a transferable AnnotationDataRecord) to the System's
 * Clipboard.
 */
public class CopyAnnotationCommand implements Command, ClientLogger {
    private String commandName;

    /**
     * Creates a new CopyAnnotationCommand instance
     *
     * @param name the name of the command
     */
    public CopyAnnotationCommand(String name) {
        commandName = name;
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver null
     * @param arguments the arguments:  <ul><li>arg[0] = the (active) annotation
     *        (Annotation)</li></ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        if (arguments[0] instanceof Annotation) {
            AnnotationDataRecord record = new AnnotationDataRecord((Annotation) arguments[0]);
            TransferableAnnotation ta = new TransferableAnnotation(record);

            if (canAccessSystemClipboard()) {
                Toolkit.getDefaultToolkit().getSystemClipboard().setContents(ta,
                    ta);
            }
        }
    }

    /**
     * Returns the name of the command.
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }

    /**
     * Performs a check on the accessibility of the system clipboard.
     *
     * @return true if the system clipboard is accessible, false otherwise
     */
    protected boolean canAccessSystemClipboard() {
        if (System.getSecurityManager() != null) {
            try {
                System.getSecurityManager().checkSystemClipboardAccess();

                return true;
            } catch (SecurityException se) {
                LOG.warning("Cannot copy, cannot access the clipboard.");

                return false;
            }
        }

        return true;
    }
}
