/*
 * File:     DeleteAnnotationCommand.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ViewerManager2;

import mpi.eudico.client.annotator.util.AnnotationDataRecord;
import mpi.eudico.client.annotator.util.AnnotationRecreator;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.AbstractAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import java.awt.Cursor;

import javax.swing.tree.DefaultMutableTreeNode;


/**
 * A command for deleting an annotation.
 *
 * @author Han Sloetjes
 */
public class DeleteAnnotationCommand implements UndoableCommand {
    private String commandName;
    private AbstractAnnotation annotation;
    private TierImpl tier;
    private DefaultMutableTreeNode annRootNode;
    private Transcription transcription;

    /**
     * Creates a new DeleteAnnotationCommand instance.
     *
     * @param name the name of the command
     */
    public DeleteAnnotationCommand(String name) {
        commandName = name;
    }

    /**
     * The undo action.
     */
    public void undo() {
        if ((tier != null) && (annRootNode != null)) {
            int curPropMode = 0;

            curPropMode = transcription.getTimeChangePropagationMode();

            if (curPropMode != Transcription.NORMAL) {
                transcription.setTimeChangePropagationMode(Transcription.NORMAL);
            }

            setWaitCursor(true);

            AnnotationRecreator.createAnnotationFromTree((TranscriptionImpl) tier.getParent(),
                annRootNode, true);

            setWaitCursor(false);

            // restore the time propagation mode
            transcription.setTimeChangePropagationMode(curPropMode);
        }
    }

    /**
     * The redo action.
     */
    public void redo() {
        if ((tier != null) && (annRootNode != null)) {
            setWaitCursor(true);

            AnnotationDataRecord annRecord = (AnnotationDataRecord) annRootNode.getUserObject();
            long begin = annRecord.getBeginTime();
            long end = annRecord.getEndTime();
            long mid = (begin + end) / 2;
            Annotation aa = tier.getAnnotationAtTime(mid);

            if (aa != null) {
                tier.removeAnnotation(aa);
            }

            setWaitCursor(false);
        }
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver the TierImpl
     * @param arguments the arguments:  <ul><li>arg[0] = the Viewer Manager
     *        (ViewerManager2)</li> <li>arg[1] = the annotation
     *        (Annotation)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        ViewerManager2 vm = (ViewerManager2) arguments[0];
        transcription = vm.getTranscription();
        tier = (TierImpl) receiver;

        setWaitCursor(true);

        annotation = (AbstractAnnotation) arguments[1];

        if (vm.getActiveAnnotation().getAnnotation() == annotation) {
            vm.getActiveAnnotation().setAnnotation(null);
        }

        annRootNode = AnnotationRecreator.createTreeForAnnotation(annotation);

        tier.removeAnnotation(annotation);

        setWaitCursor(false);
    }

    /**
     * Returns the name of the command.
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }

    /**
     * Changes the cursor to either a 'busy' cursor or the default cursor.
     *
     * @param showWaitCursor when <code>true</code> show the 'busy' cursor
     */
    private void setWaitCursor(boolean showWaitCursor) {
        if (showWaitCursor) {
            ELANCommandFactory.getRootFrame(transcription).getRootPane()
                              .setCursor(Cursor.getPredefinedCursor(
                    Cursor.WAIT_CURSOR));
        } else {
            ELANCommandFactory.getRootFrame(transcription).getRootPane()
                              .setCursor(Cursor.getDefaultCursor());
        }
    }
}
