/*
 * File:     DuplicateAnnotationCA.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ActiveAnnotationListener;
import mpi.eudico.client.annotator.ViewerManager2;

import mpi.eudico.server.corpora.clom.Annotation;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.type.Constraint;

import java.awt.event.ActionEvent;


/**
 * A command action for duplicating (copy and paste in a single operation) an annotation.
 * Based on the active annotation and the active tier in TimeLineViewer or InterlinearViewer.
 *
 * @author Han Sloetjes
 */
public class DuplicateAnnotationCA extends CommandAction
    implements ActiveAnnotationListener {
    /**
     * Creates a new DuplicateAnnotationCA instance
     *
     * @param viewerManager the Viewer Manager
     */
    public DuplicateAnnotationCA(ViewerManager2 viewerManager) {
        super(viewerManager, ELANCommandFactory.DUPLICATE_ANNOTATION);

        viewerManager.connectListener(this);
    }

    /**
     * @see mpi.eudico.client.annotator.commands.CommandAction#newCommand()
     */
    protected void newCommand() {
        command = ELANCommandFactory.createCommand(vm.getTranscription(),
                ELANCommandFactory.DUPLICATE_ANNOTATION);
    }

    /**
     * The receiver of this CommandAction is the viewer manager.
     *
     * @return the viewer manager
     */
    protected Object getReceiver() {
        return vm;
    }

    /**
     * The active tier and the active annotation.
     *
     * @return an Object array size = 2
     */
    protected Object[] getArguments() {
        return new Object[] {
            vm.getMultiTierControlPanel().getActiveTier(),
            vm.getActiveAnnotation().getAnnotation()
        };
    }

    /**
     * Override ationPerformed to perform some initial checks before calling super.actionPerformed,
     * where a new Command is created etc.
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent event) {
        if (vm.getActiveAnnotation().getAnnotation() == null) {
            return;
        }

        if ((vm.getMultiTierControlPanel() == null) ||
                (vm.getMultiTierControlPanel().getActiveTier() == null) ||
                (vm.getMultiTierControlPanel().getActiveTier() == vm.getActiveAnnotation()
                                                                        .getAnnotation()
                                                                        .getTier())) {
            return;
        }

        TierImpl t = (TierImpl) vm.getMultiTierControlPanel().getActiveTier();
        Constraint c = t.getLinguisticType().getConstraints();

        if ((c != null) &&
                ((c.getStereoType() == Constraint.SYMBOLIC_ASSOCIATION) ||
                (c.getStereoType() == Constraint.SYMBOLIC_SUBDIVISION))) {
            long mid = (vm.getActiveAnnotation().getAnnotation()
                          .getBeginTimeBoundary() +
                vm.getActiveAnnotation().getAnnotation().getEndTimeBoundary()) / 2;

            if (t.getAnnotationAtTime(mid) != null) {
                return;
            } else {
                TierImpl par = (TierImpl) t.getParentTier();

                if (par.getOverlappingAnnotations(vm.getActiveAnnotation()
                                                        .getAnnotation()
                                                        .getBeginTimeBoundary(),
                            vm.getActiveAnnotation().getAnnotation()
                                  .getEndTimeBoundary()).size() == 0) {
                    return;
                }
            }
        } else if ((c != null) &&
                ((c.getStereoType() == Constraint.TIME_SUBDIVISION) ||
                (c.getStereoType() == Constraint.INCLUDED_IN))) {
            TierImpl par = (TierImpl) t.getParentTier();
            Annotation ann1 = par.getAnnotationAtTime(vm.getActiveAnnotation()
                                                        .getAnnotation()
                                                        .getBeginTimeBoundary());
            Annotation ann2 = par.getAnnotationAtTime(vm.getActiveAnnotation()
                                                        .getAnnotation()
                                                        .getEndTimeBoundary());

            if ((ann1 != null) && (ann2 != null) && (ann1 != ann2)) {
                //return; // don't choose?
            } else if ((ann1 == null) && (ann2 == null) &&
                    (par.getOverlappingAnnotations(vm.getActiveAnnotation()
                                                         .getAnnotation()
                                                         .getBeginTimeBoundary(),
                        vm.getActiveAnnotation().getAnnotation()
                              .getEndTimeBoundary()).size() == 0)) {
                return;
            }
        }

        // let a command be created
        super.actionPerformed(event);
    }

    /**
     * @see mpi.eudico.client.annotator.ActiveAnnotationListener#updateActiveAnnotation()
     */
    public void updateActiveAnnotation() {
        if (vm.getActiveAnnotation().getAnnotation() != null) {
            setEnabled(true);
        } else {
            setEnabled(false);
        }
    }
}
