/*
 * File:     ImportPrefsCommand.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.util.ElanFileFilter;
import mpi.eudico.client.annotator.util.FileExtension;

import mpi.eudico.server.corpora.clom.Transcription;

import java.io.File;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;


/**
 * Shows an open/import dialog and passes the selected file to the Preferences
 * import method.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class ImportPrefsCommand implements Command {
    private String commandName;
    private Transcription transcription;

    /**
     * Creates a new ImportPrefsCommand instance
     *
     * @param name name of the command
     */
    public ImportPrefsCommand(String name) {
        commandName = name;
    }

    /**
     * Shows the dialog and imports the Preferences.
     *
     * @param receiver the Transcription/document to apply the preferences to
     * @param arguments null
     */
    public void execute(Object receiver, Object[] arguments) {
        transcription = (Transcription) receiver;

        if (transcription == null) {
            return;
        }

        String filePath = promptForImportFile();

        if (filePath == null) {
            return;
        }

        Preferences.importPreferences(transcription, filePath);
    }

    /**
     * Returns the name.
     *
     * @return the name!
     */
    public String getName() {
        return commandName;
    }

    /**
     * Prompts the user to browse to an ELAN xml preferences file, checks a little.
     *
     * @return the file path
     */
    private String promptForImportFile() {
        String prefDir = (String) Preferences.get("LastUsedExportDir", null);

        if (prefDir == null) {
            prefDir = System.getProperty("user.dir");
        }

        JFileChooser chooser = new JFileChooser();

        chooser.setApproveButtonText(ElanLocale.getString(
                "ImportDialog.Approve"));
        chooser.setCurrentDirectory(new File(prefDir));
        chooser.setDialogTitle(ElanLocale.getString("ImportDialog.Title.Select"));

        File prefFile = null;
        FileFilter filter = ElanFileFilter.createFileFilter(ElanFileFilter.ELAN_PREFS_TYPE);
        chooser.setAcceptAllFileFilterUsed(false);
        chooser.setFileFilter(filter);

        if (chooser.showOpenDialog(ELANCommandFactory.getRootFrame(
                        transcription)) == JFileChooser.APPROVE_OPTION) {
            File curDir = chooser.getCurrentDirectory();

            if (curDir != null) {
                Preferences.set("LastUsedExportDir", curDir.getAbsolutePath(),
                    null);
            }

            prefFile = chooser.getSelectedFile();

            if (prefFile != null) {
                String name = prefFile.getAbsolutePath();

                // check
                if (!isValidFile(name)) {
                    // message box
                    JOptionPane.showMessageDialog(ELANCommandFactory.getRootFrame(
                            transcription),
                        ElanLocale.getString("Message.InvalidFile"),
                        ElanLocale.getString("Message.Error"),
                        JOptionPane.ERROR_MESSAGE);

                    return null;
                }

                return name;
            }
        }

        return null;
    }

    private boolean isValidFile(String name) {
        if (name == null) {
            return false;
        }

        File f = new File(name);

        if (!f.exists()) {
            return false;
        }

        String lowerPathName = name.toLowerCase();

        String[] exts = FileExtension.ELAN_XML_PREFS_EXT;

        for (int i = 0; i < exts.length; i++) {
            if (lowerPathName.endsWith("." + exts[i])) {
                return true;
            }
        }

        return false;
    }
}
