/*
 * File:     MergeAnnotationWithNextCA.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ActiveAnnotationListener;
import mpi.eudico.client.annotator.ViewerManager2;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;


/**
 * An action to merge two adjacent annotations on a toplevel tier.
 * The first annotation is the active annotation.
 *
 * @author Han Sloetjes
 * @version 1.0
  */
public class MergeAnnotationWithNextCA extends CommandAction
    implements ActiveAnnotationListener {
    /**
     * Creates a new MergeAnnotationWithNextCA instance
     *
     * @param viewerManager the viewer manager
     */
    public MergeAnnotationWithNextCA(ViewerManager2 viewerManager) {
        super(viewerManager, ELANCommandFactory.MERGE_ANNOTATION_WN);

        viewerManager.connectListener(this);
        setEnabled(false);
    }

    /**
     * Creates a new merge annotations command.
     */
    protected void newCommand() {
        command = ELANCommandFactory.createCommand(vm.getTranscription(),
                ELANCommandFactory.MERGE_ANNOTATION_WN);
    }

    /**
     * Returns the receiver of the command, the transcription.
     *
     * @return the receiver of the command
     */
    protected Object getReceiver() {
        return vm.getTranscription();
    }

    /**
     * Returns the active annotation (in an object array of size 1).
     *
     * @return the active annotation
     */
    protected Object[] getArguments() {
        return new Object[] { vm.getActiveAnnotation().getAnnotation() };
    }

    /**
     * On a change of ActiveAnnotation perform a check to determine whether
     * this action should be enabled or disabled.<br>
     * If the annotation is on a top level tier the action is enabled
     *
     * @see ActiveAnnotationListener#updateActiveAnnotation()
     */
    public void updateActiveAnnotation() {
        if (vm.getActiveAnnotation().getAnnotation() == null) {
            setEnabled(false);
        } else {
            // only for top level tiers for now
            // could check if there is a next annotation
            setEnabled(!((TierImpl) vm.getActiveAnnotation().getAnnotation()
                                      .getTier()).hasParentTier());
        }
    }
}
