/*
 * File:     ModifyAnnotationCommand.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.util.AnnotationDataRecord;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.AbstractAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.ExternalReferenceImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;


/**
 * A command for modifying an annotation value.
 *
 * @author Han Sloetjes
 */
public class ModifyAnnotationCommand implements UndoableCommand {
    private String commandName;
    private Transcription transcription;
    private AnnotationDataRecord annotationRecord;
    private String oldValue;
    private String newValue;
    private Object oldExtRef;
    private Object newExtRef;

    /**
     * Creates a new ModifyAnnotationCommand instance
     *
     * @param name the name of the command
     */
    public ModifyAnnotationCommand(String name) {
        commandName = name;
    }

    /**
     * The undo action. We can not just use an object reference to the
     * annotation because the annotation might have been deleted and recreated
     * between the  calls to the execute and undo methods. The reference would
     * then be invalid.
     */
    public void undo() {
        if ((annotationRecord != null) && (transcription != null)) {
            TierImpl tier = (TierImpl) transcription.getTierWithId(annotationRecord.getTierName());
            Annotation annotation = tier.getAnnotationAtTime(annotationRecord.getBeginTime());

            // doublecheck to see if we have the right annotation
            if ((annotation != null) &&
                    (annotation.getEndTimeBoundary() == annotationRecord.getEndTime())) {
                annotation.setValue(oldValue);
                ((AbstractAnnotation) annotation).setExtRef(oldExtRef);
            }
        }
    }

    /**
     * The redo action.
     *
     * @see #undo
     */
    public void redo() {
        if ((annotationRecord != null) && (transcription != null)) {
            TierImpl tier = (TierImpl) transcription.getTierWithId(annotationRecord.getTierName());
            Annotation annotation = tier.getAnnotationAtTime(annotationRecord.getBeginTime());

            // doublecheck to see if we have the right annotation
            if ((annotation != null) &&
                    (annotation.getEndTimeBoundary() == annotationRecord.getEndTime())) {
                annotation.setValue(newValue);
                ((AbstractAnnotation) annotation).setExtRef(newExtRef);
            }
        }
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver the Annotation (AbstractAnnotation)
     * @param arguments the arguments:  <ul><li>arg[0] = the old value of the
     *        annotation (String)</li> <li>arg[1] = the new value of the
     *        annotation (String)</li> <li>arg[2] = the (new) external reference object
     *        that can be set by a CV entry</li></ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        AbstractAnnotation annotation = (AbstractAnnotation) receiver;

        if (annotation != null) {
            transcription = (Transcription) annotation.getTier().getParent();
        }

        annotationRecord = new AnnotationDataRecord(annotation);

        oldValue = (String) arguments[0];
        newValue = (String) arguments[1];
        annotation.setValue(newValue);

        // if an ext ref argument is passed, remove the current ext ref of the annotation 
        // even if the new ext ref is null
        if (arguments.length >= 2) {
            ExternalReferenceImpl eri = (ExternalReferenceImpl) arguments[2]; // can be null
            oldExtRef = annotation.getExtRef();
            newExtRef = eri;

            try {
                if (eri != null) {
                    newExtRef = eri.clone();
                }
            } catch (CloneNotSupportedException cnse) {
            }

            annotation.setExtRef(newExtRef);
        }
    }

    /**
     * Returns the name of the command.
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }
}
