/*
 * File:     ModifyAnnotationDatCatCommand.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.util.AnnotationDataRecord;

import mpi.eudico.server.corpora.clom.ExternalReference;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.AbstractAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.ExternalReferenceImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;


/**
 * Modifies the reference to an ISO data category. Note: does not yet handle
 * compound references!
 *
 * @author Han Sloetjes
 */
public class ModifyAnnotationDatCatCommand implements UndoableCommand {
    private String commandName;
    private Transcription transcription;
    private AnnotationDataRecord annotationRecord;
    private String oldValue;
    private String newValue;

    /**
     * Constructor.
     *
     * @param commandName the name of the command
     */
    public ModifyAnnotationDatCatCommand(String commandName) {
        super();
        this.commandName = commandName;
    }

    /**
     * The redo action
     */
    public void redo() {
        if ((annotationRecord != null) && (transcription != null)) {
            TierImpl tier = (TierImpl) transcription.getTierWithId(annotationRecord.getTierName());
            AbstractAnnotation annotation = (AbstractAnnotation) tier.getAnnotationAtTime(annotationRecord.getBeginTime());

            // doublecheck to see if we have the right annotation
            if ((annotation != null) &&
                    (annotation.getEndTimeBoundary() == annotationRecord.getEndTime())) {
                if (newValue == null) {
                    annotation.setExtRef(null);
                } else {
                    annotation.setExtRef(new ExternalReferenceImpl(newValue,
                            ExternalReference.ISO12620_DC_ID));
                }
            }
        }
    }

    /**
     * The undo action
     */
    public void undo() {
        if ((annotationRecord != null) && (transcription != null)) {
            TierImpl tier = (TierImpl) transcription.getTierWithId(annotationRecord.getTierName());
            AbstractAnnotation annotation = (AbstractAnnotation) tier.getAnnotationAtTime(annotationRecord.getBeginTime());

            // doublecheck to see if we have the right annotation
            if ((annotation != null) &&
                    (annotation.getEndTimeBoundary() == annotationRecord.getEndTime())) {
                if (oldValue == null) {
                    annotation.setExtRef(null);
                } else {
                    annotation.setExtRef(new ExternalReferenceImpl(oldValue,
                            ExternalReference.ISO12620_DC_ID));
                }
            }
        }
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver the Annotation (AbstractAnnotation)
     * @param arguments the arguments:  <ul><li>arg[0] = the new value of the
     *        annotation data category reference (String)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        AbstractAnnotation annotation = (AbstractAnnotation) receiver;

        if (annotation != null) {
            transcription = (Transcription) annotation.getTier().getParent();
        }

        annotationRecord = new AnnotationDataRecord(annotation);

        newValue = (String) arguments[0];

        Object extRef = annotation.getExtRef();

        if (extRef instanceof String) {
            oldValue = (String) extRef;
        } else if (extRef instanceof ExternalReference) {
            ExternalReference ef = (ExternalReference) extRef;

            if (ef.getReferenceType() == ExternalReference.ISO12620_DC_ID) {
                oldValue = ef.getValue();
            }
        }

        if (newValue != null) {
            annotation.setExtRef(new ExternalReferenceImpl(newValue,
                    ExternalReference.ISO12620_DC_ID));
        } else {
            annotation.setExtRef(null);
        }
    }

    /**
     * Returns the name
     *
     * @return the name
     */
    public String getName() {
        return commandName;
    }
}
