/*
 * File:     ShiftAllAnnotationsCommand.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.util.ClientLogger;

import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;


/**
 * SHifts all time slots in a Transcription.
 *
 * @author Han Sloetjes
 */
public class ShiftAllAnnotationsCommand implements UndoableCommand {
    private String commandName;

    // receiver; the transcription 
    private TranscriptionImpl transcription;
    private long shiftValue = 0;

    /**
     * Creates a new shift all command.
     *
     * @param name the name of the command
     */
    public ShiftAllAnnotationsCommand(String name) {
        commandName = name;
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver the Transcription
     * @param arguments the arguments: <ul><li>arg[0] = the amount of
     *        milliseconds that should be added to all time slot values
     *        (Long)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        transcription = (TranscriptionImpl) receiver;

        if ((arguments != null) && (arguments.length >= 1)) {
            shiftValue = ((Long) arguments[0]).longValue();
        }

        shiftAll(shiftValue);
    }

    /**
     * Returns the name of the command.
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }

    /**
     * The undo action. All Time Slots are shifted back the same amount.
     */
    public void undo() {
        shiftAll(-shiftValue);
    }

    /**
     * The redo action. All Time Slots are shifted again the same amount.
     */
    public void redo() {
        shiftAll(shiftValue);
    }

    /**
     * Issues the actual shifting.
     *
     * @param value the shift value in ms
     */
    private void shiftAll(long value) {
        if (transcription != null) {
            try {
                transcription.shiftAllAnnotations(value);
            } catch (IllegalArgumentException iae) {
                ClientLogger.LOG.warning(iae.getMessage());
            }
        }
    }
}
