/*
 * File:     ShiftAnnotationsCommand.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import mpi.eudico.server.corpora.event.IllegalEditException;

import javax.swing.JOptionPane;


/**
 * Command that shifts all annotations on a specified root tier that are
 * completely inside a specified time interval. The user has been prompted to
 * specify the amount of ms to shift the annotations, to the left or to the
 * right. All depending annotations are shifted as well.
 *
 * @author Han Sloetjes
 * @version 1.0, Nov 2008
 */
public class ShiftAnnotationsCommand implements UndoableCommand {
    /** Holds value of property DOCUMENT ME! */
    String commandName;

    // receiver; the transcription 

    /** Holds value of property DOCUMENT ME! */
    TranscriptionImpl transcription;
    private TierImpl tier;

    /** Holds value of property DOCUMENT ME! */
    Long bt;

    /** Holds value of property DOCUMENT ME! */
    Long et;

    /** Holds value of property DOCUMENT ME! */
    Long shiftValue;

    /**
     * Creates a new ShiftAnnotationsCommand instance
     *
     * @param name the name of the command
     */
    public ShiftAnnotationsCommand(String name) {
        commandName = name;
    }

    /**
     * Shift the annotations back.
     */
    public void redo() {
        shift(tier, shiftValue, bt, et);
    }

    /**
     * Shift the annotations again.
     */
    public void undo() {
        shift(tier, -shiftValue, ((bt == 0) ? bt : (bt + shiftValue)),
            ((et == Long.MAX_VALUE) ? et : (et + shiftValue)));
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver the Transcription
     * @param arguments the arguments: <ul><li>arg[0] = the (active) tier
     *        (TierImpl)</li> <li>arg[1] = the begin of the time interval the
     *        annotations in which are  to be shifted (Long)</li> <li>arg[2] =
     *        the end time of that interval (Long)</li> <li>arg[3] = the shift
     *        value (Long)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        transcription = (TranscriptionImpl) receiver;

        if (arguments != null) {
            tier = (TierImpl) arguments[0];
            bt = (Long) arguments[1];
            et = (Long) arguments[2];
            shiftValue = (Long) arguments[3];

            shift(tier, shiftValue, bt, et);
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param tier DOCUMENT ME!
     * @param shiftValue DOCUMENT ME!
     * @param bt DOCUMENT ME!
     * @param et DOCUMENT ME!
     */
    void shift(TierImpl tier, long shiftValue, long bt, long et) {
        if (tier != null) { //trust that the rest isn't null either

            try {
                tier.shiftAnnotations(shiftValue, bt, et);
            } catch (IllegalArgumentException iae) {
                JOptionPane.showMessageDialog(ELANCommandFactory.getRootFrame(
                        transcription),
                    ElanLocale.getString("ShiftAllDialog.Warn5") + " " +
                    iae.getMessage(), ElanLocale.getString("Message.Warning"),
                    JOptionPane.WARNING_MESSAGE);
                iae.printStackTrace();
            } catch (IllegalEditException iee) {
                JOptionPane.showMessageDialog(ELANCommandFactory.getRootFrame(
                        transcription),
                    ElanLocale.getString("ShiftAllDialog.Warn5") + " " +
                    iee.getMessage(), ElanLocale.getString("Message.Warning"),
                    JOptionPane.WARNING_MESSAGE);
                iee.printStackTrace();
            }
        }
    }

    /**
     * Returns the name of the command.
     *
     * @return the name
     */
    public String getName() {
        return commandName;
    }
}
