package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.Constants;
import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.annotator.prefs.PreferencesReader;
import mpi.eudico.client.annotator.prefs.PreferencesWriter;
import mpi.eudico.client.annotator.util.ClientLogger;
import mpi.eudico.client.annotator.util.SystemReporting;

import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JOptionPane;
import javax.swing.KeyStroke;


/**
 * A singleton class for handling keyboard shortcuts and their mapping to
 * actions. Any new action that potentially can be triggered via a keyboard
 * shortcut should be added to 1 of the categories in the private method
 * fillActionsMap().
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class ShortcutsUtil {
    private static ShortcutsUtil shortcutsUtil;

    /** annotation editing category */
    public static final String ANN_EDIT_CAT = "Frame.ShortcutFrame.Sub.AnnotationEdit";

    /** annotation navigation category */
    public static final String ANN_NAVIGATION_CAT = "Frame.ShortcutFrame.Sub.AnnotationNavigation";

    /** tier and type category */
    public static final String TIER_TYPE_CAT = "Frame.ShortcutFrame.Sub.TierType";

    /** selection category */
    public static final String SELECTION_CAT = "Frame.ShortcutFrame.Sub.Selection";

    /** media navigation category */
    public static final String MEDIA_CAT = "Frame.ShortcutFrame.Sub.MediaNavigation";

    /** document and file i/o category */
    public static final String DOCUMENT_CAT = "Frame.ShortcutFrame.Sub.Document";

    /** miscellaneous category */
    public static final String MISC_CAT = "Frame.ShortcutFrame.Sub.Misc";
    private Map<String, List<String>> shortcuttableActions;
    private Map<String, KeyStroke> shortcutKeyStrokes;

    /**
     * Creates the single ShortcutsUtil instance
     */
    private ShortcutsUtil() {
        shortcuttableActions = new LinkedHashMap<String, List<String>>(8);
        shortcutKeyStrokes = new LinkedHashMap<String, KeyStroke>(80);
        fillActionsMap();
        fillShortcutMap();
    }

    /**
     * Returns the single instance of this class
     *
     * @return the single instance of this class
     */
    public static ShortcutsUtil getInstance() {
        if (shortcutsUtil == null) {
            shortcutsUtil = new ShortcutsUtil();
        }

        return shortcutsUtil;
    }

    /**
     * Adds the constant identifiers of actions that potentially can be invoked
     * by a keyboard shortcut, to one of the categories of actions. Any new
     * action that is created has to be added here as well, if it has a
     * default shortcut it should be added to {@link #loadDefaultShortcuts()}.
     */
    private void fillActionsMap() {
        List<String> editActions = new ArrayList<String>(34);
        // add all actions that potentially can be invoked by means of
        // a keyboard shortcut and that belongs to the "annotation editing" category
        editActions.add(ELANCommandFactory.NEW_ANNOTATION);
        editActions.add(ELANCommandFactory.NEW_ANNOTATION_BEFORE);
        editActions.add(ELANCommandFactory.NEW_ANNOTATION_AFTER);
        editActions.add(ELANCommandFactory.KEY_CREATE_ANNOTATION);
        editActions.add(ELANCommandFactory.COPY_ANNOTATION);
        editActions.add(ELANCommandFactory.COPY_ANNOTATION_TREE);
        editActions.add(ELANCommandFactory.PASTE_ANNOTATION);
        editActions.add(ELANCommandFactory.PASTE_ANNOTATION_HERE);
        editActions.add(ELANCommandFactory.PASTE_ANNOTATION_TREE);
        editActions.add(ELANCommandFactory.PASTE_ANNOTATION_TREE_HERE);
        editActions.add(ELANCommandFactory.DUPLICATE_ANNOTATION);
        editActions.add(ELANCommandFactory.COPY_TO_NEXT_ANNOTATION);
        editActions.add(ELANCommandFactory.MODIFY_ANNOTATION);
        editActions.add(ELANCommandFactory.MODIFY_ANNOTATION_TIME);
        editActions.add(ELANCommandFactory.MODIFY_ANNOTATION_DC);
        editActions.add(ELANCommandFactory.SHIFT_ACTIVE_ANNOTATION);
        editActions.add(ELANCommandFactory.REMOVE_ANNOTATION_VALUE);
        editActions.add(ELANCommandFactory.DELETE_ANNOTATION);
        editActions.add(ELANCommandFactory.MERGE_ANNOTATION_WN);
        editActions.add(ELANCommandFactory.REGULAR_ANNOTATION_DLG);
        editActions.add(ELANCommandFactory.DELETE_ANNOS_IN_SELECTION);
        editActions.add(ELANCommandFactory.DELETE_ANNOS_LEFT_OF);
        editActions.add(ELANCommandFactory.DELETE_ANNOS_RIGHT_OF);
        editActions.add(ELANCommandFactory.DELETE_ALL_ANNOS_LEFT_OF);
        editActions.add(ELANCommandFactory.DELETE_ALL_ANNOS_RIGHT_OF);
        editActions.add(ELANCommandFactory.SHIFT_ACTIVE_ANNOTATION);
        editActions.add(ELANCommandFactory.SHIFT_ALL_ANNOTATIONS);
        editActions.add(ELANCommandFactory.SHIFT_ALL_ANNOS_LEFT_OF);
        editActions.add(ELANCommandFactory.SHIFT_ALL_ANNOS_RIGHT_OF);
        editActions.add(ELANCommandFactory.SHIFT_ANNOS_IN_SELECTION);
        editActions.add(ELANCommandFactory.SHIFT_ANNOS_LEFT_OF);
        editActions.add(ELANCommandFactory.SHIFT_ANNOS_RIGHT_OF);

        shortcuttableActions.put(ANN_EDIT_CAT, editActions);

        List<String> navActions = new ArrayList<String>(8);

        navActions.add(ELANCommandFactory.PREVIOUS_ANNOTATION);
        navActions.add(ELANCommandFactory.PREVIOUS_ANNOTATION_EDIT);
        navActions.add(ELANCommandFactory.NEXT_ANNOTATION);
        navActions.add(ELANCommandFactory.NEXT_ANNOTATION_EDIT);
        navActions.add(ELANCommandFactory.ANNOTATION_UP);
        navActions.add(ELANCommandFactory.ANNOTATION_DOWN);

        shortcuttableActions.put(ANN_NAVIGATION_CAT, navActions);

        List<String> tierActions = new ArrayList<String>(22);

        tierActions.add(ELANCommandFactory.ADD_TIER);
        tierActions.add(ELANCommandFactory.CHANGE_TIER);
        tierActions.add(ELANCommandFactory.DELETE_TIER);
        tierActions.add(ELANCommandFactory.REPARENT_TIER);
        tierActions.add(ELANCommandFactory.TOKENIZE_DLG);
        tierActions.add(ELANCommandFactory.FILTER_TIER);
        tierActions.add(ELANCommandFactory.COPY_TIER);
        tierActions.add(ELANCommandFactory.ANN_FROM_OVERLAP);
        tierActions.add(ELANCommandFactory.MERGE_TIERS);
        tierActions.add(ELANCommandFactory.ANN_FROM_GAPS);
        tierActions.add(ELANCommandFactory.CHANGE_CASE);
        tierActions.add(ELANCommandFactory.SEGMENTATION_DLG);
        tierActions.add(ELANCommandFactory.COMPARE_ANNOTATORS_DLG);
        tierActions.add(ELANCommandFactory.PREVIOUS_ACTIVE_TIER);
        tierActions.add(ELANCommandFactory.NEXT_ACTIVE_TIER);
        tierActions.add(ELANCommandFactory.LABEL_AND_NUMBER);
        tierActions.add(ELANCommandFactory.TIER_DEPENDENCIES);
        tierActions.add(ELANCommandFactory.ADD_TYPE);
        tierActions.add(ELANCommandFactory.CHANGE_TYPE);
        tierActions.add(ELANCommandFactory.DELETE_TYPE);

        shortcuttableActions.put(TIER_TYPE_CAT, tierActions);

        List<String> selActions = new ArrayList<String>(8);

        selActions.add(ELANCommandFactory.CLEAR_SELECTION);
        selActions.add(ELANCommandFactory.CLEAR_SELECTION_AND_MODE);
        selActions.add(ELANCommandFactory.SELECTION_BOUNDARY);
        selActions.add(ELANCommandFactory.SELECTION_MODE);
        selActions.add(ELANCommandFactory.CENTER_SELECTION);

        shortcuttableActions.put(SELECTION_CAT, selActions);

        List<String> medNavActions = new ArrayList<String>(20);

        medNavActions.add(ELANCommandFactory.PLAY_PAUSE);
        medNavActions.add(ELANCommandFactory.PLAY_SELECTION);
        medNavActions.add(ELANCommandFactory.PLAY_AROUND_SELECTION);
        medNavActions.add(ELANCommandFactory.PLAY_AROUND_SELECTION_DLG);
        medNavActions.add(ELANCommandFactory.PIXEL_LEFT);
        medNavActions.add(ELANCommandFactory.PIXEL_RIGHT);
        medNavActions.add(ELANCommandFactory.PREVIOUS_FRAME);
        medNavActions.add(ELANCommandFactory.NEXT_FRAME);
        medNavActions.add(ELANCommandFactory.SECOND_LEFT);
        medNavActions.add(ELANCommandFactory.SECOND_RIGHT);
        medNavActions.add(ELANCommandFactory.PREVIOUS_SCROLLVIEW);
        medNavActions.add(ELANCommandFactory.NEXT_SCROLLVIEW);
        medNavActions.add(ELANCommandFactory.GO_TO_BEGIN);
        medNavActions.add(ELANCommandFactory.GO_TO_END);
        medNavActions.add(ELANCommandFactory.GOTO_DLG);
        medNavActions.add(ELANCommandFactory.LOOP_MODE);
        medNavActions.add(ELANCommandFactory.PLAYBACK_TOGGLE_DLG);

        shortcuttableActions.put(MEDIA_CAT, medNavActions);

        List<String> docActions = new ArrayList<String>(44);

        docActions.add(ELANCommandFactory.NEW_DOC);
        docActions.add(ELANCommandFactory.OPEN_DOC);
        docActions.add(ELANCommandFactory.SAVE);
        docActions.add(ELANCommandFactory.SAVE_AS);
        docActions.add(ELANCommandFactory.SAVE_AS_TEMPLATE);
        docActions.add(ELANCommandFactory.SAVE_SELECTION_AS_EAF);
        docActions.add(ELANCommandFactory.MERGE_TRANSCRIPTIONS);
        docActions.add(ELANCommandFactory.IMPORT_SHOEBOX);
        docActions.add(ELANCommandFactory.IMPORT_TOOLBOX);
        docActions.add(ELANCommandFactory.IMPORT_FLEX);
        docActions.add(ELANCommandFactory.IMPORT_CHAT);
        docActions.add(ELANCommandFactory.IMPORT_TRANS);
        docActions.add(ELANCommandFactory.IMPORT_TAB);
        docActions.add(ELANCommandFactory.IMPORT_PRAAT_GRID);
        docActions.add(ELANCommandFactory.IMPORT_PREFS);
        docActions.add(ELANCommandFactory.IMPORT_TIERS);
        docActions.add(ELANCommandFactory.IMPORT_TYPES);
        docActions.add(ELANCommandFactory.EXPORT_TAB_MULTI);
        docActions.add(ELANCommandFactory.EXPORT_ANNLIST_MULTI);
        docActions.add(ELANCommandFactory.EXPORT_WORDLIST_MULTI);
        docActions.add(ELANCommandFactory.EXPORT_FILMSTRIP);
        docActions.add(ELANCommandFactory.EXPORT_HTML);
        docActions.add(ELANCommandFactory.EXPORT_IMAGE_FROM_WINDOW);
        docActions.add(ELANCommandFactory.EXPORT_INTERLINEAR);
        docActions.add(ELANCommandFactory.EXPORT_MEDIA);
        docActions.add(ELANCommandFactory.EXPORT_PRAAT_GRID);
        docActions.add(ELANCommandFactory.EXPORT_PREFS);
        docActions.add(ELANCommandFactory.EXPORT_QT_SUB);
        docActions.add(ELANCommandFactory.EXPORT_SHOEBOX);
        docActions.add(ELANCommandFactory.EXPORT_SMIL);
        docActions.add(ELANCommandFactory.EXPORT_SUBTITLES);
        docActions.add(ELANCommandFactory.EXPORT_TAB);
        docActions.add(ELANCommandFactory.EXPORT_TIGER);
        docActions.add(ELANCommandFactory.EXPORT_TOOLBOX);
        docActions.add(ELANCommandFactory.EXPORT_TRAD_TRANSCRIPT);
        docActions.add(ELANCommandFactory.EXPORT_WORDS);
        docActions.add(ELANCommandFactory.PRINT);
        docActions.add(ELANCommandFactory.PREVIEW);
        docActions.add(ELANCommandFactory.PAGESETUP);
        docActions.add(ELANCommandFactory.NEXT_WINDOW);
        docActions.add(ELANCommandFactory.PREV_WINDOW);
        docActions.add(ELANCommandFactory.CLOSE);
        docActions.add(ELANCommandFactory.EXIT);

        shortcuttableActions.put(DOCUMENT_CAT, docActions);

        List<String> miscActions = new ArrayList<String>(25);

        miscActions.add(ELANCommandFactory.UNDO);
        miscActions.add(ELANCommandFactory.REDO);
        miscActions.add(ELANCommandFactory.SEARCH_DLG);
        miscActions.add(ELANCommandFactory.SEARCH_MULTIPLE_DLG);
        miscActions.add(ELANCommandFactory.STRUCTURED_SEARCH_MULTIPLE_DLG);
        miscActions.add(ELANCommandFactory.REPLACE_MULTIPLE);
        miscActions.add(ELANCommandFactory.PLAYBACK_RATE_TOGGLE);
        miscActions.add(ELANCommandFactory.PLAYBACK_VOLUME_TOGGLE);
        miscActions.add(ELANCommandFactory.LINKED_FILES_DLG);
        miscActions.add(ELANCommandFactory.EDIT_CV_DLG);
        miscActions.add(ELANCommandFactory.EDIT_PREFS);
        miscActions.add(ELANCommandFactory.EDIT_SHORTCUTS);
        miscActions.add(ELANCommandFactory.SET_AUTHOR);
        miscActions.add(ELANCommandFactory.FONT_BROWSER);
        miscActions.add(ELANCommandFactory.SHORTCUTS);
        miscActions.add(ELANCommandFactory.SPREADSHEET);
        miscActions.add(ELANCommandFactory.STATISTICS);
        miscActions.add(ELANCommandFactory.HELP);
        miscActions.add(ELANCommandFactory.ABOUT);
        // category to be checked
        miscActions.add(ELANCommandFactory.ANNOTATION_MODE);
        miscActions.add(ELANCommandFactory.SYNC_MODE);
        miscActions.add(ELANCommandFactory.BULLDOZER_MODE);
        miscActions.add(ELANCommandFactory.TIMEPROP_NORMAL);
        miscActions.add(ELANCommandFactory.SHIFT_MODE);
        miscActions.add("MultiTierViewer.ShiftToolTip");

        shortcuttableActions.put(MISC_CAT, miscActions);
    }

    /**
     * Read from stored preferences or use defaults.
     */
    private void fillShortcutMap() {
        if (!readCurrentShortcuts()) {
            loadDefaultShortcuts();
        }
    }

    /**
     * Load default shortcuts.
     */
    private void loadDefaultShortcuts() {
        // defaults...
        shortcutKeyStrokes.put(ELANCommandFactory.NEW_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_N, ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.SELECTION_BOUNDARY,
            KeyStroke.getKeyStroke(KeyEvent.VK_SLASH,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.ADD_TYPE,
            KeyStroke.getKeyStroke(KeyEvent.VK_T,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.ADD_TIER,
            KeyStroke.getKeyStroke(KeyEvent.VK_T,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.NEW_ANNOTATION_AFTER,
            KeyStroke.getKeyStroke(KeyEvent.VK_N,
                ActionEvent.SHIFT_MASK + ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.NEW_ANNOTATION_BEFORE,
            KeyStroke.getKeyStroke(KeyEvent.VK_N,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.ANNOTATION_DOWN,
            KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.ANNOTATION_UP,
            KeyStroke.getKeyStroke(KeyEvent.VK_UP, ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.CENTER_SELECTION,
            KeyStroke.getKeyStroke(KeyEvent.VK_A, 
            		Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
            		ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.CLEAR_SELECTION_AND_MODE,
            KeyStroke.getKeyStroke(KeyEvent.VK_Z,
                ActionEvent.CTRL_MASK + ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.CLEAR_SELECTION,
            KeyStroke.getKeyStroke(KeyEvent.VK_C,
                ActionEvent.ALT_MASK + ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.CLOSE,
            KeyStroke.getKeyStroke(KeyEvent.VK_W,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.COPY_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_C,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.COPY_ANNOTATION_TREE,
            KeyStroke.getKeyStroke(KeyEvent.VK_C,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.COPY_TO_NEXT_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_D,
                ActionEvent.CTRL_MASK + ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.DELETE_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_D, ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.DELETE_TIER,
            KeyStroke.getKeyStroke(KeyEvent.VK_T,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.DUPLICATE_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_D,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.EDIT_CV_DLG,
            KeyStroke.getKeyStroke(KeyEvent.VK_C,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.GO_TO_BEGIN,
            KeyStroke.getKeyStroke(KeyEvent.VK_B,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.GOTO_DLG,
            KeyStroke.getKeyStroke(KeyEvent.VK_G,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.GO_TO_END,
            KeyStroke.getKeyStroke(KeyEvent.VK_E,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.KEY_CREATE_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.LINKED_FILES_DLG,
            KeyStroke.getKeyStroke(KeyEvent.VK_L,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.LOOP_MODE,
            KeyStroke.getKeyStroke(KeyEvent.VK_L,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.MODIFY_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_M, ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.MODIFY_ANNOTATION_DC,
            KeyStroke.getKeyStroke(KeyEvent.VK_M,
                ActionEvent.SHIFT_MASK + ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.MODIFY_ANNOTATION_TIME,
            KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.NEW_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_N, ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.NEXT_ACTIVE_TIER,
            KeyStroke.getKeyStroke(KeyEvent.VK_DOWN,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.NEXT_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT, ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.NEXT_ANNOTATION_EDIT,
            KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.ALT_MASK /*+ ActionEvent.SHIFT_MASK*/));
        shortcutKeyStrokes.put(ELANCommandFactory.NEXT_FRAME,
            KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.NEXT_SCROLLVIEW,
            KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_DOWN,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.PAGESETUP,
            KeyStroke.getKeyStroke(KeyEvent.VK_P,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PASTE_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_V,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.PASTE_ANNOTATION_HERE,
            KeyStroke.getKeyStroke(KeyEvent.VK_V,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PASTE_ANNOTATION_TREE,
            KeyStroke.getKeyStroke(KeyEvent.VK_V,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PASTE_ANNOTATION_TREE_HERE,
            KeyStroke.getKeyStroke(KeyEvent.VK_V,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.ALT_MASK + ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PIXEL_LEFT,
            KeyStroke.getKeyStroke(KeyEvent.VK_LEFT,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PIXEL_RIGHT,
            KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PLAY_AROUND_SELECTION,
            KeyStroke.getKeyStroke(KeyEvent.VK_SPACE,
                /*Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()*/
        ActionEvent.CTRL_MASK + ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PLAYBACK_RATE_TOGGLE,
            KeyStroke.getKeyStroke(KeyEvent.VK_R,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PLAYBACK_VOLUME_TOGGLE,
            KeyStroke.getKeyStroke(KeyEvent.VK_R,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PLAY_PAUSE,
            KeyStroke.getKeyStroke(KeyEvent.VK_SPACE, /*Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()*/
                ActionEvent.CTRL_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PLAY_SELECTION,
            KeyStroke.getKeyStroke(KeyEvent.VK_SPACE, ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PREVIOUS_ACTIVE_TIER,
            KeyStroke.getKeyStroke(KeyEvent.VK_UP,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.PREVIOUS_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.PREVIOUS_ANNOTATION_EDIT,
            KeyStroke.getKeyStroke(KeyEvent.VK_LEFT,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.ALT_MASK /*+ ActionEvent.SHIFT_MASK*/));
        shortcutKeyStrokes.put(ELANCommandFactory.PREVIOUS_FRAME,
            KeyStroke.getKeyStroke(KeyEvent.VK_LEFT,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.PREVIOUS_SCROLLVIEW,
            KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_UP,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.PRINT,
            KeyStroke.getKeyStroke(KeyEvent.VK_P,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.PREVIEW,
            KeyStroke.getKeyStroke(KeyEvent.VK_P,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.REDO,
            KeyStroke.getKeyStroke(KeyEvent.VK_Y,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.REMOVE_ANNOTATION_VALUE,
            KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.SAVE_AS,
            KeyStroke.getKeyStroke(KeyEvent.VK_S,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.SAVE_AS_TEMPLATE,
            KeyStroke.getKeyStroke(KeyEvent.VK_S,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK + ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.SAVE,
            KeyStroke.getKeyStroke(KeyEvent.VK_S,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.SEARCH_DLG,
            KeyStroke.getKeyStroke(KeyEvent.VK_F,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.SEARCH_MULTIPLE_DLG,
            KeyStroke.getKeyStroke(KeyEvent.VK_F,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.SECOND_LEFT,
            KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.SECOND_RIGHT,
            KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT, ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.SELECTION_MODE,
            KeyStroke.getKeyStroke(KeyEvent.VK_K,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.SHIFT_ACTIVE_ANNOTATION,
            KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.STRUCTURED_SEARCH_MULTIPLE_DLG,
            KeyStroke.getKeyStroke(KeyEvent.VK_F,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.SHIFT_MASK + ActionEvent.ALT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.UNDO,
            KeyStroke.getKeyStroke(KeyEvent.VK_Z,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.HELP,
            KeyStroke.getKeyStroke(KeyEvent.VK_H,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.EXIT,
            KeyStroke.getKeyStroke(KeyEvent.VK_Q,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.NEW_DOC,
            KeyStroke.getKeyStroke(KeyEvent.VK_N,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.NEXT_WINDOW,
            KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, ActionEvent.SHIFT_MASK));
        shortcutKeyStrokes.put(ELANCommandFactory.OPEN_DOC,
            KeyStroke.getKeyStroke(KeyEvent.VK_O,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()));
        shortcutKeyStrokes.put(ELANCommandFactory.PREV_WINDOW,
            KeyStroke.getKeyStroke(KeyEvent.VK_UP, ActionEvent.SHIFT_MASK));
        //add all actions that have no default keystroke
        addActionsWithoutShortcut();
    }
    /**
     * adds all actions that are shortcuttable but are missing from shortcutKeyStrokes to it
     */
    private void addActionsWithoutShortcut() {

    		Iterator OuterIt = shortcuttableActions.entrySet().iterator();
    		while (OuterIt.hasNext())
    		{
    			Map.Entry kvpair = (Map.Entry) OuterIt.next();
    			List<String> actionList = (List<String>) kvpair.getValue();
    			Iterator InnerIt = actionList.iterator();
    			while (InnerIt.hasNext())
    			{
    				String action = (String) InnerIt.next();
    				if (!(shortcutKeyStrokes.containsKey(action)))
    				{
    					shortcutKeyStrokes.put(action, null);
    				}
    			}
    		}
	}

	/**
     * Returns the KeyStroke for the specified action identifier.
     *
     * @param actionID the identifier, one of the constants in {@link
     *        ELANCommandFactory}.
     *
     * @return a KeyStroke or null
     */
    public KeyStroke getKeyStrokeForAction(String actionID) {
        if (actionID != null) {
            return (KeyStroke) shortcutKeyStrokes.get(actionID);
        }

        return null;
    }

    /**
     * Returns the current id- keystroke mappings.
     *
     * @return a map containing id to keystroke mappings
     */
    public Map<String, KeyStroke> getCurrentShortcuts() {
        return shortcutKeyStrokes;
    }

    /**
     * Returns a map of all action that can have a shortcut. The keys are
     * category names, the values are lists of action identifiers.
     *
     * @return a mapping of all actions, grouped per category
     */
    public Map<String, List<String>> getShortcuttableActions() {
        return shortcuttableActions;
    }

    /**
     * Returns a (friendly) description of the action (by default the tooltip
     * description). This is independent of any transcription or instantiated
     * actions.
     *
     * @param actionID the id
     *
     * @return a description or the empty string
     */
    public String getDescriptionForAction(String actionID) {
        if (actionID == null) {
            return "";
        }

        String desc = ElanLocale.getString(actionID + "ToolTip");

        if ((desc == null) || (desc.length() == 0)) {
            desc = ElanLocale.getString(actionID);
        }

        return desc;
    }
    
    
    /**
     * Returns the category this action belongs to
     *
     * @param actionID the id
     *
     * @return a category name or the empty string
     */
    public String getCategoryForAction(String actionID)
    {
        if (actionID == null) {
            return "";
        }
        Iterator it = shortcuttableActions.entrySet().iterator();
        while (it.hasNext())
        {
        		Map.Entry pairs = (Map.Entry) it.next();
        		String cat = (String) pairs.getKey();
        		List<String> actionList = (ArrayList<String>) pairs.getValue();
        		if (actionList.contains(actionID))
        		{
        			return cat;
        		}
        }
        return "";
    }
    
    /**
     * Returns a user readable, platform specific, description of the key and
     * the modifiers.
     *
     * @param ks the keystroke
     *
     * @return the description
     */
    public String getDescriptionForKeyStroke(KeyStroke ks) {
        if (ks == null) {
            return "";
        }

        String nwAcc = "";

        if (SystemReporting.isMacOS()) {
            int modifier = ks.getModifiers();

            if ((modifier & InputEvent.META_MASK) != 0) {
                nwAcc += "Command+";
            }

            if ((modifier & InputEvent.CTRL_MASK) != 0) {
                nwAcc += "Ctrl+";
            }

            if ((modifier & InputEvent.ALT_MASK) != 0) {
                nwAcc += "Alt+";
            }

            if ((modifier & InputEvent.SHIFT_MASK) != 0) {
                nwAcc += "Shift+";
            }

            if (ks.getKeyChar() == KeyEvent.CHAR_UNDEFINED) {
                nwAcc += KeyEvent.getKeyText(ks.getKeyCode());
            } else {
                nwAcc += String.valueOf(ks.getKeyChar());
            }
        } else {
            int modifier = ks.getModifiers();

            if ((modifier & InputEvent.CTRL_MASK) != 0) {
                nwAcc += "Ctrl+";
            }

            if ((modifier & InputEvent.ALT_MASK) != 0) {
                nwAcc += "Alt+";
            }

            if ((modifier & InputEvent.SHIFT_MASK) != 0) {
                nwAcc += "Shift+";
            }

            if (ks.getKeyChar() == KeyEvent.CHAR_UNDEFINED) {
                nwAcc += KeyEvent.getKeyText(ks.getKeyCode());
            } else {
                nwAcc += String.valueOf(ks.getKeyChar());
            }
        }

        return nwAcc;
    }

    /**
     * Restores the default keyboard shortcuts.
     */
    public void restoreDefaultShortcuts() 
    {
        shortcutKeyStrokes.clear();
        loadDefaultShortcuts();        
        JOptionPane.showMessageDialog( null,ElanLocale.getString("Shortcuts.Message.Restored"));
    }

    /**
     * Reads stored shortcuts form file
     *
     * @return true if stored mappings have been loaded, false otherwise
     */
    public boolean readCurrentShortcuts() 
    {

    		String pref_filepath = Constants.ELAN_DATA_DIR +
        System.getProperty("file.separator") + "shortcuts.pfsx";
    		PreferencesReader	xmlPrefsReader = new PreferencesReader();
    		HashMap<String, KeyStroke> shortcutMap = new HashMap<String, KeyStroke>();
    		HashMap<String, List<String>> shortcutMapRaw = null;
    	 	try 
    		{
    	 		shortcutMapRaw = (HashMap<String, List<String>>) xmlPrefsReader.parse(pref_filepath);
         } 
    	 	catch (Exception ex) {
             ClientLogger.LOG.warning("Could not load the keyboard shortcut preferences file");
         }

    	 	if (shortcutMapRaw != null && (!shortcutMapRaw.isEmpty()))
    	 	{
        	 	Iterator it = shortcutMapRaw.entrySet().iterator();
        	 	while (it.hasNext())
        	 	{            	 		
        	 		Map.Entry pair = (Map.Entry) it.next();
        	 		String actionName = (String) pair.getKey();
            		Object val = pair.getValue();            	 		
	
            		if (val instanceof ArrayList) 
            		{
            			ArrayList<String> codes = (ArrayList<String>) val;
            			if (codes.isEmpty())
            			{
            				shortcutMap.put(actionName, null);
            			}
            			else
            			{
	            			int keycode = Integer.parseInt(codes.get(0));
	            			int modcode = Integer.parseInt(codes.get(1));
	            			KeyStroke aks = KeyStroke.getKeyStroke(keycode,modcode);
	            			shortcutMap.put(actionName, aks);
            			}
            		}
        	 	}
    	 		
    	 		shortcutKeyStrokes = shortcutMap;
    	 		// ensure that all shortcuttable Actions are in the hash
    	 		addActionsWithoutShortcut();
    	 		return true;
    	 	}
    	 	else
    	 	{
    	        return false;    	 		
    	 	}

        
    }

    /**
     * Saves the user defined keyboard shortcut to action mappings.
     */
    public void saveCurrentShortcuts(HashMap<String, List<String>> shortcutMap) 
    {
    		String pref_filepath = Constants.ELAN_DATA_DIR +
        System.getProperty("file.separator") + "shortcuts.pfsx";
    		PreferencesWriter xmlPrefsWriter = new PreferencesWriter();    		
    		try 
    		{
    			xmlPrefsWriter.encodeAndSave(shortcutMap, pref_filepath);
    	   		JOptionPane.showMessageDialog( null, ElanLocale.getString("Shortcuts.Message.Saved"));
        } 
    		catch (Exception ex) 
    		{
            ClientLogger.LOG.warning("Could not save the keyboard shortcut preferences file");
       		JOptionPane.showMessageDialog( null, ElanLocale.getString("Shortcuts.Message.NotSaved"));
        }
 
    }
        
}
