/*
 * File:     StoreCommand.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.FrameManager;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.svg.SVGParserAndStore;

import mpi.eudico.client.annotator.util.ElanFileFilter;
import mpi.eudico.client.annotator.util.FileUtility;

import mpi.eudico.server.corpora.clom.Transcription;
import mpi.eudico.server.corpora.clom.TranscriptionStore;

import mpi.eudico.server.corpora.clomimpl.abstr.LinkedFileDescriptor;
import mpi.eudico.server.corpora.clomimpl.abstr.MediaDescriptor;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import java.io.File;
import java.io.IOException;

import java.util.Vector;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;


/**
 * Saves a transcription as an .eaf or .etf (template) file, either creating a new
 * file (Save As) or overwriting an existing file (Save).<br>
 *
 * @version Nov 2007 added support for relative media paths
 * @author Hennie Brugman
 */
public class StoreCommand implements Command {
    private String commandName;

    /**
     * Creates a new StoreCommand instance
     *
     * @param name DOCUMENT ME!
     */
    public StoreCommand(String name) {
        commandName = name;
    }

    //arguments:
    //[0]: TranscriptionStore eafTranscriptionStore
    //[1]: Boolean saveAsTemplate
    //[2]: Boolean saveNewCopy
    //[3]: Vector visibleTiers
    public void execute(Object receiver, Object[] arguments) {
        Transcription tr = (Transcription) receiver;
        TranscriptionStore eafTranscriptionStore = (TranscriptionStore) arguments[0];
        boolean saveAsTemplate = ((Boolean) arguments[1]).booleanValue();
        boolean saveNewCopy = ((Boolean) arguments[2]).booleanValue();
        Vector visibleTiers;

        if (arguments[3] != null) {
            visibleTiers = (Vector) arguments[3];
        } else {
            visibleTiers = ELANCommandFactory.getViewerManager(tr)
                                             .getMultiTierControlPanel()
                                             .getVisibleTiers();
        }

        if (saveNewCopy) {
            // prompt for new file name
            String saveDir = (String) Preferences.get("LastUsedEAFDir", null);

            if (saveDir == null) {
                saveDir = (new File(tr.getName())).getParent();

                if (saveDir == null) {
                    saveDir = System.getProperty("user.dir");
                }
            }

            // open dialog at directory of original eaf file
            JFileChooser chooser = new JFileChooser(saveDir);

            if (saveAsTemplate) {
                chooser.setFileFilter(ElanFileFilter.createFileFilter(
                        ElanFileFilter.TEMPLATE_TYPE));
                chooser.setDialogTitle(ElanLocale.getString(
                        "SaveDialog.Template.Title"));
            } else {
                chooser.setFileFilter(ElanFileFilter.createFileFilter(
                        ElanFileFilter.EAF_TYPE));
                chooser.setDialogTitle(ElanLocale.getString("SaveDialog.Title"));
            }

            int option = chooser.showSaveDialog(ELANCommandFactory.getRootFrame(
                        tr));

            if (option == JFileChooser.APPROVE_OPTION) {
                File curDir = chooser.getCurrentDirectory();

                if (curDir != null) {
                    Preferences.set("LastUsedEAFDir", curDir.getAbsolutePath(),
                        null);
                }

                File f = chooser.getSelectedFile();

                if (f != null) {
                    // make sure pathname finishes with .eaf or .etf extension
                    String pathName = f.getAbsolutePath();
                    String lowerPathName = pathName.toLowerCase();

                    if (!lowerPathName.endsWith(".eaf") &&
                            !lowerPathName.endsWith(".etf")) {
                        if (saveAsTemplate) {
                            pathName += ".etf";
                        } else {
                            pathName += ".eaf";
                        }
                    }

                    if ((new File(pathName)).exists()) {
                        int answer = JOptionPane.showConfirmDialog(ELANCommandFactory.getRootFrame(
                                    tr),
                                ElanLocale.getString("Message.Overwrite"),
                                ElanLocale.getString("SaveDialog.Message.Title"),
                                JOptionPane.YES_NO_OPTION);

                        if (answer == JOptionPane.NO_OPTION) {
                            return;
                        }
                    }

                    if (saveAsTemplate) {
                        try {
                            eafTranscriptionStore.storeTranscriptionAsTemplateIn(tr,
                                visibleTiers, pathName);
                        } catch (IOException ioe) {
                            JOptionPane.showMessageDialog(ELANCommandFactory.getRootFrame(
                                    tr),
                                
                            //ElanLocale.getString("ExportDialog.Message.Error") + "\n" +
                            "Unable to save the template file: " + "(" +
                                ioe.getMessage() + ")",
                                ElanLocale.getString("Message.Error"),
                                JOptionPane.ERROR_MESSAGE);
                        }
                    } else {
                        // checks if there are any tiers allowing graphical annotations...
                        boolean saveSVG = false;

                        if (!saveSVG) {
                            Vector tiers = tr.getTiers();
                            TierImpl tier;

                            for (int i = 0; i < tiers.size(); i++) {
                                tier = (TierImpl) tiers.get(i);

                                if (tier.getLinguisticType()
                                            .hasGraphicReferences()) {
                                    saveSVG = true;

                                    break;
                                }
                            }
                        }

                        if (!saveSVG) {
                            ((TranscriptionImpl) tr).setSVGFile(null);
                        } else {
                            int index = pathName.lastIndexOf(".eaf");
                            String svgFileName = pathName.substring(0, index) +
                                ".svg";
                            ((TranscriptionImpl) tr).setSVGFile(svgFileName);

                            //SVGParserAndStore.storeSVG(tr);
                        }

                        // check, copy and update linked files, configuration and svg files 
                        Vector linkedFiles = tr.getLinkedFileDescriptors();
                        String svgExt = ".svg";
                        String confExt = "_tsconf.xml";
                        String curExt;

                        if (linkedFiles.size() > 0) {
                            LinkedFileDescriptor lfd;

                            for (int i = 0; i < linkedFiles.size(); i++) {
                                curExt = null;
                                lfd = (LinkedFileDescriptor) linkedFiles.get(i);

                                if (lfd.linkURL.toLowerCase().endsWith(confExt)) {
                                    curExt = confExt;
                                } else if (lfd.linkURL.toLowerCase().endsWith(svgExt)) {
                                    curExt = svgExt;
                                }

                                if (curExt != null) {
                                    // ELAN generated configuration file, copy
                                    String url = pathName.substring(0,
                                            pathName.length() - 4) + curExt;
                                    System.out.println("New conf: " + url);

                                    // copy conf or svg
                                    try {
                                        File source = null;
                                        File dest = null;

                                        if (lfd.linkURL.startsWith("file:")) {
                                            source = new File(lfd.linkURL.substring(
                                                        5));
                                        } else {
                                            source = new File(lfd.linkURL);
                                        }

                                        if (url.startsWith("file:")) {
                                            dest = new File(url.substring(5));
                                        } else {
                                            dest = new File(url);
                                        }

                                        FileUtility.copyToFile(source, dest);
                                    } catch (Exception ex) {
                                        System.out.println(
                                            "Could not copy the configuration file.");
                                    }

                                    lfd.linkURL = FileUtility.pathToURLString(url);
                                    tr.setChanged();
                                }
                            }
                        }

                        // update relative media paths
                        // make sure the eaf path is treated the same way as media files,
                        // i.e. it starts with file:/// or file://
                        String fullEAFURL = FileUtility.pathToURLString(pathName);
                        Vector mediaDescriptors = tr.getMediaDescriptors();
                        MediaDescriptor md;
                        String relUrl;

                        for (int i = 0; i < mediaDescriptors.size(); i++) {
                            md = (MediaDescriptor) mediaDescriptors.elementAt(i);
                            relUrl = FileUtility.getRelativePath(fullEAFURL,
                                    md.mediaURL);
                            md.relativeMediaURL = "file:/" + relUrl;
                        }

                        // linked other files 
                        if (linkedFiles.size() > 0) {
                            LinkedFileDescriptor lfd;

                            for (int i = 0; i < linkedFiles.size(); i++) {
                                lfd = (LinkedFileDescriptor) linkedFiles.get(i);
                                relUrl = FileUtility.getRelativePath(fullEAFURL,
                                        lfd.linkURL);
                                lfd.relativeLinkURL = "file:/" + relUrl;
                            }
                        }

                        // save
                        try {
                            eafTranscriptionStore.storeTranscriptionIn(tr,
                                null, visibleTiers, pathName,
                                TranscriptionStore.EAF);
                        } catch (IOException ioe) {
                            JOptionPane.showMessageDialog(ELANCommandFactory.getRootFrame(
                                    tr),
                                
                            //ElanLocale.getString("ExportDialog.Message.Error") + "\n" +
                            "Unable to save the transcription file: " + "(" +
                                ioe.getMessage() + ")",
                                ElanLocale.getString("Message.Error"),
                                JOptionPane.ERROR_MESSAGE);
                        }

                        String name = pathName;
                        int lastSlashPos = name.lastIndexOf(System.getProperty(
                                    "file.separator"));

                        if (lastSlashPos >= 0) {
                            name = name.substring(lastSlashPos + 1);
                        }

                        //System.out.println("nm " + name);
                        tr.setName(name);

                        //tr.setName(pathName);
                        if (tr instanceof TranscriptionImpl) {
                            ((TranscriptionImpl) tr).setPathName(pathName);
                            ELANCommandFactory.getRootFrame(tr).setTitle("ELAN - " +
                                tr.getName());
                            FrameManager.getInstance().updateFrameTitle(ELANCommandFactory.getRootFrame(
                                    tr), pathName);
                        } else {
                            ELANCommandFactory.getRootFrame(tr).setTitle("ELAN - " +
                                name);
                        }

                        tr.setUnchanged();

                        // save svg
                        if (saveSVG) {
                            SVGParserAndStore.storeSVG(tr);
                        }

                        // create a new backup timer
                        if (tr instanceof TranscriptionImpl) {
                            ((BackupCA) ELANCommandFactory.getCommandAction(tr,
                                ELANCommandFactory.BACKUP)).setFilePath(pathName);
                        }
                    }
                }
            }
        } else if (tr.isChanged()) {
            // check svg
            boolean saveSVG = false;
            Vector tiers = tr.getTiers();
            TierImpl tier;

            for (int i = 0; i < tiers.size(); i++) {
                tier = (TierImpl) tiers.get(i);

                if (tier.getLinguisticType().hasGraphicReferences()) {
                    saveSVG = true;

                    break;
                }
            }

            if (!saveSVG) {
                ((TranscriptionImpl) tr).setSVGFile(null);
            }

            String svgFileName = ((TranscriptionImpl) tr).getSVGFile();

            if ((svgFileName == null) && saveSVG) {
                String pathName = ((TranscriptionImpl) tr).getPathName();
                int index = pathName.lastIndexOf(".eaf");
                String newSvgFileName = pathName.substring(0, index) + ".svg";
                ((TranscriptionImpl) tr).setSVGFile(newSvgFileName);
            }

            // check if relative media paths have to be generated or updated
            // make sure the eaf path is treated the same way as media files,
            // i.e. it starts with file:/// or file://
            String fullEAFURL = FileUtility.pathToURLString(tr.getFullPath());
            Vector mediaDescriptors = tr.getMediaDescriptors();
            MediaDescriptor md;
            String relUrl;

            for (int i = 0; i < mediaDescriptors.size(); i++) {
                md = (MediaDescriptor) mediaDescriptors.elementAt(i);
                relUrl = FileUtility.getRelativePath(fullEAFURL, md.mediaURL);
                md.relativeMediaURL = "file:/" + relUrl;
            }

            // linked other files 
            Vector linkedFiles = tr.getLinkedFileDescriptors();

            if (linkedFiles.size() > 0) {
                LinkedFileDescriptor lfd;

                for (int i = 0; i < linkedFiles.size(); i++) {
                    lfd = (LinkedFileDescriptor) linkedFiles.get(i);
                    relUrl = FileUtility.getRelativePath(fullEAFURL, lfd.linkURL);
                    lfd.relativeLinkURL = "file:/" + relUrl;
                }
            }

            try {
                eafTranscriptionStore.storeTranscription(tr, null,
                    visibleTiers, TranscriptionStore.EAF);
            } catch (IOException ioe) {
                JOptionPane.showMessageDialog(ELANCommandFactory.getRootFrame(
                        tr),
                    
                //ElanLocale.getString("ExportDialog.Message.Error") + "\n" +
                "Unable to save the transcription file: " + "(" +
                    ioe.getMessage() + ")",
                    ElanLocale.getString("Message.Error"),
                    JOptionPane.ERROR_MESSAGE);
            }

            if ((svgFileName != null) || saveSVG) {
                SVGParserAndStore.storeSVG(tr);
            }

            //SVGParserAndStore.storeSVG(tr, null);    
            tr.setUnchanged();
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public String getName() {
        return commandName;
    }
}
