/*
 * File:     MenuAction.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands.global;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.ElanLocaleListener;

import mpi.eudico.client.annotator.commands.ShortcutsUtil;

import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;


/**
 * The base class for action objects that are not connected to a particular
 * document object (Transcription). Used for JMenu's, that have an empty
 * actionPerformed method, and a few JMenuItems that are present in e.g.  an
 * empty Elan frame. Useful for handling updateLocale and setting the
 * mnemonic.
 *
 * @author Han Sloetjes, MPI
 * @version April 2009: keyboard shortcuts are now obtained via the ShortcutUtil
 * class. The accelerator keys are now user definable.
 */
public class MenuAction extends AbstractAction implements ElanLocaleListener {
    /** a prefix for mnemonic keys */
    public static final String MNEMONIC = "MNEMONIC.";

    /** the command name */
    protected String commandId;

    /**
     * Creates a new MenuAction.
     *
     * @param name the name of the command
     */
    public MenuAction(String name) {
        super(name);

        commandId = name;
        putValue(Action.ACCELERATOR_KEY,
            ShortcutsUtil.getInstance().getKeyStrokeForAction(commandId));
        updateLocale();
    }

    /**
     * Updates the NAME (what is shown in menu's and on buttons), tooltip  and
     * eventually mnemonic.
     */
    public void updateLocale() {
        if (commandId != null) {
            putValue(Action.NAME, ElanLocale.getString(commandId));

            // don't bother about an icon...
            Object desc = getValue(Action.SHORT_DESCRIPTION);

            if (desc instanceof String && (((String) desc).length() > 0)) {
                putValue(Action.SHORT_DESCRIPTION,
                    ElanLocale.getString(commandId + "ToolTip"));
            }

            // mnemonic
            String mnemonic = ElanLocale.getString(MNEMONIC + commandId);

            if (mnemonic.length() > 0) {
                try {
                    putValue(Action.MNEMONIC_KEY,
                        new Integer(mnemonic.charAt(0)));
                } catch (NumberFormatException nfe) {
                    try {
                        putValue(Action.MNEMONIC_KEY, new Integer(mnemonic));
                    } catch (NumberFormatException nfe2) {
                        putValue(Action.MNEMONIC_KEY, null);
                    }
                }
            }
        }
    }

    /**
     * The action; empty by default.
     *
     * @param e the action event
     */
    public void actionPerformed(ActionEvent e) {
    }
}
