/*
 * File:     CHATExportDlg.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * Options dialog to set all parameters for CHAT Export.
 *
 * @author Hennie Brugman
 */
package mpi.eudico.client.annotator.export;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.gui.ClosableDialog;

import mpi.eudico.client.annotator.util.ElanFileFilter;

import mpi.eudico.server.corpora.clom.Tier;
import mpi.eudico.server.corpora.clom.Transcription;
import mpi.eudico.server.corpora.clom.TranscriptionStore;

import mpi.eudico.server.corpora.clomimpl.abstr.MediaDescriptor;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;
import mpi.eudico.server.corpora.clomimpl.chat.CHATEncoderInfo;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.io.File;
import java.io.IOException;

import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.border.TitledBorder;
import javax.swing.filechooser.FileFilter;


/**
 * DOCUMENT ME!
 *
 * @author hennie
 */
public class CHATExportDlg extends ClosableDialog implements ActionListener {
    /** Holds value of property DOCUMENT ME! */
    private final int NUM_OF_COLUMNS = 7;

    /** Holds value of property DOCUMENT ME! */
    private final int NUM_OF_DEP_COLUMNS = 3;

    /** Holds value of property DOCUMENT ME! */
    private final String MAIN_TIER = "Main Tier";

    /** Holds value of property DOCUMENT ME! */
    private final String DEPENDENT_TIER = "Dependent Tier";

    /** Holds value of property DOCUMENT ME! */
    private final String LABEL = "Label";

    /** Holds value of property DOCUMENT ME! */
    private final String FULL_NAME = "Full Name";

    /** Holds value of property DOCUMENT ME! */
    private final String ROLE = "Role";

    /** Holds value of property DOCUMENT ME! */
    private final String ID = "ID";

    /** Holds value of property DOCUMENT ME! */
    private final String LANGUAGE = "Language";
    private TranscriptionImpl transcription;
    private TranscriptionStore acmTranscriptionStore;
    private Vector visibleTiers;
    private JLabel titleLabel;
    private JPanel titlePanel;
    private JComponent[][] mainTierTable;
    private JComponent[][] dependentTierTable;
    private JPanel mainTiersPanel;
    private JPanel dependentTiersPanel;
    private JPanel optionsPanel;
    private JPanel buttonPanel;
    private JButton exportButton;
    private TitledBorder mainTiersBorder;
    private TitledBorder dependentTiersBorder;
    private TitledBorder optionsBorder;
    private JCheckBox correctTimesCB;
    private JCheckBox timesOnSeparateLineCB;
    private JCheckBox includeLanguageLineCB;

    /**
     * Creates a new CHATExportDlg instance
     *
     * @param frame DOCUMENT ME!
     * @param modal DOCUMENT ME!
     * @param tr DOCUMENT ME!
     * @param acmTranscriptionStore DOCUMENT ME!
     * @param visibleTiers DOCUMENT ME!
     */
    public CHATExportDlg(JFrame frame, boolean modal, Transcription tr,
        TranscriptionStore acmTranscriptionStore, Vector visibleTiers) {
        super(frame, modal);

        //this.frame = frame;
        transcription = (TranscriptionImpl) tr;
        this.acmTranscriptionStore = acmTranscriptionStore;
        this.visibleTiers = visibleTiers;

        // create main tier table (num of root tier records, NUM_OF_COLUMNS columns each)
        Vector topTiers = transcription.getTopTiers();

        if (topTiers != null) {
            int numOfTiers = transcription.getTiers().size();
            mainTierTable = new JComponent[NUM_OF_COLUMNS][topTiers.size() + 1];
            dependentTierTable = new JComponent[NUM_OF_DEP_COLUMNS][numOfTiers -
                topTiers.size() + 1];
        }

        mainTiersPanel = new JPanel();
        dependentTiersPanel = new JPanel();
        buttonPanel = new JPanel();

        exportButton = new JButton();

        mainTiersBorder = new TitledBorder("Main tiers");
        dependentTiersBorder = new TitledBorder("Dependent tiers");

        createDialog();
        updateForLocale();
        setDefaultValues();
        pack();
        setLocationRelativeTo(getParent());

        //setResizable(false);
    }

    private void createDialog() {
        getContentPane().setLayout(new GridBagLayout());

        Insets insets = new Insets(2, 6, 2, 6);
        titleLabel = new JLabel();
        titlePanel = new JPanel();
        titlePanel.setLayout(new BorderLayout(0, 4));
        titleLabel.setFont(titleLabel.getFont().deriveFont((float) 16));
        titleLabel.setHorizontalAlignment(SwingConstants.CENTER);

        JPanel titleLabelPanel = new JPanel();
        titleLabelPanel.add(titleLabel);
        titlePanel.add(titleLabelPanel, BorderLayout.NORTH);

        GridBagConstraints gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = GridBagConstraints.NORTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = insets;
        getContentPane().add(titlePanel, gridBagConstraints);

        mainTiersPanel.setLayout(new GridBagLayout());
        dependentTiersPanel.setLayout(new GridBagLayout());
        buttonPanel.setLayout(new GridBagLayout());

        GridBagConstraints c = new GridBagConstraints();

        // main tiers panel
        JComponent tableComponent = null;

        JPanel mtPanel = new JPanel(new GridBagLayout());
        JScrollPane scrollPane = new JScrollPane(mtPanel);
        scrollPane.setBorder(null);
        mainTiersPanel.setBorder(mainTiersBorder);

        c.gridx = 0;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        c.weighty = 0.0;
        c.insets = insets;

        //getContentPane().add(mainTiersPanel, c);
        mtPanel.add(mainTiersPanel, c);

        // header row
        c = new GridBagConstraints();
        tableComponent = new JLabel(MAIN_TIER);
        mainTierTable[1][0] = tableComponent;
        c.gridx = 1;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        c.insets = insets;
        mainTiersPanel.add(tableComponent, c);

        tableComponent = new JLabel(LABEL);
        mainTierTable[2][0] = tableComponent;
        c.gridx = 2;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        c.insets = insets;
        mainTiersPanel.add(tableComponent, c);

        tableComponent = new JLabel(FULL_NAME);
        mainTierTable[3][0] = tableComponent;
        c.gridx = 3;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        c.insets = insets;
        mainTiersPanel.add(tableComponent, c);

        tableComponent = new JLabel(ROLE);
        mainTierTable[4][0] = tableComponent;
        c.gridx = 4;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        c.insets = insets;
        mainTiersPanel.add(tableComponent, c);

        tableComponent = new JLabel(ID);
        mainTierTable[5][0] = tableComponent;
        c.gridx = 5;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        c.insets = insets;
        mainTiersPanel.add(tableComponent, c);

        tableComponent = new JLabel(LANGUAGE);
        mainTierTable[5][0] = tableComponent;
        c.gridx = 6;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        c.insets = insets;
        mainTiersPanel.add(tableComponent, c);

        // row for each top level tier
        Vector topTiers = transcription.getTopTiers();

        if (topTiers != null) {
            for (int i = 0; i < topTiers.size(); i++) {
                String tName = ((Tier) topTiers.elementAt(i)).getName();

                tableComponent = new JCheckBox();
                ((JCheckBox) tableComponent).setSelected(true);
                mainTierTable[0][i + 1] = tableComponent;
                c.gridx = 0;
                c.gridy = i + 1;
                c.anchor = GridBagConstraints.WEST;
                c.insets = insets;
                mainTiersPanel.add(tableComponent, c);

                tableComponent = new JLabel(tName);
                mainTierTable[1][i + 1] = tableComponent;
                c.gridx = 1;
                c.gridy = i + 1;
                c.anchor = GridBagConstraints.WEST;
                c.insets = insets;
                mainTiersPanel.add(tableComponent, c);

                tableComponent = new JTextField(3);

                String defaultName = "*";

                if (tName.startsWith("*") && (tName.length() == 4)) {
                    defaultName = tName;
                }

                ((JTextField) tableComponent).setText(defaultName);
                mainTierTable[2][i + 1] = tableComponent;
                c.gridx = 2;
                c.gridy = i + 1;
                c.anchor = GridBagConstraints.WEST;
                c.insets = insets;
                mainTiersPanel.add(tableComponent, c);

                tableComponent = new JTextField(12);
                mainTierTable[3][i + 1] = tableComponent;
                c.gridx = 3;
                c.gridy = i + 1;
                c.anchor = GridBagConstraints.WEST;
                c.insets = insets;
                mainTiersPanel.add(tableComponent, c);

                tableComponent = new JTextField(8);
                mainTierTable[4][i + 1] = tableComponent;
                c.gridx = 4;
                c.gridy = i + 1;
                c.anchor = GridBagConstraints.WEST;
                c.insets = insets;
                mainTiersPanel.add(tableComponent, c);

                tableComponent = new JTextField(18);
                mainTierTable[5][i + 1] = tableComponent;
                c.gridx = 5;
                c.gridy = i + 1;
                c.anchor = GridBagConstraints.WEST;
                c.insets = insets;
                mainTiersPanel.add(tableComponent, c);

                tableComponent = new JTextField(8);

                String language = ((TierImpl) topTiers.elementAt(i)).getDefaultLocale()
                                   .getLanguage();

                if ((language != null) && !language.equals("")) {
                    ((JTextField) tableComponent).setText(language);
                }

                mainTierTable[6][i + 1] = tableComponent;
                c.gridx = 6;
                c.gridy = i + 1;
                c.anchor = GridBagConstraints.WEST;
                c.insets = insets;
                mainTiersPanel.add(tableComponent, c);
            }
        }

        // filler
        c.gridx = 7;
        c.gridy = 0;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        mainTiersPanel.add(new JPanel(), c);

        // dependent tiers panel
        dependentTiersPanel.setBorder(dependentTiersBorder);

        c.gridx = 0;
        c.gridy = 1;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.WEST;
        c.insets = insets;
        c.weightx = 1.0;

        //getContentPane().add(dependentTiersPanel, c);
        mtPanel.add(dependentTiersPanel, c);

        // header row
        tableComponent = new JLabel(DEPENDENT_TIER);
        dependentTierTable[1][0] = tableComponent;
        c.gridx = 1;
        c.gridy = 0;
        c.fill = GridBagConstraints.NONE;
        c.weightx = 0.0;
        c.anchor = GridBagConstraints.WEST;
        c.insets = insets;
        dependentTiersPanel.add(tableComponent, c);

        tableComponent = new JLabel(LABEL);
        dependentTierTable[1][0] = tableComponent;
        c.gridx = 2;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        c.insets = insets;
        dependentTiersPanel.add(tableComponent, c);

        // row for each dependent tier
        Vector tiers = null;

        int rowIndex = 1;

        tiers = transcription.getTiers();

        if (tiers != null) {
            for (int i = 0; i < tiers.size(); i++) {
                TierImpl t = (TierImpl) tiers.elementAt(i);

                if (t.hasParentTier()) {
                    tableComponent = new JCheckBox();
                    ((JCheckBox) tableComponent).setSelected(true);
                    dependentTierTable[0][rowIndex] = tableComponent;
                    c.gridx = 0;
                    c.gridy = rowIndex;
                    c.anchor = GridBagConstraints.WEST;
                    c.insets = insets;
                    dependentTiersPanel.add(tableComponent, c);

                    tableComponent = new JLabel(t.getName());
                    dependentTierTable[1][rowIndex] = tableComponent;
                    c.gridx = 1;
                    c.gridy = rowIndex;
                    c.anchor = GridBagConstraints.WEST;
                    c.insets = insets;
                    dependentTiersPanel.add(tableComponent, c);

                    String defaultName = "%";

                    if (t.getName().startsWith("%")) {
                        int atInd = t.getName().indexOf('@');

                        if ((atInd > 1) && (atInd <= 8)) {
                            defaultName = t.getName().substring(0, atInd);
                        }
                    }

                    tableComponent = new JTextField(7);
                    ((JTextField) tableComponent).setText(defaultName);
                    dependentTierTable[2][rowIndex] = tableComponent;
                    c.gridx = 2;
                    c.gridy = rowIndex;
                    c.anchor = GridBagConstraints.WEST;
                    c.insets = insets;
                    dependentTiersPanel.add(tableComponent, c);

                    rowIndex++;
                }
            }
        }

        // filler
        c.gridx = 3;
        c.gridy = 0;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        dependentTiersPanel.add(new JPanel(), c);

        optionsPanel = new JPanel(new GridBagLayout());
        optionsBorder = new TitledBorder("");
        optionsPanel.setBorder(optionsBorder);
        correctTimesCB = new JCheckBox();
        correctTimesCB.setSelected(true);
        timesOnSeparateLineCB = new JCheckBox();
        includeLanguageLineCB = new JCheckBox();

        c = new GridBagConstraints();
        c.gridx = 0;
        c.gridy = 0;
        c.insets = insets;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.WEST;
        c.weightx = 1.0;
        optionsPanel.add(correctTimesCB, c);

        c = new GridBagConstraints();
        c.gridx = 0;
        c.gridy = 1;
        c.insets = insets;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.WEST;
        c.weightx = 1.0;
        optionsPanel.add(timesOnSeparateLineCB, c);

        c = new GridBagConstraints();
        c.gridx = 0;
        c.gridy = 2;
        c.insets = insets;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.WEST;
        c.weightx = 1.0;
        optionsPanel.add(includeLanguageLineCB, c);

        c = new GridBagConstraints();
        c.gridx = 0;
        c.gridy = 1;
        c.insets = insets;
        c.fill = GridBagConstraints.BOTH;
        c.weightx = 1.0;
        c.weighty = 100.0;
        getContentPane().add(scrollPane, c);

        c.gridx = 0;
        c.gridy = 2;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.NORTH;
        c.weightx = 1.0;
        c.weighty = 0.0;
        c.insets = new Insets(2, 12, 2, 12);
        getContentPane().add(optionsPanel, c);

        c.gridx = 0;
        c.gridy = 3;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.SOUTH;
        c.weightx = 0.0;
        c.weighty = 0.0;
        c.insets = insets;
        getContentPane().add(buttonPanel, c);

        c.gridx = 0;
        c.gridy = 0;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.NORTH;
        c.insets = insets;
        buttonPanel.add(exportButton, c);
    }

    private void setDefaultValues() {
        //		selectionOnlyCheckBox.setSelected(false);
        //		selectionOnlyCheckBox.setEnabled(false);
        //		selectionOnlyLabel.setEnabled(false);
        //		showTierLabelCheckBox.setSelected(interlinearizer.isTierLabelsShown());		
        //		widthTextField.setText(new Integer(initialWidth).toString());
    }

    /**
     * Update the UI elements according to the current Locale and the current
     * edit mode.
     */
    private void updateForLocale() {
        setTitle(ElanLocale.getString("ExportCHATDialog.Title"));
        titleLabel.setText(ElanLocale.getString("ExportCHATDialog.Title"));
        mainTiersBorder.setTitle(ElanLocale.getString(
                "ExportCHATDialog.MainTiers"));
        dependentTiersBorder.setTitle(ElanLocale.getString(
                "ExportCHATDialog.DependentTiers"));
        optionsBorder.setTitle(ElanLocale.getString(
                "ExportDialog.Label.Options"));
        correctTimesCB.setText(ElanLocale.getString("ExportDialog.CorrectTimes"));
        timesOnSeparateLineCB.setText(ElanLocale.getString(
                "ExportCHATDialog.SeparateLine"));
        includeLanguageLineCB.setText(ElanLocale.getString(
                "ExportCHATDialog.LanguageLine"));
        exportButton.setText(ElanLocale.getString("ExportCHATDialog.Export"));
        exportButton.addActionListener(this);
    }

    //listeners
    public void actionPerformed(ActionEvent event) {
        if (event.getSource() == exportButton) {
            if (fieldsOK()) {
                showDialogAndSave();
            }
        }
    }

    private boolean fieldsOK() {
        boolean ok = true;

        // main tier table: all fields have to start with * and have length 4
        for (int i = 1; i < mainTierTable[2].length; i++) {
            if (((JCheckBox) mainTierTable[0][i]).isSelected()) {
                String text = ((JTextField) mainTierTable[2][i]).getText();

                if ((text.length() != 4) || !text.startsWith("*")) {
                    ok = false;

                    break;
                }
            }
        }

        // dependent tier table: all fields have to start with % and have length 4
        // dec 2006  HS: depending tier names can be 1 to 7 chars long
        for (int i = 1; i < dependentTierTable[2].length; i++) {
            if (((JCheckBox) dependentTierTable[0][i]).isSelected()) {
                String text = ((JTextField) dependentTierTable[2][i]).getText();

                if (((text.length() < 2) || (text.length() > 8)) ||
                        !text.startsWith("%")) {
                    ok = false;

                    break;
                }
            }
        }

        if (!ok) { // give error message
            JOptionPane.showMessageDialog(this,
                ElanLocale.getString("ExportCHATDlg.Message.WrongLabel"),
                ElanLocale.getString("Message.Error"), JOptionPane.ERROR_MESSAGE);
        }

        return ok;
    }

    private void showDialogAndSave() {
        // prompt for new file name
        String saveDir = (String) Preferences.get("LastUsedCHATDir", null);

        if (saveDir == null) {
            saveDir = (new File(transcription.getName())).getParent();

            if (saveDir == null) {
                saveDir = System.getProperty("user.dir");
            }
        }

        // open dialog at directory of original eaf file
        JFileChooser chooser = new JFileChooser(saveDir);
        FileFilter filter = ElanFileFilter.createFileFilter(ElanFileFilter.CHAT_TYPE);

        chooser.setFileFilter(filter);
        chooser.setDialogTitle(ElanLocale.getString("ExportCHATDialog.Title"));

        //	int option = chooser.showSaveDialog(ELANCommandFactory.getRootFrame(
        //				tr));
        int option = chooser.showSaveDialog(this);

        if (option == JFileChooser.APPROVE_OPTION) {
            File curDir = chooser.getCurrentDirectory();

            if (curDir != null) {
                Preferences.set("LastUsedCHATDir", curDir.getAbsolutePath(),
                    null);
            }

            File f = chooser.getSelectedFile();

            if (f != null) {
                // make sure pathname finishes with .cha extension
                String pathName = f.getAbsolutePath();
                String lowerPathName = pathName.toLowerCase();

                if (!lowerPathName.endsWith(".cha")) {
                    pathName += ".cha";
                }

                if ((new File(pathName)).exists()) {
                    int answer = JOptionPane.showConfirmDialog(null,
                            ElanLocale.getString("Message.Overwrite"),
                            ElanLocale.getString(
                                "ExportCHATDialog.Message.Title"),
                            JOptionPane.YES_NO_OPTION);

                    if (answer == JOptionPane.NO_OPTION) {
                        return;
                    }
                }

                // collect encoder information to pass on
                String[][] mainTierInfo = new String[mainTierTable.length - 1][mainTierTable[0].length -
                    1];
                String[][] dependentTierInfo = new String[dependentTierTable.length -
                    1][dependentTierTable[0].length - 1];

                int index = 0;

                for (int i = 1; i < mainTierTable[1].length; i++) {
                    if (((JCheckBox) mainTierTable[0][i]).isSelected()) {
                        mainTierInfo[0][index] = ((JLabel) mainTierTable[1][i]).getText();
                        mainTierInfo[1][index] = ((JTextField) mainTierTable[2][i]).getText();
                        mainTierInfo[2][index] = ((JTextField) mainTierTable[3][i]).getText();
                        mainTierInfo[3][index] = ((JTextField) mainTierTable[4][i]).getText();
                        mainTierInfo[4][index] = ((JTextField) mainTierTable[5][i]).getText();
                        mainTierInfo[5][index] = ((JTextField) mainTierTable[6][i]).getText();

                        index++;
                    }
                }

                index = 0;

                for (int j = 1; j < dependentTierTable[1].length; j++) {
                    if (((JCheckBox) dependentTierTable[0][j]).isSelected()) {
                        dependentTierInfo[0][index] = ((JLabel) dependentTierTable[1][j]).getText();
                        dependentTierInfo[1][index] = ((JTextField) dependentTierTable[2][j]).getText();

                        index++;
                    }
                }

                CHATEncoderInfo encoderInfo = new CHATEncoderInfo(mainTierInfo,
                        dependentTierInfo);
                encoderInfo.setCorrectAnnotationTimes(correctTimesCB.isSelected());
                encoderInfo.setTimesOnSeparateLine(timesOnSeparateLineCB.isSelected());
                encoderInfo.setIncludeLangLine(includeLanguageLineCB.isSelected());

                if (correctTimesCB.isSelected()) {
                    Vector mediaDescriptors = transcription.getMediaDescriptors();

                    if (mediaDescriptors.size() > 0) {
                        encoderInfo.setMediaOffset(((MediaDescriptor) mediaDescriptors.get(
                                0)).timeOrigin);
                    }
                }

                try {
                    acmTranscriptionStore.storeTranscriptionIn(transcription,
                        encoderInfo, visibleTiers, pathName,
                        TranscriptionStore.CHAT);
                } catch (IOException ioe) {
                    JOptionPane.showMessageDialog(this,
                        ElanLocale.getString("ExportDialog.Message.Error") +
                        "\n" + "(" + ioe.getMessage() + ")",
                        ElanLocale.getString("Message.Error"),
                        JOptionPane.ERROR_MESSAGE);
                }
            }
        }
    }
}
