/*
 * File:     ExportResultTableAsEAF.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.export;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.commands.ELANCommandFactory;

import mpi.eudico.client.annotator.grid.AnnotationTable;
import mpi.eudico.client.annotator.grid.GridViewerTableModel;

import mpi.eudico.client.annotator.search.result.model.ElanMatch;
import mpi.eudico.client.annotator.search.result.viewer.EAFResultViewerTableModel;

import mpi.eudico.client.annotator.util.AnnotationRecreator;
import mpi.eudico.client.annotator.util.ClientLogger;
import mpi.eudico.client.annotator.util.ElanFileFilter;
import mpi.eudico.client.annotator.util.FileExtension;

import mpi.eudico.client.util.TierTree;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.AnnotationCore;
import mpi.eudico.server.corpora.clom.TranscriptionStore;

import mpi.eudico.server.corpora.clomimpl.abstr.AlignableAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.LinkedFileDescriptor;
import mpi.eudico.server.corpora.clomimpl.abstr.MediaDescriptor;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;
import mpi.eudico.server.corpora.clomimpl.dobes.ACMTranscriptionStore;
import mpi.eudico.server.corpora.clomimpl.type.Constraint;
import mpi.eudico.server.corpora.clomimpl.type.IncludedIn;
import mpi.eudico.server.corpora.clomimpl.type.LinguisticType;
import mpi.eudico.server.corpora.clomimpl.type.SymbolicAssociation;
import mpi.eudico.server.corpora.clomimpl.type.SymbolicSubdivision;
import mpi.eudico.server.corpora.clomimpl.type.TimeSubdivision;

import mpi.util.CVEntry;
import mpi.util.ControlledVocabulary;

import java.io.File;
import java.io.IOException;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Vector;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.ProgressMonitor;
import javax.swing.filechooser.FileFilter;
import javax.swing.tree.DefaultMutableTreeNode;


/**
 * Export the annotations in the search result table with their context (the annotation block the annotations
 * is part of) to a new eaf file.
 */
public class ExportResultTableAsEAF implements ClientLogger {
    private ProgressMonitor monitor;

    /**
     * Creates new instance.
     */
    public ExportResultTableAsEAF() {
        super();
    }

    /**
     * Prompts for a file name, extracts the tiers (and thus types and cv's) that need to be copied
     * and copies the toplevel parent annotations of all annotations in the annotation table.
     *
     * @param table the search result table
     */
    public void exportTableAsEAF(final AnnotationTable table) {
        if ((table == null) || (table.getRowCount() == 0)) {
            // warn?
            return;
        }

        final String fileName = promptForFileName();

        if (fileName == null) {
            return;
        }

        monitor = new ProgressMonitor(null,
                ElanLocale.getString("SaveDialog.Message.Title"), "", 0, 100);
        monitor.setMillisToDecideToPopup(10);
        monitor.setMillisToPopup(10);

        new Thread() {
                public void run() {
                    try {
                        Thread.sleep(50);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }

                    TranscriptionImpl transcription = null;

                    ///GridViewerTableModel dataModel = (GridViewerTableModel) table.getModel();
                    EAFResultViewerTableModel dataModel = (EAFResultViewerTableModel) table.getModel();
                    AnnotationCore anc = null;
                    Annotation ann = null;
                    AlignableAnnotation aa = null;

                    // first get the transcription
                    anc = dataModel.getAnnotationCore(0);

                    if (anc instanceof Annotation) {
                        ann = (Annotation) anc;
                        transcription = (TranscriptionImpl) ann.getTier()
                                                               .getParent();
                    }

                    if (transcription == null) {
                        //warn message
                        LOG.warning(
                            "Could not retrieve the transcription, no results exported");
                        progressUpdate(null, 100);

                        return;
                    }

                    progressUpdate(null, 10);

                    if (isCancelled()) {
                        return;
                    }

                    ArrayList topAnnos = new ArrayList();
                    ArrayList topTiers = new ArrayList();
                    int annColumn = -1;
                    Object val;
                    ElanMatch match;

                    for (int i = 0; i < dataModel.getColumnCount(); i++) {
                        if (dataModel.getColumnName(i).equals(GridViewerTableModel.ANNOTATION)) {
                            annColumn = i;

                            break;
                        }
                    }

                    if (annColumn < 0) {
                        LOG.warning(
                            "Could not find the matches column in the table");
                        progressUpdate(null, 100);

                        return;
                    }

                    for (int i = 0; i < dataModel.getRowCount(); i++) {
                        val = dataModel.getValueAt(i, annColumn);

                        if (val instanceof ElanMatch) {
                            match = (ElanMatch) val;

                            //System.out.println("" + i + " " + match.getValue());
                            // only process toplevel matches; their children will all be checked
                            if (match.getParent() != null) {
                                continue;
                            }

                            ann = match.getAnnotation();
                            aa = rootAnnotationOf(ann);

                            if ((aa != null) && !topAnnos.contains(aa)) {
                                topAnnos.add(aa);
                            }

                            if ((aa != null) &&
                                    !topTiers.contains(aa.getTier())) {
                                topTiers.add(aa.getTier());
                            }

                            // process children
                            extractSubMatches(match, topAnnos, topTiers);
                        } else {
                            anc = dataModel.getAnnotationCore(i);

                            //System.out.println("" + i + " " + anc.getClass());
                            if (anc instanceof Annotation) {
                                ann = (Annotation) anc;
                                aa = rootAnnotationOf(ann);

                                if ((aa != null) && !topAnnos.contains(aa)) {
                                    topAnnos.add(aa);
                                }

                                if (!topTiers.contains(aa.getTier())) {
                                    topTiers.add(aa.getTier());
                                }
                            }
                        }
                    }

                    // now create a new transcription, either with only the tiers, types and cv's in use by
                    // the annotations or all tiers, types and cv's?
                    TranscriptionImpl nextTrans = new TranscriptionImpl();
                    nextTrans.setNotifying(false);
                    progressUpdate(null, 15);

                    if (isCancelled()) {
                        return;
                    }

                    copyDescriptors(transcription, nextTrans);

                    if (isCancelled()) {
                        return;
                    }

                    copyTiersTypesCvs(transcription, nextTrans, topTiers);

                    if (isCancelled()) {
                        return;
                    }

                    copyAnnotations(transcription, nextTrans, topAnnos);
                    progressUpdate(null, 90);

                    if (isCancelled()) {
                        return;
                    }

                    TranscriptionStore store = ACMTranscriptionStore.getCurrentTranscriptionStore();

                    try {
                        store.storeTranscription(nextTrans, null, new Vector(),
                            fileName, TranscriptionStore.EAF);
                    } catch (IOException ioe) {
                        JOptionPane.showMessageDialog(ELANCommandFactory.getRootFrame(
                                transcription),
                            
                        //ElanLocale.getString("ExportDialog.Message.Error") + "\n" +
                        "Unable to save the file: " + "(" + ioe.getMessage() +
                            ")", ElanLocale.getString("Message.Error"),
                            JOptionPane.ERROR_MESSAGE);
                    }

                    progressUpdate(null, 100);
                }
            }.start();
    }

    /**
     * Recursively extract annotations from sub matches.
     * @param match the (sub)match
     * @param topAnnos the list of toplevel annotations in the matches
     * @param topTiers the list of involved toplevel tiers
     */
    private void extractSubMatches(ElanMatch match, ArrayList topAnnos,
        ArrayList topTiers) {
        ElanMatch chMatch;
        Annotation ann;

        for (int j = 0; j < match.getChildCount(); j++) {
            chMatch = (ElanMatch) match.getChildAt(j);
            ann = chMatch.getAnnotation();
            ann = rootAnnotationOf(ann);

            if ((ann != null) && !topAnnos.contains(ann)) {
                topAnnos.add(ann);
            }

            if ((ann != null) && !topTiers.contains(ann.getTier())) {
                topTiers.add(ann.getTier());
            }

            // recursive extraction
            if (!chMatch.isLeaf()) {
                extractSubMatches(chMatch, topAnnos, topTiers);
            }
        }
    }

    /**
     * Copies media and linked files descriptors.
     * @param transcription source transcription
     * @param nextTrans destination transcription
     */
    private void copyDescriptors(TranscriptionImpl transcription,
        TranscriptionImpl nextTrans) {
        if ((transcription == null) || (nextTrans == null)) {
            return;
        }

        Vector mds = transcription.getMediaDescriptors();
        Vector cmds = new Vector(mds.size());
        MediaDescriptor md;

        for (int i = 0; i < mds.size(); i++) {
            md = (MediaDescriptor) mds.get(i);
            cmds.add((MediaDescriptor) md.clone());
        }

        nextTrans.setMediaDescriptors(cmds);

        // copy/clone linked files descriptors
        Vector lfds = transcription.getLinkedFileDescriptors();
        Vector clfds = new Vector(lfds.size());
        LinkedFileDescriptor lfd;

        for (int i = 0; i < lfds.size(); i++) {
            lfd = (LinkedFileDescriptor) lfds.get(i);
            clfds.add((LinkedFileDescriptor) lfd.clone());
        }

        nextTrans.setLinkedFileDescriptors(clfds);
        progressUpdate(null, 20);
    }

    /**
     * Copies tiers, linguistic types and cv's.
     * @param transcription source transcription
     * @param nextTrans destination transcription
     * @param topTiers a list of top level tiers, all depending tiers will also be copied
     */
    private void copyTiersTypesCvs(TranscriptionImpl transcription,
        TranscriptionImpl nextTrans, ArrayList topTiers) {
        if ((transcription == null) || (nextTrans == null) ||
                (topTiers == null)) {
            return;
        }

        TierImpl tier;
        ArrayList tiersToCopy = new ArrayList();

        for (int i = 0; i < topTiers.size(); i++) {
            tier = (TierImpl) topTiers.get(i);
            tiersToCopy.add(tier);
            tiersToCopy.addAll(tier.getDependentTiers());
        }

        ArrayList typesToCopy = new ArrayList();
        ArrayList cvsToCopy = new ArrayList();
        ControlledVocabulary cv;
        String cvName;

        for (int i = 0; i < tiersToCopy.size(); i++) {
            tier = (TierImpl) tiersToCopy.get(i);

            if (!typesToCopy.contains(tier.getLinguisticType())) {
                typesToCopy.add(tier.getLinguisticType());

                if (tier.getLinguisticType().isUsingControlledVocabulary()) {
                    cvName = tier.getLinguisticType()
                                 .getControlledVocabylaryName();
                    cv = transcription.getControlledVocabulary(cvName);

                    if ((cv != null) && !cvsToCopy.contains(cv)) {
                        cvsToCopy.add(cv);
                    }
                }
            }
        }

        // copy cv's
        Vector cvc = new Vector(cvsToCopy.size());
        ControlledVocabulary cv1;
        ControlledVocabulary cv2;
        CVEntry[] entries;
        CVEntry ent1;
        CVEntry ent2;

        for (int i = 0; i < cvsToCopy.size(); i++) {
            cv1 = (ControlledVocabulary) cvsToCopy.get(i);
            cv2 = new ControlledVocabulary(cv1.getName(), cv1.getDescription());
            entries = cv1.getEntries();

            for (int j = 0; j < entries.length; j++) {
                ent1 = entries[j];
                ent2 = new CVEntry(ent1.getValue(), ent1.getDescription());
                cv2.addEntry(ent2);
            }

            cvc.add(cv2);
        }

        nextTrans.setControlledVocabularies(cvc);

        // copy ling types
        Vector typc = new Vector(typesToCopy.size());
        LinguisticType lt1;
        LinguisticType lt2;
        Constraint con1;
        Constraint con2 = null;

        for (int i = 0; i < typesToCopy.size(); i++) {
            lt1 = (LinguisticType) typesToCopy.get(i);
            lt2 = new LinguisticType(lt1.getLinguisticTypeName());
            lt2.setTimeAlignable(lt1.isTimeAlignable());
            lt2.setGraphicReferences(lt1.hasGraphicReferences());
            lt2.setControlledVocabularyName(lt1.getControlledVocabylaryName());
            con1 = lt1.getConstraints();

            if (con1 != null) {
                switch (con1.getStereoType()) {
                case Constraint.TIME_SUBDIVISION:
                    con2 = new TimeSubdivision();

                    break;

                case Constraint.SYMBOLIC_SUBDIVISION:
                    con2 = new SymbolicSubdivision();

                    break;

                case Constraint.SYMBOLIC_ASSOCIATION:
                    con2 = new SymbolicAssociation();

                    break;

                case Constraint.INCLUDED_IN:
                    con2 = new IncludedIn();
                }

                lt2.addConstraint(con2);
            }

            typc.add(lt2);
        }

        nextTrans.setLinguisticTypes(typc);

        // copy tiers
        TierTree tree = new TierTree(transcription);
        DefaultMutableTreeNode root = tree.getTree();
        DefaultMutableTreeNode node;
        TierImpl t1;
        TierImpl t2;
        TierImpl copyTier;
        String name;
        String parentName;

        Enumeration en = root.breadthFirstEnumeration();
        en.nextElement();

        while (en.hasMoreElements()) {
            copyTier = null; // reset
            node = (DefaultMutableTreeNode) en.nextElement();
            name = (String) node.getUserObject();
            t1 = (TierImpl) transcription.getTierWithId(name);

            if (!tiersToCopy.contains(t1)) {
                continue;
            }

            if (t1 != null) {
                lt1 = t1.getLinguisticType();
                lt2 = nextTrans.getLinguisticTypeByName(lt1.getLinguisticTypeName());

                if (lt2 != null) {
                    if (t1.hasParentTier()) {
                        parentName = t1.getParentTier().getName();
                        t2 = (TierImpl) nextTrans.getTierWithId(parentName);

                        if (t2 != null) {
                            copyTier = new TierImpl(t2, name,
                                    t1.getParticipant(), nextTrans, lt2);
                            copyTier.setDefaultLocale(t1.getDefaultLocale());
                            copyTier.setAnnotator(t1.getAnnotator());
                        } else {
                            LOG.warning("The parent tier: " + parentName +
                                " for tier: " + name +
                                " was not found in the destination transcription");
                        }
                    } else {
                        copyTier = new TierImpl(name, t1.getParticipant(),
                                nextTrans, lt2);
                        copyTier.setDefaultLocale(t1.getDefaultLocale());
                        copyTier.setAnnotator(t1.getAnnotator());
                    }
                } else {
                    LOG.warning("Could not add tier: " + name +
                        " because the Linguistic Type was not found in the destination transcription.");
                }
            }

            if (copyTier != null) {
                nextTrans.addTier(copyTier);
            }
        }

        progressUpdate(null, 30);
    }

    /**
     * Copies (toplevel) annotations and all depending annotations.
     * @param transcription source transcription
     * @param nextTrans destination transcription
     * @param topAnnos the toplevel annotations
     */
    private void copyAnnotations(TranscriptionImpl transcription,
        TranscriptionImpl nextTrans, ArrayList topAnnos) {
        if ((transcription == null) || (nextTrans == null) ||
                (topAnnos == null)) {
            return;
        }

        // copy the annotations
        AlignableAnnotation aa;
        AlignableAnnotation copyAnn;
        DefaultMutableTreeNode record;
        int tp = 90 - 30;
        float incr = (float) tp / Math.max(topAnnos.size(), 1); //prevent division by 0

        for (int i = 0; i < topAnnos.size(); i++) {
            aa = (AlignableAnnotation) topAnnos.get(i);
            record = AnnotationRecreator.createTreeForAnnotation(aa);
            copyAnn = (AlignableAnnotation) AnnotationRecreator.createAnnotationFromTree(nextTrans,
                    record);

            if (copyAnn == null) {
                LOG.warning("Could not copy annotation: " + aa.getValue() +
                    " (" + aa.getBeginTimeBoundary() + " - " +
                    aa.getEndTimeBoundary() + ")");
            }

            progressUpdate(null, (30 + (int) (i * incr)));

            if (isCancelled()) {
                return;
            }
        }
    }

    /**
     * Finds the root or toplevel annotation for the specified annotation.
     * @param ann the annotation to find the root for
     * @return the toplevel annotation
     */
    private AlignableAnnotation rootAnnotationOf(Annotation ann) {
        if (ann == null) {
            return null;
        }

        TierImpl tier = (TierImpl) ann.getTier();

        if (tier.hasParentTier()) {
            tier = tier.getRootTier();

            return (AlignableAnnotation) tier.getAnnotationAtTime((ann.getBeginTimeBoundary() +
                ann.getEndTimeBoundary()) / 2);
        } else {
            return (AlignableAnnotation) ann;
        }
    }

    /**
     * Prompt for a filename and location.
     *
     * @return a path as a string
     */
    private String promptForFileName() {
        String exportDir = (String) Preferences.get("LastUsedEAFDir", null);

        if (exportDir == null) {
            exportDir = System.getProperty("user.dir");
        }

        JFileChooser chooser = new JFileChooser();
        chooser.setCurrentDirectory(new File(exportDir));
        chooser.setDialogTitle(ElanLocale.getString("SaveDialog.Title"));

        File exportFile = null;
        FileFilter filter = ElanFileFilter.createFileFilter(ElanFileFilter.EAF_TYPE);
        chooser.setFileFilter(filter);

        if (chooser.showSaveDialog(null) == JFileChooser.APPROVE_OPTION) {
            File curDir = chooser.getCurrentDirectory();

            if (curDir != null) {
                Preferences.set("LastUsedEAFDir", curDir.getAbsolutePath(), null);
            }

            exportFile = chooser.getSelectedFile();

            if (exportFile != null) {
                String name = exportFile.getAbsolutePath();
                String lowerPathName = name.toLowerCase();

                String[] exts = FileExtension.EAF_EXT;
                boolean validExt = false;

                for (int i = 0; i < exts.length; i++) {
                    if (lowerPathName.endsWith("." + exts[i])) {
                        validExt = true;

                        break;
                    }
                }

                if (!validExt) {
                    name += ("." + exts[0]);
                    exportFile = new File(name);
                }

                if (exportFile.exists()) {
                    int answer = JOptionPane.showConfirmDialog(null,
                            ElanLocale.getString("Message.Overwrite"),
                            ElanLocale.getString("SaveDialog.Message.Title"),
                            JOptionPane.YES_NO_OPTION);

                    if (answer == JOptionPane.NO_OPTION) {
                        return promptForFileName();
                    } else {
                        return name;
                    }
                } else {
                    return name;
                }
            } else {
                return null;
            }
        } else {
            // save dialog canceled
            return null;
        }
    }

    /**
     * Updates the progress bar and the message of the monitor.
     * @param note the progress message
     * @param progress the progress value
     */
    private void progressUpdate(String note, int progress) {
        if (monitor != null) {
            if (note != null) {
                monitor.setNote(note);
            }

            monitor.setProgress(progress);
        }
    }

    /**
     * Checks whether the operation has been canceled via the progress monitor.
     *
     * @return true if the cancel button of the monitor has been clicked, false otherwise
     */
    private boolean isCancelled() {
        if (monitor != null) {
            return monitor.isCanceled();
        }

        return false;
    }
}
