package mpi.eudico.client.annotator.export;

import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.annotator.util.ElanFileFilter;
import mpi.eudico.client.annotator.util.FileExtension;

import mpi.eudico.client.util.CheckBoxTableCellRenderer;
import mpi.eudico.client.util.Transcription2WordList;

import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.ParseException;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.dobes.EAFSkeletonParser;

import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;

import java.io.File;
import java.io.IOException;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.swing.ButtonGroup;
import javax.swing.DefaultCellEditor;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;


/**
 * Tier selection dialog for export of a word list, either from a single file
 * or from multiple files.
 *
 * @author Han Sloetjes
 */
public class ExportWordListDialog extends AbstractTierExportDialog 
    implements ChangeListener {
    private List files;
    // hier, kan hiermee vastgelegd worden dat eenzelfde tiernaam verschillende types kan hebben in meerdere files?
    private Map<String, String> tierTypeMap = new LinkedHashMap<String, String>();
    private List<String> tierNames = new ArrayList<String>();
    private List<String> typeNames = new ArrayList<String>();
    private JRadioButton tierRB;
    private JRadioButton typeRB;
    
    private JRadioButton customDelimRB;
    private JLabel tokenDelimLabel;
    private JRadioButton defaultDelimRB;
    private JTextField customDelimField;
    private ButtonGroup delimButtonGroup;
    private JCheckBox countTokensCB;
    
    public static int WORDS = 0;
    public static int ANNOTATIONS = 1;
    
    private int mode = WORDS;

    /**
     * Constructor for single file.
     *
     * @param parent the parent frame
     * @param modal the modal flag
     * @param transcription the (single) transcription
     */
    public ExportWordListDialog(Frame parent, boolean modal,
        Transcription transcription) {
        super(parent, modal, transcription, null);
        makeLayout();
        extractTiers();
        postInit();
    }

    /**
     * Constructor for multiple files.
     *
     * @param parent the parent frame
     * @param modal the modal flag
     * @param files a list of eaf files
     */
    public ExportWordListDialog(Frame parent, boolean modal, List files) {
    	    this(parent, modal, files, WORDS);
    }

    /**
     * Constructor for multiple files.
     *
     * @param parent the parent frame
     * @param modal the modal flag
     * @param files a list of eaf files
     * @param mode the mode, WORDS or ANNOTATION export
     */
    public ExportWordListDialog(Frame parent, boolean modal, List files, int mode) {
        super(parent, modal, null, null);
        this.files = files;
        if (mode == ANNOTATIONS || mode == WORDS) {
        		this.mode = mode;
        }
        makeLayout();
        extractTiersFromFiles();
        postInit();
    }
    
    /**
     * Gets the tier names in case of "tiers of type" mode and calls the super implementation 
     * in case of tier by name mode.
     * 
	 * @see mpi.eudico.client.annotator.export.AbstractTierExportDialog#getSelectedTiers()
	 */
	@Override
	protected List getSelectedTiers() {
		if (typeRB.isSelected()) {
			List<String> selTierNames = new ArrayList<String>();
			
	        if (model != null) {
	            int includeCol = model.findColumn(EXPORT_COLUMN);
	            int nameCol = model.findColumn(TIER_NAME_COLUMN);
	            ArrayList<String> selectedTypes = new ArrayList<String>();
	            // get selected types
	            for (int i = 0; i < model.getRowCount(); i++) {
	                Boolean include = (Boolean) model.getValueAt(i, includeCol);

	                if (include.booleanValue()) {
	                	selectedTypes.add((String) model.getValueAt(i, nameCol));
	                }
	            }
	    		String nt;
	    		boolean select;
	            for (int i = 0; i < tierNames.size(); i++) {
	            	nt = tierNames.get(i);
	            	select = selectedTypes.contains(tierTypeMap.get(nt));
	            	
            		if (select) {
            			selTierNames.add(nt);
            		}

	            }
	        }
	        
	        return selTierNames;
		} else {
			return super.getSelectedTiers();
		}
	}

	/**
     * Extract candidate tiers for export.
     */
    protected void extractTiers() {
        if (model != null) {
            for (int i = model.getRowCount() - 1; i >= 0; i--) {
                model.removeRow(i);
            }

            if (transcription != null) {
                Vector v = transcription.getTiers();
                TierImpl t;

                for (int i = 0; i < v.size(); i++) {
                    t = (TierImpl) v.get(i);
                    tierNames.add(t.getName());
                    if (!typeNames.contains(t.getLinguisticType().getLinguisticTypeName())) {
                    	typeNames.add(t.getLinguisticType().getLinguisticTypeName());
                    }
                    tierTypeMap.put(t.getName(), t.getLinguisticType().getLinguisticTypeName());

                    // add tiers first
                    model.addRow(new Object[] { Boolean.TRUE, t.getName() });
                }
            }
        }
    }

    /**
     * Extracts all unique tiers from multiple files.
     */
    private void extractTiersFromFiles() {
        if ((files == null) || (files.size() == 0)) {
            return;
        }

        if (model != null) {
            for (int i = model.getRowCount() - 1; i >= 0; i--) {
                model.removeRow(i);
            }

            //ArrayList tierNames = new ArrayList();
            ArrayList pts = null;
            EAFSkeletonParser parser = null;
            String path;

            for (int i = 0; i < files.size(); i++) {
                path = ((File) files.get(i)).getAbsolutePath();

                try {
                    parser = new EAFSkeletonParser(path);
                    parser.parse();
                    pts = parser.getTiers();

                    TierImpl t;

                    for (int j = 0; j < pts.size(); j++) {
                        t = (TierImpl) pts.get(j);

                        if (!tierNames.contains(t.getName())) {
                            tierNames.add(t.getName());
                        }
                        if (!tierTypeMap.containsKey(t.getName())) {
                        	tierTypeMap.put(t.getName(), t.getLinguisticType().getLinguisticTypeName());
                        }
                        if (!typeNames.contains(t.getLinguisticType().getLinguisticTypeName())) {
                        	typeNames.add(t.getLinguisticType().getLinguisticTypeName());
                        }
                    }
                } catch (ParseException pe) {
                    LOG.warning(pe.getMessage());

                    //pe.printStackTrace();
                } catch (Exception ex) {
                    LOG.warning("Could not load file: " + path);
                }
            }

            for (int i = 0; i < tierNames.size(); i++) {
                model.addRow(new Object[] { Boolean.TRUE, tierNames.get(i) });
            }
        }
    }
    
    private void updateTable() {
        if (model != null) {
            int includeCol = model.findColumn(EXPORT_COLUMN);
            int nameCol = model.findColumn(TIER_NAME_COLUMN);

            ArrayList<String> selectedTiersOrTypes = new ArrayList<String>();

            // add selected tiers in the right order
            for (int i = 0; i < model.getRowCount(); i++) {
                Boolean include = (Boolean) model.getValueAt(i, includeCol);

                if (include.booleanValue()) {
                	selectedTiersOrTypes.add((String) model.getValueAt(i, nameCol));
                }
            }
            
            for (int i = model.getRowCount() - 1; i >= 0; i--) {
                model.removeRow(i);
            }
            boolean select;
	    	if (tierRB.isSelected()) {
	    		// previously types were selected..	    		
	    		String nt;
	            for (int i = 0; i < tierNames.size(); i++) {
	            	nt = tierNames.get(i);
	            	select = selectedTiersOrTypes.contains(tierTypeMap.get(nt));
	            	
            		model.addRow(new Object[] { new Boolean(select), nt });

	            }
	    	} else {
	    		// previously tiers were selected
	    		String ntype;
	    		ArrayList<String> selTypes = new ArrayList<String>(10);
	    		for (int i = 0; i < selectedTiersOrTypes.size(); i++) {
	    			ntype = tierTypeMap.get(selectedTiersOrTypes.get(i));
	    			if (!selTypes.contains(ntype)) {
	    				selTypes.add(ntype);
	    			}
	    		}
	    		
	            for (int i = 0; i < typeNames.size(); i++) {
	            	ntype = typeNames.get(i);
	            	select = selTypes.contains(ntype);
	            	
	                model.addRow(new Object[] { new Boolean(select), typeNames.get(i) });
	            }
	    	}
        }
    }

    /**
     * Calls the super implementation and sets some properties of the tier
     * table.
     */
    protected void makeLayout() {
        super.makeLayout();
        JPanel tierButtonPanel = new JPanel(new GridBagLayout());
        tierRB = new JRadioButton(ElanLocale.getString("ExportDialog.Label.TiersByName"));
        tierRB.setSelected(true);
        typeRB = new JRadioButton(ElanLocale.getString("ExportDialog.Label.TiersOfType"));
        ButtonGroup bg = new ButtonGroup();
        bg.add(tierRB);
        bg.add(typeRB);
        tierRB.addActionListener(this);
        typeRB.addActionListener(this);
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.fill = GridBagConstraints.NONE;
        gbc.gridx = 0;
        gbc.gridy = 0;
        tierButtonPanel.add(tierRB, gbc);
        gbc.gridx = 1;
        tierButtonPanel.add(typeRB, gbc);
        gbc.gridx = 2;
        gbc.weightx = 1.0;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        tierButtonPanel.add(new JPanel(), gbc);       
        gbc.gridx = 1;
        gbc.gridwidth = 2;
        gbc.insets = insets;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        tierSelectionPanel.add(tierButtonPanel, gbc);
        
        model.setColumnIdentifiers(new String[] { EXPORT_COLUMN, TIER_NAME_COLUMN });
        tierTable.getColumn(EXPORT_COLUMN)
                 .setCellEditor(new DefaultCellEditor(new JCheckBox()));
        tierTable.getColumn(EXPORT_COLUMN)
                 .setCellRenderer(new CheckBoxTableCellRenderer());
        tierTable.getColumn(EXPORT_COLUMN).setMaxWidth(30);
        tierTable.setShowVerticalLines(false);
        tierTable.setTableHeader(null);
        countTokensCB = new JCheckBox();
        
        // options
        if (mode == WORDS) {
            delimButtonGroup = new ButtonGroup();
            tokenDelimLabel = new JLabel();
            defaultDelimRB = new JRadioButton();
            customDelimRB = new JRadioButton();
            customDelimField = new JTextField();
            
            GridBagConstraints gridBagConstraints;
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = 0;
            gridBagConstraints.gridwidth = 2;
            gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.weightx = 1.0;
            gridBagConstraints.insets = insets;
            optionsPanel.add(tokenDelimLabel, gridBagConstraints);

            defaultDelimRB.setSelected(true);
            defaultDelimRB.addChangeListener(this);
            delimButtonGroup.add(defaultDelimRB);
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = 1;
            gridBagConstraints.gridwidth = 2;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.insets = insets;
            optionsPanel.add(defaultDelimRB, gridBagConstraints);

            customDelimRB.addChangeListener(this);
            delimButtonGroup.add(customDelimRB);
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = 2;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.insets = insets;
            optionsPanel.add(customDelimRB, gridBagConstraints);

            customDelimField.setEnabled(false);
            customDelimField.setColumns(6);
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 1;
            gridBagConstraints.gridy = 2;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.insets = insets;
            optionsPanel.add(customDelimField, gridBagConstraints);
            
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = 3;
            gridBagConstraints.gridwidth = 2;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.insets = new Insets(12, 6, 4, 6);
            optionsPanel.add(countTokensCB, gridBagConstraints);

        } else {
            GridBagConstraints gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = 0;
            gridBagConstraints.gridwidth = 2;
            gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.weightx = 1.0;
            gridBagConstraints.insets = insets;
            optionsPanel.add(countTokensCB, gridBagConstraints);
            
            //getContentPane().remove(optionsPanel);
        }
        
        updateLocale();
    }

    /**
     * Applies localized text values ui elements.
     */
    protected void updateLocale() {
        super.updateLocale();
        titleLabel.setText(ElanLocale.getString("ExportDialog.WordList.Title"));
        
        if (mode == WORDS) {
        	    titleLabel.setText(ElanLocale.getString("ExportDialog.WordList.Title"));
	        	tokenDelimLabel.setText(ElanLocale.getString(
	            "TokenizeDialog.Label.TokenDelimiter"));
	        defaultDelimRB.setText(ElanLocale.getString(
	            "Button.Default") + "( . , ! ? \" \' )");
	        customDelimRB.setText(ElanLocale.getString(
	            "TokenizeDialog.RadioButton.Custom"));
        } else if (mode == ANNOTATIONS) {
            titleLabel.setText(ElanLocale.getString("ExportDialog.AnnotationList.Title"));
        }
        countTokensCB.setText(ElanLocale.getString("ExportDialog.WordList.CountOccur"));
    }

    /**
     * @see mpi.eudico.client.annotator.export.AbstractBasicExportDialog#startExport()
     */
    protected boolean startExport() throws IOException {
        List selectedTiers = getSelectedTiers();

        if (selectedTiers.size() == 0) {
            JOptionPane.showMessageDialog(this,
                ElanLocale.getString("ExportTradTranscript.Message.NoTiers"),
                ElanLocale.getString("Message.Warning"),
                JOptionPane.WARNING_MESSAGE);

            return false;
        }

        // prompt for file name and location
        File exportFile = null;
        if (mode == WORDS) {
	        	exportFile = promptForFile(ElanLocale.getString(
	            "ExportDialog.WordList.Title"), FileExtension.TEXT_EXT,
	        ElanFileFilter.createFileFilter(ElanFileFilter.TEXT_TYPE), true);      	
        } else if (mode == ANNOTATIONS) {
            exportFile = promptForFile(ElanLocale.getString(
                "ExportDialog.AnnotationList.Title"), FileExtension.TEXT_EXT,
            ElanFileFilter.createFileFilter(ElanFileFilter.TEXT_TYPE), true);
        }

        if (exportFile == null) {
            return false;
        }
        String delimiters = null;
        if (mode == WORDS && customDelimRB.isSelected()) {
            delimiters = customDelimField.getText();
        }
        boolean countOccurrences = countTokensCB.isSelected();
        
        Transcription2WordList twl = new Transcription2WordList();

        try {
            if (transcription != null) {
                twl.exportWords(transcription, selectedTiers, exportFile,
                    encoding, delimiters, countOccurrences);
            } else {
            	   if (mode == ANNOTATIONS) {
            		   twl.exportWords(files, selectedTiers, exportFile, encoding, new String(""), countOccurrences);   
            	   } else if (mode == WORDS) {
            		   twl.exportWords(files, selectedTiers, exportFile, encoding, delimiters, countOccurrences);   
            	   }
            }
        } catch (IOException ioe) {
            JOptionPane.showMessageDialog(this,
                ElanLocale.getString("ExportDialog.Message.Error"),
                ElanLocale.getString("Message.Error"),
                JOptionPane.WARNING_MESSAGE);

            return false;
        }

        return true;
    }
    
    /* (non-Javadoc)
	 * @see mpi.eudico.client.annotator.export.AbstractBasicExportDialog#actionPerformed(java.awt.event.ActionEvent)
	 */
	@Override
	public void actionPerformed(ActionEvent ae) {
		if (ae.getSource() == tierRB) {
			updateTable();
		} else if (ae.getSource() == typeRB) {
			updateTable();
		} else {
			super.actionPerformed(ae);
		}
	}

	/**
     * The state changed event handling.
     *
     * @param ce the change event
     */
    public void stateChanged(ChangeEvent ce) {
    	if (ce.getSource() == defaultDelimRB || ce.getSource() == customDelimField) {
	        if (defaultDelimRB.isSelected()) {
	            customDelimField.setEnabled(false);
	        } else {
	            customDelimField.setEnabled(true);
	            customDelimField.requestFocus();
	        }
    	}
    }
}
