/*
 * File:     ImageExporter.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.export;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.util.ElanFileFilter;

import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.image.BufferedImage;

import java.io.File;
import java.io.IOException;

import javax.imageio.ImageIO;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;


/**
 * A class to export a BufferedImage to file.
 *
 * @author Han Sloetjes
 */
public class ImageExporter {
    /** Holds value of property DOCUMENT ME! */
    Frame parentFrame = null;

    /**
     * Creates a new ImageExporter instance
     */
    public ImageExporter() {
    }

    /**
     * Creates a new ImageExporter instance
     */
    public ImageExporter(Frame frame) {
        parentFrame = frame;
    }

    /**
     * Shows a save dialog with .jpg and .png as file filters and  writes the
     * image to file.
     *
     * @param image theimage to save
     */
    public void exportImage(Image image) {
        BufferedImage img;

        if (!(image instanceof BufferedImage)) {
            img = imageToBufferedImage(image);
        } else {
            img = (BufferedImage) image;
        }

        if (img == null) {
            JOptionPane.showMessageDialog(parentFrame,
                ElanLocale.getString("ImageExporter.Message.NoImage"),
                ElanLocale.getString("Message.Warning"),
                JOptionPane.WARNING_MESSAGE);

            return;
        }

        String saveDir = (String) Preferences.get("MediaDir", null);

        if (saveDir == null) {
            saveDir = System.getProperty("user.dir");
        }

        JFileChooser chooser = new JFileChooser(saveDir);
        chooser.setFileFilter(ElanFileFilter.createFileFilter(
                ElanFileFilter.IMAGE_TYPE));

        int option = chooser.showSaveDialog(parentFrame);

        if (option == JFileChooser.APPROVE_OPTION) {
            String imageIOType = "jpg";
            File curDir = chooser.getCurrentDirectory();

            if (curDir != null) {
                Preferences.set("MediaDir", curDir.getAbsolutePath(), null);
            }

            File saveFile = chooser.getSelectedFile();

            if (saveFile != null) {
                String fileName = saveFile.getAbsolutePath();
                String lowerFileName = fileName.toLowerCase();

                if (lowerFileName.endsWith("png")) {
                    imageIOType = "png";
                } else if (lowerFileName.endsWith("bmp")) {
                    imageIOType = "bmp";
                } else if (!lowerFileName.endsWith("jpg") &&
                        !lowerFileName.endsWith("jpeg")) {
                    fileName += ".jpg";
                }

                final File newSaveFile = new File(fileName);

                if (newSaveFile.exists()) {
                    int answer = JOptionPane.showConfirmDialog(parentFrame,
                            ElanLocale.getString("Message.Overwrite"),
                            ElanLocale.getString("SaveDialog.Message.Title"),
                            JOptionPane.YES_NO_OPTION);

                    if (answer == JOptionPane.NO_OPTION) {
                        return;
                    }
                }

                // Cocoa QT currently delivers png images with alpha. Remove alpha in case of saving as jpg or bmp
                // implement a more elegant way of fixing the transparency later...
                if (img.getColorModel().hasAlpha() &&
                        !imageIOType.equals("png")) {
                    BufferedImage flatImg = new BufferedImage(img.getWidth(),
                            img.getHeight(), BufferedImage.TYPE_INT_RGB);
                    flatImg.getGraphics().drawImage(img, 0, 0, null);
                    img = flatImg;
                }

                try {
                    ImageIO.write(img, imageIOType, newSaveFile);
                } catch (IOException ioe) {
                    ioe.printStackTrace();
                    JOptionPane.showMessageDialog(parentFrame,
                        ElanLocale.getString("ExportDialog.Message.Error"),
                        ElanLocale.getString("Message.Error"),
                        JOptionPane.ERROR_MESSAGE);
                }
            }
        }
    }

    private BufferedImage imageToBufferedImage(Image image) {
        if (image == null) {
            return null;
        }

        BufferedImage bufImg = new BufferedImage(image.getWidth(null),
                image.getHeight(null), BufferedImage.TYPE_INT_RGB);
        Graphics g = bufImg.createGraphics();
        g.drawImage(image, 0, 0, null);
        g.dispose();

        return bufImg;
    }
}
