/*
 * File:     AnnotationTable.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.grid;

import mpi.eudico.client.annotator.Constants;
import mpi.eudico.client.annotator.ElanLocale;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.KeyEvent;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.TableModelEvent;
import javax.swing.table.TableColumn;
import javax.swing.table.TableModel;


/**
 * JTable for showing Annotations
 * Extracted from GridViewer on Jun 29, 2004
 * @author Alexander Klassmann
 * @version Jun 29, 2004
 */
public class AnnotationTable extends JTable {
    private static int MIN_COLUMN_WIDTH = 15; // swing default

    /** stores default minimal widths of columns */
    private static HashMap preferredWidths = new HashMap();

    /** stores default maximal widths of columns */
    private static HashMap maxWidths = new HashMap();

    //annotation columns don't have a default width; cf adjustAnnotationColumns()
    static {
        preferredWidths.put(GridViewerTableModel.TIMEPOINT, new Integer(15));
        preferredWidths.put(GridViewerTableModel.COUNT, new Integer(40));
        preferredWidths.put(GridViewerTableModel.FILENAME, new Integer(100));
        preferredWidths.put(GridViewerTableModel.TIERNAME, new Integer(80));
        preferredWidths.put(GridViewerTableModel.LEFTCONTEXT, new Integer(100));
        preferredWidths.put(GridViewerTableModel.RIGHTCONTEXT, new Integer(100));
        preferredWidths.put(GridViewerTableModel.BEGINTIME, new Integer(80));
        preferredWidths.put(GridViewerTableModel.ENDTIME, new Integer(80));
        preferredWidths.put(GridViewerTableModel.DURATION, new Integer(80));
        maxWidths.put(GridViewerTableModel.TIMEPOINT, new Integer(15));
        maxWidths.put(GridViewerTableModel.COUNT, new Integer(40));
        maxWidths.put(GridViewerTableModel.BEGINTIME, new Integer(120));
        maxWidths.put(GridViewerTableModel.ENDTIME, new Integer(120));
        maxWidths.put(GridViewerTableModel.DURATION, new Integer(120));
    }

    /** stores width of parent component */
    private int width = -1;

    /** preferred font per tier */
    private HashMap prefTierFonts;
    private boolean deselectCommits = false;

    /**
     *
     * @see javax.swing.JTable#JTable(TableModel)
     */
    public AnnotationTable(TableModel dataModel) {
        super(dataModel);

        prefTierFonts = new HashMap();
        setFont(Constants.DEFAULTFONT);
        setSelectionMode(ListSelectionModel.SINGLE_INTERVAL_SELECTION);
        setAutoResizeMode(JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);

        for (int i = 0; i < dataModel.getColumnCount(); i++) {
            TableColumn column = getColumnModel().getColumn(i);
            String columnName = dataModel.getColumnName(i);
            column.setIdentifier(columnName);
            setColumnVisible(columnName, true);
        }

        getColumn(GridViewerTableModel.TIMEPOINT).setResizable(false);
        getColumn(GridViewerTableModel.COUNT).setResizable(false);

        getTableHeader().setReorderingAllowed(false);

        //after this component gets visible, add component listener to parent component		
        addComponentListener(new ComponentAdapter() {
                public void componentResized(ComponentEvent e) {
                    //table is visible ?
                    if (getWidth() > 0) {
                        //component listener to parent component (e.g. viewport of scrollpane)
                        getParent().addComponentListener(new ComponentAdapter() {
                                //adjust columns if width of viewport changes
                                public void componentResized(ComponentEvent e) {
                                    if (getParent().getWidth() != width) {
                                        width = getParent().getWidth();
                                        adjustAnnotationColumns();
                                    }
                                }
                            });
                        removeComponentListener(this);
                        adjustAnnotationColumns();
                    }
                }
            });
    }

    /**
     * DOCUMENT ME!
     *
     * @param ks DOCUMENT ME!
     * @param e DOCUMENT ME!
     * @param condition DOCUMENT ME!
     * @param pressed DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    protected boolean processKeyBinding(KeyStroke ks, KeyEvent e,
        int condition, boolean pressed) {
        return false;
    }

    /**
     * Override to set rendering hints.
     */

    /*
    protected void paintComponent(Graphics g) {
         only for J 1.4, new solutions for 1.5 and 1.6
        if (g instanceof Graphics2D) {
            ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
                    RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
        }
        super.paintComponent(g);
    }
    */
    public Dimension getPreferredScrollableViewportSize() {
        return new Dimension(640, 200);
    }

    /**
     * Restores the default row height after an inline edit operation.
     * If this method is called by the GridEditor, the editor may be called to commit
     * instead of cancel, depending on a preference flag.
     *
     * @param e the change event
     */
    public void editingStopped(ChangeEvent e) {
        if (deselectCommits && e.getSource() instanceof GridEditor) {
            ((GridEditor) e.getSource()).commitEdit();
        }

        super.editingStopped(e);
        setRowHeight(getRowHeight());
    }

    /**
     * Restores the default row height after an inline edit operation.
     *
     * @param e the change event
     */
    public void editingCanceled(ChangeEvent e) {
        super.editingCanceled(e);
        setRowHeight(getRowHeight());
    }

    /**
     * sets column width default value if true, to 0 if false
     * @param columnName
     * @param visible
     */
    public void setColumnVisible(String columnName, boolean visible) {
        TableColumn column = null;

        try {
            column = getColumn(columnName);

            // attention: don't change order of settings! A.K.
            if (visible) {
                column.setResizable(true);
                column.setMaxWidth(getDefaultMaxWidth(columnName));
                column.setMinWidth(MIN_COLUMN_WIDTH);
                column.setPreferredWidth(getDefaultPreferredWidth(columnName));
            } else {
                column.setMinWidth(0);
                column.setMaxWidth(0); // -> preferredWidth = 0
                column.setResizable(false);
            }
        } catch (IllegalArgumentException e) {
            System.out.println("Warning : no column with name " + columnName);
        }
    }

    /**
     * distributes remaining table space among annotation columns
     * to be called after other columns had been set/removed or component resized
     */
    protected void adjustAnnotationColumns() {
        if (getParent() != null) {
            Set visibleAnnotationColumns = new HashSet();
            int sumOfOtherWidths = 0;

            for (int i = 0; i < getColumnCount(); i++) {
                TableColumn column = getColumnModel().getColumn(i);

                if (!preferredWidths.containsKey(column.getIdentifier()) &&
                        (column.getMaxWidth() > 0)) {
                    visibleAnnotationColumns.add(column);
                } else {
                    sumOfOtherWidths += column.getPreferredWidth();
                }
            }

            int remainingSpace = getParent().getWidth() - sumOfOtherWidths;

            if (remainingSpace > 0) {
                for (Iterator iter = visibleAnnotationColumns.iterator();
                        iter.hasNext();) {
                    ((TableColumn) iter.next()).setPreferredWidth(remainingSpace / visibleAnnotationColumns.size());
                }
            }
        }
    }

    /**
     * returns if column is visible (e.g. width>0)
     * @param columnName
     * @return boolean
     */
    public boolean isColumnVisible(String columnName) {
        TableColumn column = null;

        try {
            column = getColumn(columnName);
        } catch (IllegalArgumentException e) {
            System.out.println("Warning : no column with name " + columnName);
        }

        return (column != null) ? (column.getWidth() > 0) : false;
    }

    /**
     * Returns default width for columns.
     * @param columnName the column name
     * @return the default preferred width
     */
    private int getDefaultPreferredWidth(String columnName) {
        return preferredWidths.containsKey(columnName)
        ? ((Integer) preferredWidths.get(columnName)).intValue()
        : MIN_COLUMN_WIDTH;

        //swing default
    }

    /**
     * Returns default width for columns.
     * @param columnName the column name
     * @return the default max width
     */
    private int getDefaultMaxWidth(String columnName) {
        return maxWidths.containsKey(columnName)
        ? ((Integer) maxWidths.get(columnName)).intValue() : Integer.MAX_VALUE;

        //swing default
    }

    /**
      * method from ElanLocaleListener not implemented in AbstractViewer
      */
    public void updateLocale() {
        for (int i = 0; i < dataModel.getColumnCount(); i++) {
            int index = getColumnModel().getColumnIndex(dataModel.getColumnName(
                        i));
            getColumnModel().getColumn(index).setHeaderValue(ElanLocale.getString(
                    "Frame.GridFrame." + dataModel.getColumnName(i)));
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param fontSize DOCUMENT ME!
     */
    public void setFontSize(int fontSize) {
        setFont(Constants.DEFAULTFONT.deriveFont((float) fontSize));
        setRowHeight((((fontSize + 2) < 16) ? 16 : (fontSize + 3)));

        // update preferred fonts
        if (prefTierFonts.size() > 0) {
            Object key;
            Font f;
            Iterator keyIt = prefTierFonts.keySet().iterator();

            while (keyIt.hasNext()) {
                key = keyIt.next();
                f = (Font) prefTierFonts.get(key);

                if (f != null) {
                    prefTierFonts.put(key, f.deriveFont((float) fontSize));
                }
            }
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public int getFontSize() {
        return getFont().getSize();
    }

    /**
     * Sets the names of the fonts for tiers. Based on this map and the current
     * fontsize Font objects are created.
     *
     * @param fontNames a map of tier name to font name<br>
     * NB the table model identifier for the annotations ofthe main tier in the
     * table is used for that tier instead of its name
     */
    public void setFontsForTiers(HashMap fontNames) {
        prefTierFonts.clear();

        if (fontNames != null) {
            Iterator keyIt = fontNames.keySet().iterator();
            String key;
            String fn;

            while (keyIt.hasNext()) {
                key = (String) keyIt.next();
                fn = (String) fontNames.get(key);

                if (fn != null) {
                    prefTierFonts.put(key,
                        new Font(fn, Font.PLAIN, getFontSize()));
                }
            }
        }
    }

    /**
     * Returns the font for the specified column. Only applicable for annotation
     * columns.
     *
     * @param column the column index
     * @return the (preferred) font
     */
    public Font getFontForColumn(int column) {
        if (dataModel instanceof GridViewerTableModel &&
                (prefTierFonts.size() > 0)) {
            String name = ((GridViewerTableModel) dataModel).getColumnName(column);
            Font f = (Font) prefTierFonts.get(name);

            if (f == null) {
                return getFont();
            } else {
                return f;
            }
        }

        return getFont();
    }

    /**
     * toggles string representation of time (HH:MM:SS.sss versus milliseconds)
     */
    public void toggleTimeFormat() {
        if (dataModel instanceof GridViewerTableModel) {
            String timeFormat = GridViewerTableModel.HHMMSSsss.equals(((GridViewerTableModel) dataModel).getTimeFormat())
                ? GridViewerTableModel.MILLISECONDS
                : GridViewerTableModel.HHMMSSsss;
            ((GridViewerTableModel) dataModel).setTimeFormat(timeFormat);

            repaint();
        }
    }

    /**
     * Sets the time format for the table. Supported are hh:mm:ss:ms, pal, ntsc, ms
     *
     * @param format the format
     */
    public void setTimeFormat(String format) {
        if (format != null) {
            ((GridViewerTableModel) dataModel).setTimeFormat(format);

            repaint();
        }
    }

    /**
     * invokes adjustment of annotation columns
     * @see javax.swing.event.TableModelListener#tableChanged(TableModelEvent)
     */
    public void tableChanged(TableModelEvent e) {
        super.tableChanged(e);
        adjustAnnotationColumns();
    }

    /**
     * When true deselecting a cell that is being edited leads to a commit of the changes.
     * @param deselectCommits The deselectCommits to set.
     */
    public void setDeselectCommits(boolean deselectCommits) {
        this.deselectCommits = deselectCommits;
    }
}
