/*
 * File:     ShortcutEditPanel.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.gui;

import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.annotator.commands.ShortcutsUtil;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.AbstractButton;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;


/**
 * A panel to change / remove a single keyboard shortcut combination.
 * 
 * @author alekoe
 *
 */
public class ShortcutEditPanel extends JPanel implements KeyListener,
    ActionListener {
    /** Holds value of property DOCUMENT ME! */
    protected JTextField entryField;

    /** Holds value of property DOCUMENT ME! */
    protected KeyStroke newShortcut;

    /** Holds value of property DOCUMENT ME! */
    protected String newShortcutAsText;

    /** Holds value of property DOCUMENT ME! */
    protected KeyStroke currentShortcut;

    /** Holds value of property DOCUMENT ME! */
    protected String currentShortCutAsText;

    /** Holds value of property DOCUMENT ME! */
    protected String currentAction;

    /** Holds value of property DOCUMENT ME! */
    protected JButton cancelButton;

    /** Holds value of property DOCUMENT ME! */
    protected JButton okButton;

    /** Holds value of property DOCUMENT ME! */
    protected JButton removeButton;
    private ShortcutPanel shortcutPanel;

    /**
 * Constructor
 * @param shortcutPanel the calling shortcut panel
 * @param actionID the ID of the action for which this window is created
 * @param actionKey the KeyStroke currently assigned to this action
 */
    public ShortcutEditPanel(ShortcutPanel shortcutPanel, String actionID,
        KeyStroke actionKey) {
        super();

        this.shortcutPanel = shortcutPanel;
        this.currentAction = actionID;
        this.currentShortcut = actionKey;
        this.currentShortCutAsText = ShortcutsUtil.getInstance()
                                                  .getDescriptionForKeyStroke(currentShortcut);
        newShortcut = currentShortcut;

        JLabel descField = new JLabel(ElanLocale.getString(
                    "Shortcuts.Editor.Title"));
        add(descField);

        entryField = new JTextField();

        // how big should it be 
        entryField.setPreferredSize(new Dimension(170, 25));

        // configure as keylistener
        entryField.setFocusable(true);
        entryField.setFocusTraversalKeysEnabled(false);
        entryField.addKeyListener(this);
        entryField.setText(currentShortCutAsText);
        add(entryField);

        okButton = new JButton(ElanLocale.getString("Button.OK"));
        okButton.addActionListener(this);
        okButton.setVerticalTextPosition(AbstractButton.CENTER);
        okButton.setHorizontalTextPosition(AbstractButton.LEADING);
        add(okButton);

        removeButton = new JButton(ElanLocale.getString(
                    "Shortcuts.Button.Remove"));
        removeButton.addActionListener(this);
        removeButton.setVerticalTextPosition(AbstractButton.CENTER);
        removeButton.setHorizontalTextPosition(AbstractButton.LEADING);
        add(removeButton);

        cancelButton = new JButton(ElanLocale.getString("Button.Cancel"));
        cancelButton.addActionListener(this);
        cancelButton.setVerticalTextPosition(AbstractButton.CENTER);
        cancelButton.setHorizontalTextPosition(AbstractButton.LEADING);
        add(cancelButton);
    }

    /**
 * Returns the new value for the action or null if nothing changed.
 *  
 * @return the new KeyStroke for the action
 */
    public KeyStroke getValue() {
        return newShortcut;
    }

    /**
 * actionListener method that catches whether a button is pressed and either
 * closes the window and hands the new shortcut to the parent ShortcutPanel (OK button)
 * closes the window without changing anything (Cancel button)
 * empties the text field (Remove button)
 * 
 */
    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == cancelButton) {
            SwingUtilities.getWindowAncestor(this).setVisible(false);
        } else if (e.getSource() == removeButton) {
            newShortcut = null;
            newShortcutAsText = "";
            entryField.setText(newShortcutAsText);
            entryField.requestFocus();
        } else if (e.getSource() == okButton) {
            // new shortcut has been set, or it is unchanged
            SwingUtilities.getWindowAncestor(this).setVisible(false);

            if (shortcutPanel != null) {
                shortcutPanel.changeShortcut(newShortcut);
            }
        }
    }

    /**
 * 
 * creates the window for editing a specific shortcut and shows it to the user
 * 
 * @param owner the ShortcutPanel from which this method was called
 * @param actionID the ID of the action for which the shortcut should be changed
 */
    public static void createAndShowGUI(ShortcutPanel caller, String actionID) {
        //Create and set up the window.
        JDialog owner = (JDialog) javax.swing.SwingUtilities.getWindowAncestor(caller);
        final ShortcutsUtil scu = ShortcutsUtil.getInstance();
        String actionName = scu.getDescriptionForAction(actionID);
        KeyStroke actionKeyStroke = caller.allShortCuts.get(actionID);
        JDialog frame = new JDialog(owner,
                (ElanLocale.getString("Shortcuts.Editor.Label") + " '" +
                actionName + "'"), true);

        //Create and set up the content pane.
        ShortcutEditPanel newContentPane = new ShortcutEditPanel(caller,
                actionID, actionKeyStroke);

        newContentPane.setOpaque(true); //content panes must be opaque
        frame.setContentPane(newContentPane);

        //Display the window.
        frame.pack();
        frame.setLocationRelativeTo(owner);
        frame.setVisible(true);
    }

    /**
 * action that is fired every time a key is pressed
 * the pressed key(s) is/are saved in member variables
 */
    public void keyPressed(KeyEvent ke) {
        entryField.setText("");
        newShortcut = KeyStroke.getKeyStrokeForEvent(ke);

        String myKeyString = KeyEvent.getKeyText(ke.getKeyCode());
        String myModifierString = KeyEvent.getModifiersExText(ke.getModifiersEx());

        if (ke.getModifiersEx() == 0) {
            newShortcutAsText = myKeyString;
        } else {
            newShortcutAsText = myModifierString + " + " + myKeyString;
        }
    }

    /**
 * action that is fired when a pressed key is released
 * after all keys are released, the key combination will be shown in the text box 
 */
    public void keyReleased(KeyEvent ke) {
        // only change the field's content after the keys have been released
        entryField.setText(newShortcutAsText);
    }

    /**
 *  has to be implemented, but is not used
 */
    public void keyTyped(KeyEvent ke) {
    }
}
