package mpi.eudico.client.annotator.gui;

import java.awt.Dialog;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.swing.AbstractButton;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.commands.ShortcutsUtil;
import mpi.eudico.client.annotator.prefs.gui.EditPrefsDialog;

/**
 * A panel showing all actions that can have a shortcut in a table an 
 * that allows to select one and change the shortcut.
 * 
 * @author alekoe
 *
 */
public class ShortcutPanel extends JPanel implements ActionListener {
	
	private JTable table;
	private String[][] data;
	public Map<String,KeyStroke> allShortCuts;
	private JComboBox sortBox;
	private JButton cancelButton;
	private JButton editButton;
	private JButton reloadButton;
	private JButton saveButton;
	private ShortcutsUtil scu = ShortcutsUtil.getInstance();
	public Boolean replaceShortcut = false;
	
	/**
	 * Constructor
	 */
    public ShortcutPanel() {
        super();
        setLayout(new GridBagLayout());
        
        allShortCuts = new LinkedHashMap<String, KeyStroke>(scu.getCurrentShortcuts());
        String[] columnNames = {ElanLocale.getString("Shortcuts.Table.Description"),
        							ElanLocale.getString("Shortcuts.Table.Category"),
        							ElanLocale.getString("Shortcuts.Table.Key"),
        							"ActionID",
        							"Keycode",
        							"Modifiercode"};
        ArrayList<String> allActions = new ArrayList<String>();
        Iterator it = allShortCuts.entrySet().iterator();
        while (it.hasNext())
        {
        		Map.Entry pairs = (Map.Entry) it.next();
        		String anAction = (String) pairs.getKey();
        		allActions.add(anAction);
        	}
        
        String[] nameArray = (String[]) allActions.toArray(new String[0]);
        
        int numberOfShortCuts = allShortCuts.size();
    
        data = new String[allShortCuts.size()][];
        
        for (int i=0;i<numberOfShortCuts;i++)        
        {        		
        		String actionName = nameArray[i]; 
        		String description = scu.getDescriptionForAction(actionName);
        		String category = ElanLocale.getString(scu.getCategoryForAction(actionName));
        		KeyStroke ks = allShortCuts.get(actionName);
        		String keyStrokeName = scu.getDescriptionForKeyStroke(ks);
        		String keyCode = "";
        		String modCode = "";
        		if (ks != null)
        		{
        			keyCode = Integer.toString(ks.getKeyCode());
        			modCode = Integer.toString(ks.getModifiers());
        		}
        		
        		String[] row = {description,category,keyStrokeName,actionName,keyCode,modCode};
        		data[i] = row;
        }
        
        DefaultTableModel model = new DefaultTableModel(data,columnNames);
        table = new JTable(model){
        		// Override isCellEditable
        		public boolean isCellEditable(int rowIndex, int colIndex) 
        		{
        			return false;   //Disallow the editing of any cell
        		}
        };
        
        table.getTableHeader().setReorderingAllowed(false);
        
        Collections.sort(model.getDataVector(), new ColumnSorter(0));
        
        // hide columns ActionID, Keycode, Modifiercode
        table.getColumn("ActionID").setMinWidth(0);
        table.getColumn("ActionID").setMaxWidth(0);
        table.getColumn("Keycode").setMinWidth(0);
        table.getColumn("Keycode").setMaxWidth(0);
        table.getColumn("Modifiercode").setMinWidth(0);
        table.getColumn("Modifiercode").setMaxWidth(0);
        table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    
        //Create the scroll pane and add the table to it.
        JScrollPane scrollPane = new JScrollPane(table);
        
        //Add the scroll pane to this panel.
        GridBagConstraints gbc = new GridBagConstraints();
        Insets insets = new Insets(2, 6, 2, 6);
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        gbc.insets = insets;
        add(scrollPane, gbc);
        
        JPanel buttonPanel = new JPanel(new GridLayout(1, 6, 2, 2));

        JLabel sortLabel = new JLabel(ElanLocale.getString("Shortcuts.Table.SortBy"));
        sortLabel.setHorizontalAlignment(JLabel.RIGHT);
        sortLabel.setVerticalTextPosition(AbstractButton.BOTTOM);
        sortLabel.setHorizontalTextPosition(AbstractButton.LEADING);
        
        buttonPanel.add(sortLabel);
        
        String[] sortOptions = new String[] {ElanLocale.getString("Shortcuts.Table.Description"),ElanLocale.getString("Shortcuts.Table.Category"),ElanLocale.getString("Shortcuts.Table.Key")} ;
        sortBox = new JComboBox(sortOptions);
        sortBox.addActionListener(this);
        buttonPanel.add(sortBox);
        
        editButton = new JButton(ElanLocale.getString("Shortcuts.Button.Edit"));        
        editButton.addActionListener(this);        
        editButton.setVerticalTextPosition(AbstractButton.CENTER);
        editButton.setHorizontalTextPosition(AbstractButton.LEADING);
        buttonPanel.add(editButton);
        
        saveButton = new JButton(ElanLocale.getString("Button.Save"));        
        saveButton.addActionListener(this);        
        saveButton.setVerticalTextPosition(AbstractButton.CENTER);
        saveButton.setHorizontalTextPosition(AbstractButton.LEADING);
        buttonPanel.add(saveButton);
        
        reloadButton = new JButton(ElanLocale.getString("Shortcuts.Button.Default"));
        reloadButton.addActionListener(this);        
        reloadButton.setVerticalTextPosition(AbstractButton.CENTER);
        reloadButton.setHorizontalTextPosition(AbstractButton.LEADING);
        buttonPanel.add(reloadButton);

        cancelButton = new JButton(ElanLocale.getString("Button.Cancel"));
        cancelButton.addActionListener(this);        
        cancelButton.setVerticalTextPosition(AbstractButton.CENTER);
        cancelButton.setHorizontalTextPosition(AbstractButton.LEADING);
        buttonPanel.add(cancelButton);

        gbc.gridy = 1;
        gbc.fill = GridBagConstraints.NONE;
        gbc.weighty = 0.0;
        gbc.weightx = 0.0;
        add(buttonPanel, gbc);
    }
    
    
    /**
     * actionListener method that catches whether a button is pressed and either
     * closes the window without changing anything (Cancel button)
     * opens an edit window for changing a shortcut (Edit button)
     * saves the current shortcuts to a preferences file (Save button)
     * reloads the default preferences (Reload button) 
     * 
     * if the event comes from the sortBox, sorting is triggered
     */
    public void actionPerformed( ActionEvent e )
    {
    		if (e.getSource() == sortBox)
    		{
	  	  	JComboBox cb = (JComboBox)e.getSource();
	  	  	int index = cb.getSelectedIndex();
	  	  	DefaultTableModel dtmodel = (DefaultTableModel)table.getModel();
	  	  	Vector data = dtmodel.getDataVector();
	  	    Collections.sort(data, new ColumnSorter(index));
	  	    dtmodel.fireTableStructureChanged();
	        // rehide columns ActionID, Keycode, Modifiercode
	        table.getColumn("ActionID").setMinWidth(0);
	        table.getColumn("ActionID").setMaxWidth(0);
	        table.getColumn("Keycode").setMinWidth(0);
	        table.getColumn("Keycode").setMaxWidth(0);
	        table.getColumn("Modifiercode").setMinWidth(0);
	        table.getColumn("Modifiercode").setMaxWidth(0);
	  	    
    		}
    		if (e.getSource() == editButton)
    		{
	    		int row = table.getSelectedRow();
	    		if (row > -1)
	    		{
	      		int column = 3;
	      		String selectedAction = (String) table.getValueAt(row, column);
	    	  		createEditWindow(selectedAction);
	    		}
    		}
    		if (e.getSource() == saveButton)
    		{
			int rowNumber = table.getRowCount();
      		HashMap<String, List<String>> shortcutMap = new HashMap<String, List<String>>(rowNumber);
      		
      		for (int i=0;i<rowNumber;i++)
      		{
      			String actionName = (String) table.getValueAt(i, 3);
      			String keycode = (String) table.getValueAt(i, 4);
      			String modcode = (String) table.getValueAt(i, 5);
      			ArrayList<String> codes = new ArrayList<String>(2);
      			codes.add(keycode);
      			codes.add(modcode);
      			shortcutMap.put(actionName,codes);
      		}
      		scu.saveCurrentShortcuts(shortcutMap);
      		SwingUtilities.getWindowAncestor(this).setVisible(false);
		  	//JDialog jd = (JDialog) this.getParent().getParent().getParent();
		  	//jd.setVisible(false);
		  	scu.readCurrentShortcuts();
    		}
    		if (e.getSource() == reloadButton)
    		{
        	  	scu.restoreDefaultShortcuts();        	  	
        	  	refreshTable();
        	  	allShortCuts = scu.getCurrentShortcuts();
    		}
	    	if (e.getSource() == cancelButton)
	    	{
	    		SwingUtilities.getWindowAncestor(this).setVisible(false);
		  	//JDialog jd = (JDialog) this.getParent().getParent().getParent();
		  	//jd.setVisible(false);
	    	}
	  }
    
    /**
     * a method to ensure that the shortcuts stored in the member variable and 
     * those that are displayed in the table are the same
     */
    private void refreshTable()
    {
    		Map<String,KeyStroke> allShortCuts = scu.getCurrentShortcuts();
    		
    		int noOfRows = table.getRowCount();
    		for (int i=0;i<noOfRows;i++)
    		{
    			// Initialise everything with 0
    			int tableKeyCode = 0;
    			int tableModCode = 0;
    			int dataKeyCode = 0;
    			int dataModCode = 0;

    			// get shortcut currently stored in the table
    			String tableKeyCodeAsString = (String) table.getValueAt(i, 4);
    			String tableModCodeAsString = (String) table.getValueAt(i, 5);
    			// if there is any, get the corresponding key codes
    			if (tableKeyCodeAsString != "")
    			{
    				tableKeyCode = Integer.parseInt(tableKeyCodeAsString);
    			}
    			if (tableModCodeAsString != "")
    			{
    				tableModCode = Integer.parseInt(tableModCodeAsString);
    			}
    			// get the key stroke from ShortcutUtils.shortcutKeyStrokes
    			String action = (String) table.getValueAt(i, 3);
    			KeyStroke aKeyStroke = allShortCuts.get(action);
    			// if there is any, get the corresponding key codes
    			if (aKeyStroke != null)
    			{
    				dataKeyCode = aKeyStroke.getKeyCode();
        			dataModCode = aKeyStroke.getModifiers();
    			}
    			
    			// if current keystroke and the one displayed in the table mismatch
    			if ((tableKeyCode != dataKeyCode) || (tableModCode != dataModCode))
    			{
    				// update the table
    				table.setValueAt(Integer.toString(dataKeyCode), i, 4);
    				table.setValueAt(Integer.toString(dataModCode), i, 5);
    				String keyStrokeName = scu.getDescriptionForKeyStroke(aKeyStroke);
    				table.setValueAt(keyStrokeName, i, 2);
    			}    			
    		}
    }
    
    /**
     * creates a new window to edit the shortcut for the selected action 
     * @param selectedAction the action selected in the table for which the shortcut should be edited
     */
    private void createEditWindow(String selectedAction)
    {
    		//disableButtons();
    		ShortcutEditPanel.createAndShowGUI(this,selectedAction); // blocks
    		//enableButtons();
    }    
    
    /**
     * enables all buttons and the sortBox
     */
    /*
    public void enableButtons()
    {
    		sortBox.setEnabled(true);
    		cancelButton.setEnabled(true);
    		editButton.setEnabled(true);
    		reloadButton.setEnabled(true);
    		saveButton.setEnabled(true);
    }
     */
    /**
     * disables all buttons and the sortBox
     */
    /*
    public void disableButtons()
    {
    		sortBox.setEnabled(false);
    		cancelButton.setEnabled(false);
    		editButton.setEnabled(false);
    		reloadButton.setEnabled(false);
    		saveButton.setEnabled(false);
    }
    */

    /**
     * gets the name of the action currently associated with a certain keyStroke
     * @param ks the keyStroke for which to look up the action
     * @return the action currently associated with the keyStroke 
     * 			or NULL if no action uses this keyStroke
     */
    public final String getActionNameForKeyStroke(KeyStroke ks)
    {
    		for(String action : allShortCuts.keySet())
    		{
    			KeyStroke a_keystroke = allShortCuts.get(action);
    			if (a_keystroke != null)
    			{
    				if(a_keystroke.equals(ks)) 
    				{
    	                return action;    				
    				}            
            }
        }
        return null;	
    }
    
    /**
     * changes the shortcut for the currently selected action to the specified keyStroke
     * @param ks the new shortcut keyStroke
     */
    public void changeShortcut(KeyStroke ks)
    {
    		int row = table.getSelectedRow();
    		int desc_col = 2;
    		int action_col = 3;
    		int keycode_col = 4;
    		int mod_col = 5;
    		String actionDesc = scu.getDescriptionForAction(getActionNameForKeyStroke(ks));
    		String keydesc = scu.getDescriptionForKeyStroke(ks);
    		String keycodestring = "";
    		String modstring = "";
    		if (ks != null)
    		{
    			if(allShortCuts.containsValue(ks))
    			{    			
    				// show the ShortcutReplace dialog
    				ShortcutReplaceDialog srd = new ShortcutReplaceDialog(this,ks);
				srd.pack();
    			    srd.setLocationRelativeTo((JDialog) javax.swing.SwingUtilities.getWindowAncestor(this));
    			    srd.setModal(true);
    				srd.setVisible(true);
    				
    				// if the user chose replace, do this
    				if (replaceShortcut)
    				{
    					//get action currently associated with KeyStroke ks
    					String oldAction = getActionNameForKeyStroke(ks);
    					// get the corresponding table row
    					int oldRow = getRowForAction(oldAction);
    					if (oldRow == -1) {return;} // this shouldn't be possible but better safe than sorry
    					// update the table
    					table.setValueAt("", oldRow, desc_col);
	    		    		table.setValueAt("", oldRow, keycode_col);
	    		    		table.setValueAt("", oldRow, mod_col);
	    		    		// update the hash
	    		    		allShortCuts.put(oldAction, null);
    					//JOptionPane.showMessageDialog( null, "Shortcut has been replaced");
    				}
    				else
    				{
    					return;
    				}
    			}
    			keycodestring = Integer.toString(ks.getKeyCode());
    			modstring = Integer.toString(ks.getModifiers());
    		}

    		String action = (String) table.getValueAt(row, action_col);
    		table.setValueAt(keydesc, row, desc_col);
    		table.setValueAt(keycodestring, row, keycode_col);
    		table.setValueAt(modstring, row, mod_col);
    		allShortCuts.put(action, ks);
    }
    
 

    /**
     * searches the table to find the row in which a certain action is displayed
     * @param oldAction the action ID of the action 
     * @return the index of the action's row or -1 if it cannot be found 
     */
    private int getRowForAction(String oldAction) 
    {
		// get no of rows
    		int noOfRows = table.getRowCount();
    		// get no of the column containing the action IDs
    		int action_col = 3;
    		// iterate through table to find the correct row no
    		for (int i=0;i<noOfRows;i++)
    		{
    			String tableresult = (String)table.getValueAt(i, action_col);
    			if (tableresult.compareTo(oldAction) == 0)
    			{
    				// return the row no
    				return i;
    			}
    		}
    		// return a negative value if the action couldn't be found in the table
		return -1;
	}


	/**
     * Create the GUI and show it. 
     */
    public static void createAndShowGUI(Window owner) {
        //Create and set up the window.
    	JDialog frame = null;
    	if (owner instanceof Dialog) {
    		frame = new JDialog((Dialog)owner,ElanLocale.getString("Shortcuts.Table.Title"),true);
    	} else if (owner instanceof Frame) {
    		frame = new JDialog((Frame)owner,ElanLocale.getString("Shortcuts.Table.Title"),true);
    	}
    	
    	if (frame == null) {
    		return;
    	}
        //Create and set up the content pane.
        ShortcutPanel newContentPane = new ShortcutPanel();
        newContentPane.setOpaque(true); //content panes must be opaque
        frame.setContentPane(newContentPane);

        //Display the window.
        frame.pack();
        frame.setLocationRelativeTo(owner);
        frame.setVisible(true);
    }
    
}
