package mpi.eudico.client.annotator.md.imdi;

import mpi.eudico.client.annotator.Constants;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.md.DefaultMDViewerComponent;
import mpi.eudico.client.annotator.md.spi.MDServiceProvider;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import java.util.Map;
import java.util.ResourceBundle;

import javax.swing.ButtonGroup;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.table.DefaultTableModel;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;


/**
 * Metadata viewer for IMDI metadata. Currently extends the
 * DefaultMDViewerComponent  by setting a custom table renderer for the
 * metadata keys. It might be necessary to have more flexibility in the
 * future. This viewer could then extend JPanel and implement
 * MDViewerComponent. A tree view could be offered as an alternative in that
 * case.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class ImdiMDViewerPanel extends DefaultMDViewerComponent
    implements ActionListener, MouseListener {
    /** Holds value of property DOCUMENT ME! */
    protected JTree tree;
    private final int TABLE = 0;
    private final int TREE = 1;
    private int mode;
    private JPopupMenu popup;
    private JRadioButtonMenuItem tableMI;
    private JRadioButtonMenuItem treeMI;

    /**
     * Creates a new ImdiMDViewerPanel instance
     */
    private ImdiMDViewerPanel() {
        super();
    }

    /**
     * Creates a new ImdiMDViewerPanel instance
     *
     * @param provider the metadata provider
     */
    public ImdiMDViewerPanel(MDServiceProvider provider) {
        super(provider);
    }

    /**
     * Initializes the components. Adds a custom renderer to the table.
     */
    protected void initComponents() {
    	mode = TABLE;
        Object val = Preferences.get("Metadata.IMDI.ViewMode", null);

        if (val instanceof String) {
            if ("Tree".equals(val)) {
                mode = TREE;
            }
        }

        //super.initComponents();
        scrollPane = new JScrollPane();
        setLayout(new GridBagLayout());

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weighty = 1.0;
        gbc.weightx = 1.0;
        add(scrollPane, gbc);

        if (mode == TABLE) {
            model = new DefaultTableModel(0, 2);
            model.setColumnIdentifiers(new String[] { keyColumn, valColumn });
            mdTable = new JTable(model);
            mdTable.setEnabled(false);
            mdTable.getColumn(keyColumn).setCellRenderer(new ImdiKeyRenderer());
            mdTable.setBackground(Constants.DEFAULTBACKGROUNDCOLOR);
            scrollPane.setViewportView(mdTable);
            mdTable.addMouseListener(this);
        } else {
            tree = new JTree(new DefaultMutableTreeNode(ImdiConstants.SESSION));
            ((DefaultTreeCellRenderer) tree.getCellRenderer()).setOpenIcon(null);
            ((DefaultTreeCellRenderer) tree.getCellRenderer()).setClosedIcon(null);
            ((DefaultTreeCellRenderer) tree.getCellRenderer()).setLeafIcon(null);
            ((DefaultTreeCellRenderer) tree.getCellRenderer()).setTextNonSelectionColor(Color.BLACK);
            ((DefaultTreeCellRenderer) tree.getCellRenderer()).setTextSelectionColor(Color.BLACK);
            //tree.setEditable(false);
            //tree.setEnabled(false);
            //tree.putClientProperty("JTree.lineStyle", "Horizontal");
            scrollPane.setViewportView(tree);
            tree.addMouseListener(this);
        }
    }

    /**
     * @see mpi.eudico.client.annotator.md.DefaultMDViewerComponent#setSelectedKeysAndValues(java.util.Map)
     */
    @Override
    public void setSelectedKeysAndValues(Map keysAndValuesMap) {
        if (mode == TREE) {
            /*
               if (tree == null) {
                   tree = new JTree(((ImdiFileServiceProvider) provider).getSelectedAsTree());
                   ((DefaultTreeCellRenderer) tree.getCellRenderer()).setOpenIcon(null);
                   ((DefaultTreeCellRenderer) tree.getCellRenderer()).setClosedIcon(null);
                   ((DefaultTreeCellRenderer) tree.getCellRenderer()).setLeafIcon(null);
                   tree.setEditable(false);
                   tree.setEnabled(false);
                   //tree.putClientProperty("JTree.lineStyle", "Horizontal");
                   scrollPane.setViewportView(tree);
                   expandAllRows(tree);
                   if (mdTable != null) {
                       mdTable.removeMouseListener(this);
                   }
                   tree.addMouseListener(this);
               } else {*/
            ((DefaultTreeModel) tree.getModel()).setRoot(((ImdiFileServiceProvider) provider).getSelectedAsTree());
            expandAllRows(tree);

            //}
        } else {
            super.setSelectedKeysAndValues(keysAndValuesMap);
        }
    }

    private void expandAllRows(JTree tree) {
        for (int i = 0; i < tree.getRowCount(); i++) {
            tree.expandRow(i);
        }
    }

    private void createPopupMenu() {
        popup = new JPopupMenu("Popup");

        ButtonGroup group = new ButtonGroup();
        tableMI = new JRadioButtonMenuItem();
        tableMI.setSelected(mode == TABLE);
        group.add(tableMI);
        tableMI.addActionListener(this);
        popup.add(tableMI);
        treeMI = new JRadioButtonMenuItem();
        treeMI.setSelected(mode == TREE);
        group.add(treeMI);
        treeMI.addActionListener(this);
        popup.add(treeMI);

        if (bundle != null) {
            String text = bundle.getString("MetadataViewer.TableView");

            if (text != null) {
                tableMI.setText(text);
            } else {
                tableMI.setText("Table View");
            }

            text = bundle.getString("MetadataViewer.TreeView");

            if (text != null) {
                treeMI.setText(text);
            } else {
                treeMI.setText("Tree View");
            }
        } else {
            tableMI.setText("Table View");
            treeMI.setText("Tree View");
        }
    }

    /**
     * @see mpi.eudico.client.annotator.md.DefaultMDViewerComponent#setResourceBundle(java.util.ResourceBundle)
     */
    @Override
    public void setResourceBundle(ResourceBundle bundle) {
        super.setResourceBundle(bundle);

        // update popup menu
        if (popup != null) {
            String text = bundle.getString("MetadataViewer.TableView");

            if (text != null) {
                tableMI.setText(text);
            } else {
                tableMI.setText("Table View");
            }

            text = bundle.getString("MetadataViewer.TreeView");

            if (text != null) {
                treeMI.setText(text);
            } else {
                treeMI.setText("Tree View");
            }
        }
    }

    /**
     * Menu items action handling.
     *
     * @param e event
     */
    public void actionPerformed(ActionEvent e) {
        if (e.getSource() == tableMI) {
            mode = TABLE;

            if (tree != null) {
                tree.removeMouseListener(this);
            }

            model = new DefaultTableModel(0, 2);
            model.setColumnIdentifiers(new String[] { keyColumn, valColumn });
            mdTable = new JTable(model);
            mdTable.setEnabled(false);
            mdTable.getColumn(keyColumn).setCellRenderer(new ImdiKeyRenderer());
            mdTable.setBackground(Constants.DEFAULTBACKGROUNDCOLOR);
            scrollPane.setViewportView(mdTable);
            mdTable.addMouseListener(this);
            setSelectedKeysAndValues(provider.getSelectedKeysAndValues());
            tree = null;
            Preferences.set("Metadata.IMDI.ViewMode", "Table", null, false, false);
        } else if (e.getSource() == treeMI) {
            mode = TREE;

            if (mdTable != null) {
                mdTable.removeMouseListener(this);
            }

            tree = new JTree(new DefaultMutableTreeNode(ImdiConstants.SESSION));
            ((DefaultTreeCellRenderer) tree.getCellRenderer()).setOpenIcon(null);
            ((DefaultTreeCellRenderer) tree.getCellRenderer()).setClosedIcon(null);
            ((DefaultTreeCellRenderer) tree.getCellRenderer()).setLeafIcon(null);
            ((DefaultTreeCellRenderer) tree.getCellRenderer()).setTextNonSelectionColor(Color.BLACK);
            ((DefaultTreeCellRenderer) tree.getCellRenderer()).setTextSelectionColor(Color.BLACK);

            //tree.setEditable(false);
            //tree.setEnabled(false);
            //tree.putClientProperty("JTree.lineStyle", "Horizontal");
            scrollPane.setViewportView(tree);
            tree.addMouseListener(this);
            setSelectedKeysAndValues(null);
            mdTable = null;
            Preferences.set("Metadata.IMDI.ViewMode", "Tree", null, false, false);
        }
    }

    /**
     * Shows the popup menu.
     *
     * @param e event
     */
    public void mousePressed(MouseEvent e) {
        if ((SwingUtilities.isRightMouseButton(e) &&
                ((e.getButton() == MouseEvent.BUTTON1) ^ e.isMetaDown())) ||
                e.isPopupTrigger()) {
            if (popup == null) {
                createPopupMenu();
            }

            Point pp = e.getPoint();

            if ((popup.getWidth() == 0) || (popup.getHeight() == 0)) {
                popup.show(this, pp.x, pp.y);
            } else {
                popup.show(this, pp.x, pp.y);
                SwingUtilities.convertPointToScreen(pp, ImdiMDViewerPanel.this);

                Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
                Window w = SwingUtilities.windowForComponent(ImdiMDViewerPanel.this);

                if ((pp.x + popup.getWidth()) > d.width) {
                    pp.x -= popup.getWidth();
                }

                //this does not account for a desktop taskbar
                if ((pp.y + popup.getHeight()) > d.height) {
                    pp.y -= popup.getHeight();
                }

                //keep it in the window then
                if ((pp.y + popup.getHeight()) > (w.getLocationOnScreen().y +
                        w.getHeight())) {
                    pp.y -= popup.getHeight();
                }

                popup.setLocation(pp);
            }
        }
    }

    /**
     * Stub
     *
     * @param e event
     */
    public void mouseClicked(MouseEvent e) {
        // stub
    }

    /**
     * Stub
     *
     * @param e event
     */
    public void mouseEntered(MouseEvent e) {
        // stub
    }

    /**
     * Stub
     *
     * @param e event
     */
    public void mouseExited(MouseEvent e) {
        // stub
    }

    /**
     * Stub
     *
     * @param e event
     */
    public void mouseReleased(MouseEvent e) {
        // stub
    }
}
