/*
 * File:     PreferencesWriter.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.prefs;

import mpi.eudico.client.annotator.util.ClientLogger;

import mpi.eudico.util.IoUtil;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import java.io.IOException;

import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;


/**
 * Encodes and saves the preferences in an xml file.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class PreferencesWriter implements ClientLogger, PrefConstants {
    private DocumentBuilderFactory dbf;
    private DocumentBuilder db;
    private PrefObjectConverter poConverter;

    /** Holds value of the current version */
    public final String VERSION = "1.0";

    /**
     * Constructor. Instantiates the DocumentBuilder.
     */
    public PreferencesWriter() {
        super();
        poConverter = new PrefObjectConverter();

        try {
            dbf = DocumentBuilderFactory.newInstance();
            db = dbf.newDocumentBuilder();
        } catch (FactoryConfigurationError fce) {
            LOG.severe("Unable to create a Document Builder: " +
                fce.getMessage());
        } catch (ParserConfigurationException pce) {
            LOG.severe("Unable to create a Document Builder: " +
                pce.getMessage());
        }
    }

    /**
     * Creates the DOM for the objects in the prefs map and stores in the
     * specified location.
     *
     * @param prefs the preferences objects
     * @param path the file path
     */
    public synchronized void encodeAndSave(Map prefs, String path) {
        if ((prefs == null) || (path == null)) {
            return;
        }

        Element root = createDOM(prefs);

        try {
            IoUtil.writeEncodedFile("UTF-8", path, root);
        } catch (IOException ioe) {
            LOG.severe("Could not save the preferences xml file to: " + path +
                "\n" + " Cause: " + ioe.getMessage());
        } catch (Exception e) {
            LOG.severe("Could not save the preferences xml file to: " + path +
                "\n" + " Cause: " + e.getMessage());
        }
    }

    /**
     * Creates a Document object, iterates over the key-value pairs in the map
     * and returns the root element.
     *
     * @param prefs the preferences objects
     *
     * @return the document element
     */
    private Element createDOM(Map prefs) {
        if (db != null) {
            Document doc = db.newDocument();
            Element root = doc.createElement("preferences");
            root.setAttribute("xmlns:xsi",
                "http://www.w3.org/2001/XMLSchema-instance");
            root.setAttribute("xsi:noNamespaceSchemaLocation",
                "http://www.mpi.nl/tools/elan/Prefs_v1.0.xsd");
            root.setAttribute(VERS_ATTR, VERSION);
            doc.appendChild(root);

            // iterate over the objects in the prefs map and create and add elements 
            String key;
            Object keyObj;
            Object value;
            Element nextElem;
            Iterator keyIt = prefs.keySet().iterator();

            while (keyIt.hasNext()) {
                keyObj = keyIt.next();

                if (keyObj instanceof String) {
                    key = (String) keyObj;
                    value = prefs.get(key);

                    if ((key != null) && (value != null)) {
                        nextElem = createPrefElement(doc, key, value);

                        if (nextElem != null) {
                            root.appendChild(nextElem);
                        }
                    }
                }
            }

            return root;
        }

        return null;
    }

    /**
     * Creates  a <code>pref</code>, <code>prefGroup</code> or
     * <code>prefList</code>  Element, depending on the nature of the
     * specified value.
     *
     * @param doc the DOM document
     * @param key the preferences key attribute
     * @param value a Map, List or a single Object to store
     *
     * @return the element
     */
    private Element createPrefElement(Document doc, String key, Object value) {
        Element pref = null;

        if (value instanceof Map) {
            pref = doc.createElement(PREF_GROUP);
            pref.setAttribute(KEY_ATTR, key);

            // add the key-value pairs as pref elements
            Iterator keyIt = ((Map) value).keySet().iterator();
            String childKey;
            Object obj;
            Element childElem;

            while (keyIt.hasNext()) {
                childKey = (String) keyIt.next();
                obj = ((Map) value).get(childKey);

                if (obj != null) {
                    childElem = createPrefElement(doc, childKey, obj);

                    if (childElem != null) {
                        pref.appendChild(childElem);
                    }
                }
            }
        } else if (value instanceof List) {
            pref = doc.createElement(PREF_LIST);
            pref.setAttribute(KEY_ATTR, key);

            // add the values as the know object types elements
            List l = (List) value;
            Object obj;
            Element childElem;

            for (int i = 0; i < l.size(); i++) {
                obj = l.get(i);
                childElem = createContentELement(doc, obj);

                if (childElem != null) {
                    pref.appendChild(childElem);
                }
            }
        } else if (value instanceof Object[]) {
            pref = doc.createElement(PREF_LIST);
            pref.setAttribute(KEY_ATTR, key);

            // add the values as the know object types elements
            Object[] objArray = (Object[]) value;
            Object obj;
            Element childElem;

            for (int i = 0; i < objArray.length; i++) {
                obj = objArray[i];
                childElem = createContentELement(doc, obj);

                if (childElem != null) {
                    pref.appendChild(childElem);
                }
            }
        } else {
            pref = doc.createElement(PREF);
            pref.setAttribute(KEY_ATTR, key);

            Element childElem = createContentELement(doc, value);

            if (childElem != null) {
                pref.appendChild(childElem);
            }
        }

        return pref;
    }

    /**
     * Creates the content element which is one of the primitives wrappers, a
     * String or Object element.
     *
     * @param doc the DOM document
     * @param value a Map, List or a single Object to store
     *
     * @return the content element
     */
    private Element createContentELement(Document doc, Object value) {
        if (value == null) {
            return null;
        }

        Element objPref = null;

        if (value instanceof String) {
            objPref = doc.createElement(STRING);
            objPref.appendChild(doc.createTextNode((String) value));
        } else if (value instanceof Boolean) {
            objPref = doc.createElement(BOOLEAN);
            objPref.appendChild(doc.createTextNode(((Boolean) value).toString()));
        } else if (value instanceof Integer) {
            objPref = doc.createElement(INT);
            objPref.appendChild(doc.createTextNode(((Integer) value).toString()));
        } else if (value instanceof Long) {
            objPref = doc.createElement(LONG);
            objPref.appendChild(doc.createTextNode(((Long) value).toString()));
        } else if (value instanceof Float) {
            objPref = doc.createElement(FLOAT);
            objPref.appendChild(doc.createTextNode(((Float) value).toString()));
        } else if (value instanceof Double) {
            objPref = doc.createElement(DOUBLE);
            objPref.appendChild(doc.createTextNode(((Double) value).toString()));
        } else {
            objPref = doc.createElement(OBJECT);
            objPref.setAttribute(CLASS_ATTR, value.getClass().getName());
            objPref.appendChild(doc.createTextNode(poConverter.objectToString(
                        value)));
        }

        return objPref;
    }
}
