/*
 * File:     EditingPanel.java
 * Project:  MPI Linguistic Application
 * Date:     25 August 2009
 *
 * Copyright (C) 2001-2009  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.prefs.gui;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.prefs.PreferenceEditor;

import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;

import java.util.HashMap;
import java.util.Map;

import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;


/**
 * Panel showing options for:<br>
 * - flag to set that deselecting the edit box commits changes (default is cancel)<br>
 * - flag to set that Enter commits the changes (default is adding a new line
 * char)<br>
 * - flag to set that the selection should be cleared after creation of a new annotation,
 * or modifying the time alignement etc.
 *
 * @author Han Sloetjes
 */
public class EditingPanel extends JPanel implements PreferenceEditor {
    /** deselect commits preference, default is false */
    private JCheckBox deselectCB;

    /** enter commits (in addition to Ctrl + Enter), default false */
    private JCheckBox enterCommitsCB;

    /** clear the selection after creation or modification of a new annotation */
    private JCheckBox clearSelectionCB;
    private boolean origDeselectFlag = false;
    private boolean origEnterFlag = false;
    private boolean origClearSelFlag = false;

    /**
     * Creates a new EditingPanel instance
     */
    public EditingPanel() {
        super();
        readPrefs();
        initComponents();
    }

    private void readPrefs() {
        Object val = Preferences.get("InlineEdit.DeselectCommits", null);

        if (val instanceof Boolean) {
            origDeselectFlag = ((Boolean) val).booleanValue();
        }

        val = Preferences.get("InlineEdit.EnterCommits", null);

        if (val instanceof Boolean) {
            origEnterFlag = ((Boolean) val).booleanValue();
        }

        val = Preferences.get("ClearSelectionAfterCreation", null);

        if (val instanceof Boolean) {
            origClearSelFlag = ((Boolean) val).booleanValue();
        }
    }

    private void initComponents() {
        setLayout(new GridBagLayout());

        Insets insets = new Insets(2, 0, 2, 0);
        deselectCB = new JCheckBox(ElanLocale.getString(
                    "PreferencesDialog.Edit.Deselect"), origDeselectFlag);
        enterCommitsCB = new JCheckBox(ElanLocale.getString(
                    "PreferencesDialog.Edit.EnterCommits"), origEnterFlag);
        clearSelectionCB = new JCheckBox(ElanLocale.getString(
                    "PreferencesDialog.Edit.ClearSelection"), origClearSelFlag);
        deselectCB.setFont(deselectCB.getFont().deriveFont(Font.PLAIN));
        enterCommitsCB.setFont(deselectCB.getFont());
        clearSelectionCB.setFont(deselectCB.getFont());

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        gbc.insets = insets;
        add(new JLabel(ElanLocale.getString("PreferencesDialog.Category.Edit")),
            gbc);

        gbc.gridy = 1;
        add(deselectCB, gbc);

        gbc.gridy = 2;
        add(enterCommitsCB, gbc);

        gbc.gridy = 3;
        add(clearSelectionCB, gbc);

        gbc.gridy = 4;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weighty = 1.0;
        add(new JPanel(), gbc); // filler
    }

    /**
     * Returns the changed prefs.
     *
     * @return a map containing the changed preferences, key-value pairs, or null
     */
    public Map getChangedPreferences() {
        if (isChanged()) {
            Map chMap = new HashMap(3);

            if (deselectCB.isSelected() != origDeselectFlag) {
                chMap.put("InlineEdit.DeselectCommits",
                    new Boolean(deselectCB.isSelected()));
            }

            if (enterCommitsCB.isSelected() != origEnterFlag) {
                chMap.put("InlineEdit.EnterCommits",
                    new Boolean(enterCommitsCB.isSelected()));
            }

            if (clearSelectionCB.isSelected() != origClearSelFlag) {
                chMap.put("ClearSelectionAfterCreation",
                    new Boolean(clearSelectionCB.isSelected()));
            }

            return chMap;
        }

        return null;
    }

    /**
     * Returns whether anything has changed.
     *
     * @return whether anything has changed
     */
    public boolean isChanged() {
        if ((deselectCB.isSelected() != origDeselectFlag) ||
                (enterCommitsCB.isSelected() != origEnterFlag) ||
                (clearSelectionCB.isSelected() != origClearSelFlag)) {
            return true;
        }

        return false;
    }
}
